/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/muxers/rtmpstate.h"

#include "twitchsdk/broadcast/internal/streamstats.h"

ttv::broadcast::rtmpinputbuffer_t ttv::broadcast::RtmpState::mInputBuffer;
ttv::broadcast::rtmpinputbuffer_t::size_type ttv::broadcast::RtmpState::mInputBufferPos = 0;

TTV_ErrorCode ttv::broadcast::RtmpState::AppendChunkData(
  const uint8_t* buffer, size_t length, RtmpMessageDetails& chunkDetails) {
  size_t pos = 0;
  TTV_ErrorCode ret = TTV_EC_SUCCESS;
  size_t currentChunkSpace = GetContext()->mChunkSpace;

  assert(chunkDetails.IsNew() == false || currentChunkSpace == 0);

  while (length - pos > 0 && TTV_SUCCEEDED(ret)) {
    if (currentChunkSpace == 0) {
      RtmpMessageheader header;
      auto headerLength = chunkDetails.PackMessageHeader(header);
      ret = GetContext()->mSocket.Send(header, headerLength, true);
      currentChunkSpace = GetContext()->mOutgoingChunkSize;
    }

    if (TTV_SUCCEEDED(ret)) {
      auto amountToCopy = static_cast<uint32_t>(std::min(length - pos, currentChunkSpace));
      ret = GetContext()->mSocket.Send(&buffer[pos], amountToCopy, true);

      pos += amountToCopy;
      currentChunkSpace -= amountToCopy;
      chunkDetails.DataWritten(amountToCopy);
    }
  }
  GetContext()->mChunkSpace = currentChunkSpace;

  GetContext()->mStreamStats->Add(StreamStats::StatType::RtmpTotalBytesSent, GetContext()->mSocket.TotalSent());

  return ret;
}

TTV_ErrorCode ttv::broadcast::RtmpState::PollForInput() {
  TTV_ErrorCode blockingModeRet = GetContext()->mSocket.SetBlockingMode(false);
  ASSERT_ON_ERROR(blockingModeRet);

  TTV_ErrorCode ret = TTV_EC_SUCCESS;
  if (TTV_SUCCEEDED(blockingModeRet)) {
    ret = _PollForInput();
  }

  blockingModeRet = GetContext()->mSocket.SetBlockingMode(true);
  ASSERT_ON_ERROR(blockingModeRet);

  return ret;
}

TTV_ErrorCode ttv::broadcast::RtmpState::_PollForInput() {
  size_t received;

  TTV_ErrorCode ret =
    GetContext()->mSocket.Recv(&mInputBuffer[mInputBufferPos], mInputBuffer.size() - mInputBufferPos, received, false);
  if (TTV_EC_SOCKET_EWOULDBLOCK == ret) {
    return TTV_EC_SUCCESS;
  }

  if (TTV_SUCCEEDED(ret)) {
    mInputBufferPos += received;
    assert(mInputBufferPos < mInputBuffer.size());

    if (mInputBufferPos > 0) {
      rtmpinputbuffer_t::size_type readPos = 0;

      auto chunkSize = GetChunkSize(static_cast<char>(mInputBuffer[0]));
      while ((readPos + chunkSize) < mInputBufferPos) {
        ChunkHeader h = PopulateChunkHeader(&mInputBuffer[readPos]);
        assert(chunkSize + h.packetLength < mInputBuffer.size());

        readPos += chunkSize;

        // Handle chunk splitting
        size_t totalBytesWithHeaders = h.packetLength + h.packetLength / GetContext()->mIncomingChunkSize;
        if (h.packetLength > GetContext()->mIncomingChunkSize) {
          for (size_t i = 0; i < h.packetLength / GetContext()->mIncomingChunkSize; ++i) {
            size_t seperatorPos = GetContext()->mIncomingChunkSize * (i + 1);
            memmove(&mInputBuffer[readPos + seperatorPos], &mInputBuffer[readPos + seperatorPos + 1],
              totalBytesWithHeaders - seperatorPos - 1);
          }
        }

        if (readPos + h.packetLength <= mInputBufferPos) {
          switch (h.messageType) {
            case RTMP_PKT_CHUNK_SIZE:
              HandleIncomingChunkSize(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_ABORT_MSG:
              HandleIncomingAbortMsg(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_BYTES_READ:
              HandleIncomingBytesRead(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_CONTROL:
              HandleIncomingControl(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_WINACKSIZE:
              HandleIncomingWinacksize(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_PEER_BW:
              HandleIncomingPeerBW(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_EDGE_ORIGIN:
              HandleIncomingEdgeOrigin(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AUDIO:
              HandleIncomingAudio(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_VIDEO:
              HandleIncomingVideo(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AMF3_DATA:
              HandleIncomingAmf3Data(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AMF3_SO:
              HandleIncomingAmf3SharedObject(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AMF3:
              HandleIncomingAmf3(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AMF0_DATA:
              HandleIncomingAmf0Data(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AMF0_SO:
              HandleIncomingAmf0SharedObject(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AMF0:
              HandleIncomingAmf0(h, &mInputBuffer[readPos]);
              break;
            case RTMP_PKT_AGGREGATE:
              HandleIncomingAggregate(h, &mInputBuffer[readPos]);
              break;
            default:
              // should never get here
              TTV_ASSERT(false);
              break;
          }
        }

        readPos += totalBytesWithHeaders;
        chunkSize = GetChunkSize(static_cast<char>(mInputBuffer[readPos]));
      }

      auto dataLeft = mInputBufferPos - readPos;

      if (dataLeft > 0) {
        memmove(&mInputBuffer[0], &mInputBuffer[readPos], mInputBufferPos - readPos);
        mInputBufferPos = dataLeft;
      } else {
        mInputBufferPos = 0;
      }
    }
  }

  return ret;
}

void ttv::broadcast::RtmpState::HandleIncomingChunkSize(ChunkHeader /*header*/, const uint8_t* data) {
  auto p = reinterpret_cast<const uint32_t*>(data);
  GetContext()->mIncomingChunkSize = BigToLittle(p[0]);
}

void ttv::broadcast::RtmpState::HandleIncomingAbortMsg(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAbortMsg nop");
}

void ttv::broadcast::RtmpState::HandleIncomingBytesRead(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingBytesRead nop");
}

void ttv::broadcast::RtmpState::HandleIncomingControl(ChunkHeader header, const uint8_t* data) {
  static const size_t kPingPacketLength = 6;

  // This should never occur in the middle of a chunk
  assert(GetContext()->mChunkSpace == 0);
  assert(data);
  assert(header.packetLength > 0);

  CtlType type = static_cast<CtlType>(data[1]);  // lint !e415 I know that the data won't be out of bounds
  switch (type) {
    case RTMP_CTL_PING: {
      TTV_ASSERT(header.packetLength == kPingPacketLength);
      if (header.packetLength == kPingPacketLength) {
        uint8_t outData[kPingPacketLength];
        memcpy(outData, data, kPingPacketLength);
        outData[1] = static_cast<uint8_t>(RTMP_CTL_PONG);

        TTV_ErrorCode err = GetContext()->mSocket.Send(reinterpret_cast<const uint8_t*>(&header), sizeof(header), true);

        if (TTV_SUCCEEDED(err)) {
          (void)GetContext()->mSocket.Send(outData, kPingPacketLength, true);
          (void)GetContext()->mSocket.FlushCache();
        }
      }
      break;
    }
    case RTMP_CTL_STREAM_BEGIN:
      // The server sends this event to notify the client
      // that a stream has become functional and can be
      // used for communication. By default, this event
      // is sent on ID 0 after the application connect
      // command is successfully received from the
      // client. The event data is 4-byte and represents
      // the stream ID of the stream that became
      // functional
      //
      break;
    default:
      ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingControl with unhandled type receieved");
      break;
  }
}

void ttv::broadcast::RtmpState::HandleIncomingWinacksize(ChunkHeader /*header*/, const uint8_t* data) {
  auto p = reinterpret_cast<const uint32_t*>(data);
  GetContext()->mWindowAckSize = BigToLittle(p[0]);
}

void ttv::broadcast::RtmpState::HandleIncomingPeerBW(ChunkHeader /*header*/, const uint8_t* data) {
  auto p = reinterpret_cast<const uint32_t*>(data);
  GetContext()->mIncomingPeerBW = BigToLittle(p[0]);
}

void ttv::broadcast::RtmpState::HandleIncomingEdgeOrigin(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingEdgeOrigin nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAudio(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAudio nop");
}

void ttv::broadcast::RtmpState::HandleIncomingVideo(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingVideo nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAmf3Data(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAmf3Data nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAmf3SharedObject(ChunkHeader /*header*/, uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAmf3SharedObject nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAmf3(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAmf3 nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAmf0Data(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAmf0Data nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAmf0SharedObject(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAmf0SharedObject nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAmf0(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAmf0 nop");
}

void ttv::broadcast::RtmpState::HandleIncomingAggregate(ChunkHeader /*header*/, const uint8_t* /*data*/) {
  ttv::trace::Message("rtmp", MessageLevel::Info, "HandleIncomingAggregate nop");
}

ttv::broadcast::RtmpIdleState::~RtmpIdleState() {}

ttv::broadcast::RtmpErrorState::~RtmpErrorState() {}
