/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/muxers/rtmpstream.h"

#include "twitchsdk/broadcast/internal/muxers/rtmpconnectstate.h"
#include "twitchsdk/broadcast/internal/muxers/rtmpcreatestreamstate.h"
#include "twitchsdk/broadcast/internal/muxers/rtmphandshakestate.h"
#include "twitchsdk/broadcast/internal/muxers/rtmpinitializestate.h"
#include "twitchsdk/broadcast/internal/muxers/rtmppublishstate.h"
#include "twitchsdk/broadcast/internal/muxers/rtmpsendvideostate.h"
#include "twitchsdk/broadcast/internal/muxers/rtmpshutdownstate.h"
#include "twitchsdk/broadcast/internal/streamstats.h"
#include "twitchsdk/core/thread.h"

ttv::broadcast::RtmpStream::RtmpStream(std::shared_ptr<StreamStats> streamStats) : mContext(streamStats) {
  mStates[static_cast<uint32_t>(RtmpContext::State::Idle)] = std::make_unique<RtmpIdleState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::Initialize)] = std::make_unique<RtmpInitializeState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::Handshake)] = std::make_unique<RtmpHandshakeState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::Connect)] = std::make_unique<RtmpConnectState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::CreateStream)] = std::make_unique<RtmpCreateStreamState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::Publish)] = std::make_unique<RtmpPublishState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::SendVideo)] = std::make_unique<RtmpSendVideoState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::Shutdown)] = std::make_unique<RtmpShutdownState>(&mContext);
  mStates[static_cast<uint32_t>(RtmpContext::State::Error)] = std::make_unique<RtmpErrorState>(&mContext);
}

bool ttv::broadcast::RtmpStream::ChangeState() {
  assert(mContext.GetNextState() >= RtmpContext::State::Invalid && mContext.GetNextState() < RtmpContext::State::Count);
  TTV_ASSERT(mContext.GetCurrentState() >= RtmpContext::State::Invalid &&
             mContext.GetCurrentState() < RtmpContext::State::Count);

  if (mContext.GetCurrentState() != mContext.GetNextState()) {
    if (mContext.GetCurrentState() != RtmpContext::State::Invalid) {
      mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->OnExit();
    }
    mContext.SetCurrentStateToNext();
    if (mContext.GetCurrentState() != RtmpContext::State::Invalid) {
      mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->OnEnter();
    }

    mContext.mStreamStats->Add(StreamStats::StatType::RtmpState, static_cast<uint64_t>(mContext.GetCurrentState()));
    return true;
  }

  return false;
}

// TODO: This really should run async
void ttv::broadcast::RtmpStream::Start(const std::string& url) {
  mContext.mURL = url;
  mContext.SetNextState(RtmpContext::State::Initialize);

  while (mContext.GetCurrentState() != RtmpContext::State::SendVideo &&
         mContext.GetCurrentState() != RtmpContext::State::Error) {
    ChangeState();

    if (TTV_SUCCEEDED(mContext.mLastError) && mContext.GetCurrentState() != RtmpContext::State::Invalid) {
      mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->Update();
    }

    ttv::Sleep(1);
  }
}

void ttv::broadcast::RtmpStream::Update() {
  ChangeState();
  if (mContext.GetCurrentState() != RtmpContext::State::Invalid) {
    mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->Update();
  }
}

void ttv::broadcast::RtmpStream::Stop() {
  mContext.SetNextState(RtmpContext::State::Shutdown);

  while (
    mContext.GetCurrentState() != RtmpContext::State::Idle && mContext.GetCurrentState() != RtmpContext::State::Error) {
    ChangeState();
    if (mContext.GetCurrentState() != RtmpContext::State::Invalid) {
      mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->Update();
    }

    ttv::Sleep(1);
  }
}

TTV_ErrorCode ttv::broadcast::RtmpStream::BeginFLVChunk(flv::TagTypes type, uint32_t timestamp, size_t length) {
  if (mContext.GetCurrentState() == RtmpContext::State::Error) {
    return mContext.mLastError;
  }

  assert(mChunkDetails.IsValid() == false);
  assert(mContext.GetCurrentState() == RtmpContext::State::SendVideo);

  if (mContext.GetCurrentState() == RtmpContext::State::SendVideo) {
    auto packetType = RTMP_PKT_INVALID;
    auto chunkChannel = RTMP_CHANNEL_INVALID;

    switch (type) {
      case flv::Video:
        chunkChannel = RTMP_CHANNEL_SOURCE;
        packetType = RTMP_PKT_VIDEO;
        break;
      case flv::Audio:
        chunkChannel = RTMP_CHANNEL_SOURCE;
        packetType = RTMP_PKT_AUDIO;
        break;
      case flv::Meta:
        chunkChannel = RTMP_CHANNEL_SOURCE;
        packetType = RTMP_PKT_AMF0_DATA;
        break;
      default:
        assert(false && "Invalid type");
        break;
    }

    mContext.mAmfEncoder.Reset();
    if (type == flv::Meta) {
      mContext.mAmfEncoder.String("@setDataFrame");
    }

    mChunkDetails = RtmpMessageDetails(
      chunkChannel, timestamp, static_cast<uint32_t>(length + mContext.mAmfEncoder.GetBuffer().size()), packetType, 1);

    TTV_ErrorCode ret = TTV_EC_SUCCESS;

    if (type == flv::Meta) {
      // TODO: This is fugly, find a better solution
      ret = mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->AppendChunkData(
        &mContext.mAmfEncoder.GetBuffer()[0], mContext.mAmfEncoder.GetBuffer().size(), mChunkDetails);
    }
    return ret;
  }

  return TTV_EC_BROADCAST_RTMP_UNABLE_TO_SEND_DATA;
}

TTV_ErrorCode ttv::broadcast::RtmpStream::AddFLVData(const uint8_t* data, size_t length) {
  if (mContext.GetCurrentState() == RtmpContext::State::Error) {
    return mContext.mLastError;
  }

  assert(mChunkDetails.IsValid());
  assert(mContext.GetCurrentState() == RtmpContext::State::SendVideo);

  if (mContext.GetCurrentState() == RtmpContext::State::SendVideo) {
    // TODO: This is fugly, find a better solution
    return mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->AppendChunkData(data, length, mChunkDetails);
  }

  return TTV_EC_BROADCAST_RTMP_UNABLE_TO_SEND_DATA;
}

TTV_ErrorCode ttv::broadcast::RtmpStream::EndFLVChunk() {
  if (mContext.GetCurrentState() == RtmpContext::State::Error) {
    return mContext.mLastError;
  }

  assert(mChunkDetails.IsValid());
  assert(mContext.GetCurrentState() == RtmpContext::State::SendVideo);

  if (mContext.GetCurrentState() == RtmpContext::State::SendVideo) {
    mChunkDetails = RtmpMessageDetails();
    // TODO: This is fugly, find a better solution
    mStates[static_cast<uint32_t>(mContext.GetCurrentState())]->EndChunk();
    return TTV_EC_SUCCESS;
  }

  return TTV_EC_BROADCAST_RTMP_UNABLE_TO_SEND_DATA;
}

TTV_ErrorCode ttv::broadcast::RtmpStream::GetError() {
  if (mContext.GetCurrentState() == RtmpContext::State::Error) {
    return mContext.mLastError;
  } else {
    return TTV_EC_SUCCESS;
  }
}
