/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/passthroughaudiocapture.h"

#include "twitchsdk/broadcast/iaudiomixer.h"
#include "twitchsdk/broadcast/ipreencodedaudioframereceiver.h"
#include "twitchsdk/core/memory.h"
#include "twitchsdk/core/systemclock.h"

namespace {
const char* kLoggerName = "PassThroughAudioCapture";
}

ttv::broadcast::PassThroughAudioCapture::PassThroughAudioCapture() : mAudioFormat(AudioFormat::PCM), mNumChannels(2) {}

std::string ttv::broadcast::PassThroughAudioCapture::GetName() const {
  return kLoggerName;
}

uint32_t ttv::broadcast::PassThroughAudioCapture::GetNumChannels() const {
  return mNumChannels;
}

TTV_ErrorCode ttv::broadcast::PassThroughAudioCapture::EnqueueAudioPacket(
  std::vector<uint8_t>&& audioPacket, uint64_t timestamp) {
  if (!mStarted) {
    return TTV_EC_INVALID_STATE;
  } else if (audioPacket.empty()) {
    return TTV_EC_INVALID_ARG;
  }

  auto entry = std::make_shared<QueueEntry>();
  entry->packet = std::move(audioPacket);
  entry->timestamp = timestamp;

  mAudioPacketQueue.push(entry);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughAudioCapture::SetAudioFormat(AudioFormat format) {
  if (mStarted) {
    return TTV_EC_INVALID_STATE;
  }

  mAudioFormat = format;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughAudioCapture::SetNumChannels(uint32_t numChannels) {
  if (mStarted) {
    return TTV_EC_INVALID_STATE;
  }

  mNumChannels = numChannels;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughAudioCapture::Process(
  const std::shared_ptr<IAudioMixer>& mixer, uint64_t& lastSampleTime) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto receiver = mixer->GetReceiverImplementation(IPreEncodedAudioFrameReceiver::GetReceiverTypeId());
  TTV_ASSERT(receiver != nullptr);
  auto preEncodedReceiver = std::static_pointer_cast<IPreEncodedAudioFrameReceiver>(receiver);

  std::shared_ptr<QueueEntry> entry;
  if (mAudioPacketQueue.try_pop(entry)) {
    // Package the frame
    std::shared_ptr<AudioFrame> audioFrame;
    ec = preEncodedReceiver->PackageFrame(
      std::move(entry->packet), mAudioFormat, mNumChannels, entry->timestamp, audioFrame);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
    TTV_ASSERT(audioFrame != nullptr);

    // Submit for processing
    ec = mixer->SubmitFrame(mAudioLayer, audioFrame);

    lastSampleTime = entry->timestamp;
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::PassThroughAudioCapture::Stop() {
  TTV_ErrorCode ec = AudioCaptureBase::Stop();

  if (TTV_SUCCEEDED(ec)) {
    mAudioPacketQueue.clear();
  }

  return ec;
}
