/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/passthroughvideocapture.h"

#include "twitchsdk/broadcast/ipreencodedvideoframereceiver.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/ivideoframequeue.h"
#include "twitchsdk/broadcast/ivideoframereceiver.h"
#include "twitchsdk/core/memory.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

namespace {
const char* kLoggerName = "PassThroughVideoCapture";
}

ttv::broadcast::PassThroughVideoCapture::PassThroughVideoCapture() : mInitialized(false), mStarted(false) {}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::EnqueueVideoPacket(
  std::vector<uint8_t>&& videoPacket, bool keyframe, uint64_t timestamp) {
  if (!mStarted) {
    return TTV_EC_INVALID_STATE;
  } else if (videoPacket.empty()) {
    return TTV_EC_INVALID_ARG;
  }

  auto entry = std::make_shared<QueueEntry>();
  entry->packet = std::move(videoPacket);
  entry->timestamp = timestamp;
  entry->keyframe = keyframe;

  mVideoPacketQueue.push(entry);

  // Allow the capture thread to continue
  mCondition.notify_one();

  return TTV_EC_SUCCESS;
}

std::string ttv::broadcast::PassThroughVideoCapture::GetName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder) {
  if (mStarted) {
    return TTV_EC_INVALID_STATE;
  }

  mVideoEncoder = encoder;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::SetFrameQueue(const std::shared_ptr<IVideoFrameQueue>& queue) {
  if (mStarted) {
    return TTV_EC_INVALID_STATE;
  }

  mFrameQueue = queue;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::Initialize() {
  if (mInitialized) {
    return TTV_EC_INVALID_STATE;
  }

  mInitialized = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::Start(const VideoParams& videoParams) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "PassThroughVideoCapture::Start()");

  if (!mInitialized || mStarted || mVideoEncoder == nullptr || mFrameQueue == nullptr) {
    return TTV_EC_INVALID_STATE;
  }

  // We require a IPreEncodedVideoFrameReceiver to submit to
  if (!mVideoEncoder->SupportsReceiverProtocol(IPreEncodedVideoFrameReceiver::GetReceiverTypeId())) {
    return TTV_EC_BROADCAST_INVALID_SUBMISSION_METHOD;
  }

  // Extract the IRawVideoFrameReceiver
  auto receiver = mVideoEncoder->GetReceiverImplementation(IPreEncodedVideoFrameReceiver::GetReceiverTypeId());
  TTV_ASSERT(receiver != nullptr);
  mReceiver = std::static_pointer_cast<IPreEncodedVideoFrameReceiver>(receiver);

  mVideoParams = videoParams;

  auto generateFrameFunc = [this]() {
    // Lock the mutex
    std::unique_lock<std::mutex> lock(mMutex);

    while (mStarted) {
      // ttv::trace::Message(kLoggerName, MessageLevel::Debug, "Tick");

      std::shared_ptr<QueueEntry> entry;
      if (mVideoPacketQueue.try_pop(entry)) {
        // Package the frame
        std::shared_ptr<VideoFrame> videoFrame;
        TTV_ErrorCode ec =
          mReceiver->PackageFrame(std::move(entry->packet), entry->keyframe, entry->timestamp, videoFrame);
        TTV_ASSERT(TTV_SUCCEEDED(ec));
        TTV_ASSERT(videoFrame != nullptr);

        // Submit for processing
        ec = mFrameQueue->AddVideoFrame(videoFrame);

        // An error while processing will stop the capture
        if (TTV_FAILED(ec)) {
          break;
        }
      } else {
        // Wait for the client to submit a frame
        mCondition.wait_for(lock, std::chrono::milliseconds(10));
      }
    }

    mStarted = false;
  };

  TTV_ErrorCode ec = CreateThread(generateFrameFunc, "ttv::broadcast::PassThroughVideoCapture", mThread);

  if (TTV_SUCCEEDED(ec)) {
    mStarted = true;
    mThread->Run();
  } else {
    mThread.reset();
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::Stop() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "PassThroughVideoCapture::Stop()");

  mStarted = false;

  mVideoPacketQueue.clear();

  if (mThread != nullptr) {
    mCondition.notify_all();

    mThread->Join();
    mThread.reset();
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PassThroughVideoCapture::Shutdown() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "PassThroughVideoCapture::Shutdown()");

  TTV_ErrorCode ec = Stop();

  if (TTV_SUCCEEDED(ec)) {
    mInitialized = false;
  }

  return ec;
}
