/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/pcmaudiomixer.h"

namespace {
const uint kMixerDuration = 5 * ttv::broadcast::kAudioEncodeRate;  // Number of sample store in the mixer buffer.
}

ttv::broadcast::PcmAudioMixer::PcmAudioMixer(uint32_t numChannels, uint32_t numSamplesPerChannelPerPacket)
    : mNumChannels(numChannels),
      mNumSamplesPerChannelPerPacket(numSamplesPerChannelPerPacket),
      mMixerBuffer(((kMixerDuration * numChannels) / numSamplesPerChannelPerPacket) *
                   numSamplesPerChannelPerPacket)  // Round down to the nearest packet size
      ,
      mNextPacketTime(0) {
  std::fill(mMixerBuffer.begin(), mMixerBuffer.end(), 0);
  ttv::trace::Message("PcmAudioMixer", MessageLevel::Info, "PcmAudioMixer created");
}

ttv::broadcast::PcmAudioMixer::~PcmAudioMixer() {
  ttv::trace::Message("PcmAudioMixer", MessageLevel::Info, "PcmAudioMixer destroyed");
}

TTV_ErrorCode ttv::broadcast::PcmAudioMixer::SubmitAudioSamples(
  const int16_t* data, uint64_t startTime, size_t length, float volume) {
  TTV_ASSERT(data != nullptr);

  // TODO: Make sure this doesn't go past end
  uint32_t bufferRelativePosition = static_cast<uint32_t>(startTime * mNumChannels) % mMixerBuffer.size();

  // TODO: this is a read/modify/write and is extremely slow. We really want to be able to
  // mix in multiple buffers at once and do it SIMD. That would take a big memory commitment
  // as we would have to store data for each device by itself.

  const float clampedVolume = fminf(fmaxf(0.f, volume), 1.f);
  // Use x^4 as a curve for volume scaling.
  const float volumeScale = powf(clampedVolume, 4.f);

  uint32_t dataPos = 0;
  while (dataPos < length * mNumChannels) {
    mMixerBuffer[bufferRelativePosition++] += static_cast<int32_t>(static_cast<float>(data[dataPos++]) * volumeScale);
    bufferRelativePosition %= mMixerBuffer.size();
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PcmAudioMixer::GetNextPacketTime(uint64_t& nextPacketTime) const {
  nextPacketTime = mNextPacketTime;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::PcmAudioMixer::GetPacket(int16_t* buffer, uint64_t& nextPacketTime) {
  TTV_ASSERT(buffer != nullptr);

  nextPacketTime = mNextPacketTime;

  const uint32_t read = static_cast<uint32_t>((mNextPacketTime * mNumChannels) % mMixerBuffer.size());

  // Because we round to packet size, packets should NEVER straddle the end of the buffer
  TTV_ASSERT(read + mNumSamplesPerChannelPerPacket < mMixerBuffer.size());

  // Convert our 32bit samples to 16bit, making sure to clamp them
  const int16_t upper_bound = static_cast<int16_t>(0x7FFF);
  const int16_t lower_bound = ~upper_bound;

  for (uint32_t pos = 0; pos < mNumSamplesPerChannelPerPacket * mNumChannels; ++pos) {
    int16_t value = static_cast<int16_t>(mMixerBuffer[read + pos] >> 1);

    buffer[pos] = std::max(lower_bound, std::min(upper_bound, value));
    mMixerBuffer[read + pos] = 0;
  }

  mNextPacketTime = mNextPacketTime + mNumSamplesPerChannelPerPacket;

  return TTV_EC_SUCCESS;
}
