/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/streamer.h"

#include "twitchsdk/broadcast/audioconstants.h"
#include "twitchsdk/broadcast/broadcastlistener.h"
#include "twitchsdk/broadcast/generated/getstreamkey.h"
#include "twitchsdk/broadcast/iaudiocapture.h"
#include "twitchsdk/broadcast/iaudioencoder.h"
#include "twitchsdk/broadcast/internal/audiostreamer.h"
#include "twitchsdk/broadcast/internal/framewriter.h"
#include "twitchsdk/broadcast/internal/muxers/flvmuxer.h"
#include "twitchsdk/broadcast/internal/muxers/flvmuxerasync.h"
#include "twitchsdk/broadcast/internal/os.h"
#include "twitchsdk/broadcast/internal/streamstats.h"
#include "twitchsdk/broadcast/internal/videostreamer.h"
#include "twitchsdk/broadcast/ivideocapture.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/ivideoframequeue.h"
#include "twitchsdk/broadcast/videoframe.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/internal/graphql/utilities/graphqlutilities.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/getstreamtask.h"
#include "twitchsdk/core/task/graphqltask.h"
#include "twitchsdk/core/task/lambdatask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/trackingcontext.h"
#include "twitchsdk/core/user/user.h"

#include <sstream>

// TODO: Move metadata to a separate component

// NOTE: This forward declaration is defined in the generated library code using the project generation Python scripts
// in core.
const char* TTV_GetVersionString();

namespace {
using namespace ttv::broadcast;

const char* kLoggerName = "Streamer";

const char* kAbsVideoEncoderSuffix = "-ABS";

const uint32_t kQueueDelayForWarningMilliseconds = 5000;
const uint32_t kQueueDelayForForcedStopMilliseconds = 15000;
const uint32_t kBandwidthWarningIntervalMilliseconds = 1000;

const char* kRequiredScopes[] = {
  "user_read", "channel_read", "channel_editor", "channel_commercial", "sdk_broadcast", "metadata_events_edit"};

inline std::string VersionString() {
#if _DEBUG
  return std::string("sdk_DEV");
#else
  return std::string("sdk_") + TTV_GetVersionString();
#endif
}

class VideoFrameQueueWrapper : public IVideoFrameQueue {
 public:
  using AddFunc = std::function<TTV_ErrorCode(const std::shared_ptr<VideoFrame>& frame)>;

  VideoFrameQueueWrapper(AddFunc&& addFunc) : mAddFunc(addFunc) {}

  virtual TTV_ErrorCode AddVideoFrame(const std::shared_ptr<VideoFrame>& frame) override { return mAddFunc(frame); }

 private:
  AddFunc mAddFunc;
};

std::string TrackingStringForConnectionType(ConnectionType connectionType) {
  switch (connectionType) {
    case ConnectionType::Wifi:
      return "wifi";
    case ConnectionType::Ethernet:
      return "ethernet";
    case ConnectionType::Cellular:
      return "cellular";
    default:
      return "unknown";
  }
}
}  // namespace

namespace ttv {
TTV_ErrorCode GetClientId(std::string& clientId);
}

ttv::broadcast::Streamer::Streamer(const std::shared_ptr<User>& user, const std::shared_ptr<StreamerContext>& context)
    : UserComponent(user),
      mContext(context),
      mChannelId(0),
      mBroadcasterSoftware("sdk"),
      mInitialTime(0),
      mStreamerState(StreamerState::Stopped),
      mStreamInfoFetchTimer(
        {// Custom backoff table:
         // Try once every 10 seconds for 30 seconds, then just fetch every minute if we're still failing.
          0, 10000, 10000, 10000, 60000},
        500),
      mLastReportedStreamerState(StreamerState::Stopped),
      mStateChangeError(TTV_EC_SUCCESS),
      mBandwidthWarningState(TTV_EC_SUCCESS),
      mFirstFrameSubmitted(false),
      mBandwidthTestMode(false),
      mForceArchiveBroadcast(false),
      mForceDontArchiveBroadcast(false) {
  ttv::trace::Message("Streamer", MessageLevel::Info, "Streamer created");
}

ttv::broadcast::Streamer::~Streamer() {
  ttv::trace::Message("Streamer", MessageLevel::Info, "Streamer destroyed");
}

void ttv::broadcast::Streamer::AddListener(const std::shared_ptr<IListener>& listener) {
  mListeners.AddListener(listener);
}

void ttv::broadcast::Streamer::RemoveListener(const std::shared_ptr<IListener>& listener) {
  mListeners.RemoveListener(listener);
}

void ttv::broadcast::Streamer::AddBandwidthStatListener(const std::shared_ptr<IBandwidthStatListener>& listener) {
  mBandwidthStatListeners.AddListener(listener);
}

void ttv::broadcast::Streamer::RemoveBandwidthStatListener(const std::shared_ptr<IBandwidthStatListener>& listener) {
  mBandwidthStatListeners.RemoveListener(listener);
}

TTV_ErrorCode ttv::broadcast::Streamer::SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder) {
  TTV_ASSERT(mStreamerState == StreamerState::Stopped);

  mVideoEncoder = encoder;

  mVideoStreamer->SetEncoder(encoder);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::Streamer::SetAudioEncoder(const std::shared_ptr<IAudioEncoder>& encoder) {
  TTV_ASSERT(mStreamerState == StreamerState::Stopped);

  mAudioEncoder = encoder;

  mAudioStreamer->SetEncoder(encoder);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::Streamer::SetCustomMuxer(const std::shared_ptr<IMuxer>& muxer) {
  TTV_ASSERT(mStreamerState == StreamerState::Stopped);

  mCustomMuxer = muxer;

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::Streamer::SetTaskRunner(std::shared_ptr<TaskRunner> /*taskRunner*/) {
  // We have our own to ensure tasks run immediately
}

TTV_ErrorCode ttv::broadcast::Streamer::Initialize() {
  ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::Initialize()");

  TTV_ErrorCode ec = Component::Initialize();

  if (TTV_SUCCEEDED(ec)) {
    if (IsValidOSVersion() == false) {
      ec = TTV_EC_OS_TOO_OLD;
    }
  }

  std::shared_ptr<VideoStreamer> videoStreamer;

  if (TTV_SUCCEEDED(ec)) {
    mVideoStreamer = std::make_shared<VideoStreamer>();
    mVideoStreamer->SetCapturer(mVideoCapturer);
  }

  if (TTV_SUCCEEDED(ec)) {
    mAudioStreamer = std::make_shared<AudioStreamer>();
  }

  mStreamStats = std::make_shared<StreamStats>();
  mTaskRunner = std::make_shared<TaskRunner>("Streamer");

  TTV_ASSERT(mContext->sharedTrackingContext != nullptr);
  mTrackingContext = std::make_shared<TrackingContext>(mContext->sharedTrackingContext);

  if (TTV_FAILED(ec)) {
    CompleteShutdown();
  }

  return ec;
}

void ttv::broadcast::Streamer::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  if (mState == State::Initialized) {
    if (!mOAuthIssue) {
      // Issue GetStreamInfo request if scheduled
      if ((mChannelId != 0) && (mStreamInfo == nullptr) && mStreamInfoFetchTimer.CheckNextRetry()) {
        TTV_ErrorCode ec = GetStreamInfo();

        // Error occurred so schedule another request
        if (TTV_FAILED(ec)) {
          mStreamInfoFetchTimer.ScheduleNextRetry();
        }
      }
    }

    if (mStreamStats != nullptr) {
      mStreamStats->Flush();
    }

    BandwidthStat stat;
    while (mStatQueue.try_pop(stat)) {
      mBandwidthStatListeners.Invoke(
        [&stat](const std::shared_ptr<IBandwidthStatListener>& listener) { listener->ReceivedBandwidthStat(stat); });
    }

    if (mMinuteBroadcastTrackingTimer.Check(true)) {
      TrackMinuteBroadcast();
      mMinuteBroadcastTrackingTimer.Set(60 * 1000);
    }
  }

  UserComponent::Update();
}

TTV_ErrorCode ttv::broadcast::Streamer::Shutdown() {
  ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::Shutdown()");

  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    InternalStop(TTV_EC_SHUTTING_DOWN, false, nullptr);
  }

  return ec;
}

bool ttv::broadcast::Streamer::CheckShutdown() {
  if (!UserComponent::CheckShutdown()) {
    return false;
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  return mStreamerState == StreamerState::Stopped;
}

void ttv::broadcast::Streamer::CompleteShutdown() {
  ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::CompleteShutdown()");

  UserComponent::CompleteShutdown();

  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  TTV_ASSERT(mStreamerState == StreamerState::Stopped);
  TTV_ASSERT(nullptr == mFrameWriter);
  TTV_ASSERT(nullptr == mFlvMuxer);

  TTV_ASSERT(mVideoStreamer == nullptr || mVideoStreamer.unique());
  mVideoStreamer.reset();

  // Deallocate the streamers here so we can control destruction order
  TTV_ASSERT(mAudioStreamer == nullptr || mAudioStreamer.unique());
  mAudioStreamer.reset();

  mCustomMuxer.reset();

  Component::CompleteShutdown();
}

std::string ttv::broadcast::Streamer::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::broadcast::Streamer::GetStreamInfo() {
  ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::GetStreamInfo()");

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  GetStreamTask::Callback callback = [this, cleanup](GetStreamTask* source, TTV_ErrorCode ec,
                                       const std::shared_ptr<GetStreamTask::Result>& result) {
    CompleteTask(source);

    if (TTV_SUCCEEDED(ec) && result != nullptr && result->streamInfo != nullptr) {
      mStreamInfo = std::move(result->streamInfo);

      // Notify clients of broadcast_id
      mListeners.Invoke([this, ec](const std::shared_ptr<IListener>& listener) {
        listener->OnStreamInfoFetched(this, ec, *mStreamInfo);
      });
    } else {
      mStreamInfoFetchTimer.ScheduleNextRetry();
    }

    cleanup(ec);
  };

  auto task = std::make_shared<GetStreamTask>(mChannelId, oauthToken->GetToken(), callback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to start task, can't request stream info");

    cleanup(ec);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::Streamer::ValidateParams(
  const VideoParams& videoParams, const IngestServer& /*ingestServer*/) const {
  TTV_RETURN_ON_NULL(mVideoStreamer, TTV_EC_NOT_INITIALIZED);

  // TODO: we should do validation of audio params and ingestServer here

  return mVideoStreamer->ValidateVideoParams(videoParams);
}

ttv::broadcast::Streamer::StreamerState ttv::broadcast::Streamer::GetStreamerState() const {
  std::unique_lock<std::mutex> lock(mStateMutex);
  return mStreamerState;
}

// This is called on the main client thread and kicks off an async start of broadcasting
TTV_ErrorCode ttv::broadcast::Streamer::Start(const StartParams& params, StartCallback&& callback) {
  AutoTracer tracer("Streamer", MessageLevel::Debug, "Streamer::Start()");

  // Set up a new local broadcast ID for tracking.
  mTrackingContext->SetProperty("debug_broadcast_id", GetGuid());

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  if (params.ingestServer.serverUrl == "" && params.outputFile == L"" && mCustomMuxer == nullptr) {
    ttv::trace::Message(
      "Streamer", MessageLevel::Debug, "Streamer::Start() - No RTMP, FLV file path or custom muxer specified");
    ec = TTV_EC_BROADCAST_INVALID_INGEST_SERVER;
  } else if (mVideoEncoder == nullptr) {
    ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::Start() - No video encoder set");
    ec = TTV_EC_BROADCAST_INVALID_ENCODER;
  } else if (mVideoCapturer == nullptr) {
    ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::Start() - No video capturer set");
    ec = TTV_EC_BROADCAST_INVALID_VIDEO_CAPTURER;
  } else if (mAudioEncoder == nullptr && mAudioStreamer->HasEnabledCapturers()) {
    ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::Start() - No audio encoder set");
    ec = TTV_EC_BROADCAST_INVALID_ENCODER;
  }

  if (TTV_SUCCEEDED(ec)) {
    // Warn that we can't respect the ABS flag
    if (params.videoParams.automaticBitRateAdjustmentEnabled && !mVideoEncoder->SupportsBitRateAdjustment()) {
      ttv::trace::Message("Streamer", MessageLevel::Warning,
        "Streamer::Start() - Automatic bit rate adjustment enabled but video encoder does not support it, using constant bitrate: %u kbps",
        params.videoParams.initialKbps);
    }

    bool busy = false;

    if (mContext->isBusy.compare_exchange_strong(busy, true)) {
      std::unique_lock<std::mutex> lock(mStateMutex);
      if (mStreamerState == StreamerState::Stopped) {
        mStreamerState = StreamerState::Starting;
        mStateChangeError = TTV_EC_SUCCESS;
        NotifyStreamerStateChanged();
      } else {
        mContext->isBusy = false;

        ec = TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
      }
    } else {
      ec = TTV_EC_BROADCAST_STREAM_ALREADY_STARTED;
    }
  }

  if (TTV_SUCCEEDED(ec)) {
    // This is marked mutable because we need to std::move the callback into KickOffStart, and captures
    // in lambdas are const by default. Because we own the code, we know that channelInfoCallback is
    // only ever invoked once, so we are safe to std::move (and therefore mutate) the nested callback.
    auto channelInfoCallback = [this, params, callback = std::move(callback)](TTV_ErrorCode callbackEc) mutable {
      // Ready to start internal encoding and streaming objects
      if (TTV_SUCCEEDED(callbackEc)) {
        KickOffStart(params, std::move(callback));
      } else {
        TrackStartFailure(callbackEc, false);

        InternalStop(callbackEc, false, nullptr);

        if (callback != nullptr) {
          callback(callbackEc);
        }
      }
    };

    // Fetch channel info if not set to write to file only
    if (params.ingestServer.serverUrl != "") {
      // First fetch latest channel info and stream key
      ec = GetStreamKey(std::move(channelInfoCallback));
    } else {
      channelInfoCallback(TTV_EC_SUCCESS);
    }
  }

  if (TTV_FAILED(ec)) {
    TrackStartFailure(ec, true);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::Streamer::KickOffStart(const StartParams& params, StartCallback&& callback) {
  AutoTracer tracer("Streamer", MessageLevel::Debug, "Streamer::KickOffStart()");

  // Called on the main thread
  LambdaTask::CompleteCallback taskCallback = [this, callback = std::move(callback),
                                                initialKbps = params.videoParams.initialKbps](
                                                LambdaTask* source, TTV_ErrorCode ec) {
    CompleteTask(source);

    if (TTV_SUCCEEDED(ec)) {
      if (!mBandwidthTestMode) {
        mTrackingContext->TrackEvent("mobile_broadcast_start",
          {{"category", mContext->category}, {"stream_name", mContext->title}, {"video_kbps", initialKbps},
            {"stream_connection", TrackingStringForConnectionType(mContext->connectionType)}});

        mMinuteBroadcastTrackingTimer.Set(60 * 1000);
      }
    } else {
      TrackStartFailure(ec, false);
    }

    if (callback != nullptr) {
      callback(ec);
    }

    mListeners.Invoke([this, ec](std::shared_ptr<IListener> listener) { listener->OnStartComplete(this, ec); });
  };

  // Called on a worker thread
  LambdaTask::JobCallback job = [this, params]() -> TTV_ErrorCode {
    mBandwidthTestMode = HasFlag(params.flags, StreamStartFlags::BandwidthTest);
    return InternalStart(params);
  };

  auto task = std::make_shared<LambdaTask>("StartStream", job, taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to start task, can't start stream async");

    TrackStartFailure(ec, false);
  }

  return ec;
}

// This is called on a worker thread
TTV_ErrorCode ttv::broadcast::Streamer::InternalStart(const StartParams& startParams) {
  AutoTracer tracer("Streamer", MessageLevel::Debug, "Streamer::InternalStart()");

  MuxerParameters params;
  AudioParams audioParams;
  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  bool audioEnabled = false;

  TTV_ASSERT(!mFirstFrameSubmitted);
  TTV_ASSERT(mInitialTime == 0);

  // Make sure we have a stream key if we want to use RTMP
  bool shouldUseRtmp = !startParams.ingestServer.serverUrl.empty();
  bool shouldWriteFlvFile = !startParams.outputFile.empty();
  if (shouldUseRtmp && mStreamKey.empty()) {
    ec = TTV_EC_NEED_TO_LOGIN;
  }

  if (TTV_SUCCEEDED(ec)) {
    // Initialize the FlvMuxer if necessary
    if (shouldUseRtmp || shouldWriteFlvFile) {
      TTV_ASSERT(mFlvMuxer == nullptr);
      if (startParams.enableAsyncFlvMuxer) {
        mFlvMuxer = std::make_shared<FlvMuxerAsync>(mStreamStats);
      } else {
        mFlvMuxer = std::make_shared<FlvMuxer>(mStreamStats);
      }
    }

    if (!mStreamKey.empty()) {
      ec = SetIngestServer(startParams.ingestServer);

      if (TTV_SUCCEEDED(ec)) {
        ec = SetStreamName(mStreamKey, startParams.videoParams.automaticBitRateAdjustmentEnabled);
        ASSERT_ON_ERROR(ec);
      }
    }
  }

  // Setup the FlvMuxer IVideoEncoder
  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(mStreamStats != nullptr);

    // Determine if audio is enabled
    std::vector<std::shared_ptr<IAudioCapture>> audioCapturers;
    mAudioStreamer->GetEnabledCapturers(audioCapturers);

    audioParams.numInputAudioLayers = static_cast<uint32_t>(audioCapturers.size());
    audioEnabled = !audioCapturers.empty();

    if (audioEnabled) {
      // TODO: We assume all capturers use the same number of channels
      audioParams.numChannels = audioCapturers[0]->GetNumChannels();
    }

    // Initialize the frame writer
    TTV_ASSERT(nullptr == mFrameWriter);
    mFrameWriter = std::make_shared<FrameWriter>(audioEnabled);

    mFrameWriter->SetStreamAbortCallback([this](FrameWriter* /*source*/, TTV_ErrorCode callbackEc) {
      TrackStreamFailure(callbackEc);
      InternalStop(callbackEc, false, nullptr);
    });

    mFrameWriter->SetDelayStateChangedCallback([this](FrameWriter* /*source*/, FrameWriter::DelayState state) {
      if (mFrameWriterDelayStateChangedCallback != nullptr) {
        mFrameWriterDelayStateChangedCallback(state);
      }
    });

    mFrameWriter->SetBandwidthStatCallback(
      [this](FrameWriter* /*source*/, const BandwidthStat& stat) { mStatQueue.push(stat); });

    mFrameWriter->SetWarningDelayThresholdMilliseconds(kQueueDelayForWarningMilliseconds);
    mFrameWriter->SetErrorDelayThresholdMilliseconds(kQueueDelayForForcedStopMilliseconds);

    ec = mVideoEncoder->SetFrameWriter(mFrameWriter);
    ASSERT_ON_ERROR(ec);
  }

  // Setup the IAudioEncoder
  if (TTV_SUCCEEDED(ec)) {
    if (mAudioEncoder != nullptr) {
      ec = mAudioEncoder->SetFrameWriter(mFrameWriter);
      ASSERT_ON_ERROR(ec);
    }
  }

  // Setup the AudioStreamer
  if (TTV_SUCCEEDED(ec)) {
    params.audioEnabled = audioEnabled;
    params.audioSampleRate = kAudioEncodeRate;
    params.audioSampleSize = kInputAudioBitsPerSample;
    params.audioStereo = audioParams.numChannels == 2;

    if (audioEnabled) {
      mAudioEncoder->GetAudioEncodingFormat(params.audioFormat);

      ec = mAudioStreamer->Initialize(audioParams);
      if (TTV_FAILED(ec)) {
        ttv::trace::Message(
          "Streamer", MessageLevel::Error, "Inside Streamer::Start - Call to AudioStreamer::Initialize() failed");
      }
    } else {
      params.audioFormat = AudioFormat::None;
    }
  }

  // Setup the IVideoCapture and VideoStreamer
  if (TTV_SUCCEEDED(ec)) {
    auto frameQueueWrapper = std::make_shared<VideoFrameQueueWrapper>(
      [this](const std::shared_ptr<VideoFrame>& frame) -> TTV_ErrorCode { return AddVideoFrame(frame); });
    mVideoCapturer->SetFrameQueue(frameQueueWrapper);

    ec = mVideoStreamer->Initialize(startParams.videoParams);
    if (TTV_FAILED(ec)) {
      ttv::trace::Message(
        "Streamer", MessageLevel::Error, "Inside Streamer::Start - Call to VideoStreamer::Initialize() failed");
    }
  }

  // Grab the SPS and PPS
  if (TTV_SUCCEEDED(ec)) {
    ec = mVideoEncoder->GetSpsPps(params.videoSps, params.videoPps);
    if (TTV_FAILED(ec)) {
      ttv::trace::Message(
        "Streamer", MessageLevel::Error, "Inside Streamer::Start - Call to mVideoStreamer->GetSpsPps failed");
    }
  }

  // Start muxers
  if (TTV_SUCCEEDED(ec)) {
    params.videoWidth = startParams.videoParams.outputWidth;
    params.videoHeight = startParams.videoParams.outputHeight;
    params.frameRate = startParams.videoParams.targetFramesPerSecond;
    params.appVersion = VersionString();

    // Start the FLV muxer which will implicitly connect to the backend via RTMP
    if (mFlvMuxer != nullptr) {
      mFlvMuxer->SetFlvPath(startParams.outputFile);
      mFlvMuxer->SetRtmpUrl(mOutputStreamName);

      // NOTE: This will not return until the RTMP handshake is complete
      ec = mFlvMuxer->Start(params);
      if (TTV_FAILED(ec)) {
        ttv::trace::Message(
          "Streamer", MessageLevel::Error, "Inside Streamer::Start - Call to mFlvMuxer->Start(params) failed");
      }
    }

    // Start the custom muxer
    if (TTV_SUCCEEDED(ec)) {
      if (mCustomMuxer != nullptr) {
        TTV_ErrorCode customMuxerErrorCode = mCustomMuxer->Start(params);
        if (TTV_FAILED(customMuxerErrorCode)) {
          ttv::trace::Message(
            "Streamer", MessageLevel::Error, "Inside Streamer::Start - Call to mCustomMuxer->Start(params) failed");
        }

        customMuxerErrorCode = mCustomMuxer->WriteVideoSpsPps(params.videoSps, params.videoPps);

        if (mFlvMuxer == nullptr) {
          ec = customMuxerErrorCode;
        }
      }
    }
  }

  // Allow frames to be written to the muxer now that the connection has been established
  if (TTV_SUCCEEDED(ec)) {
    mFrameWriter->SetFlvMuxer(mFlvMuxer);
    mFrameWriter->SetCustomMuxer(mCustomMuxer);

    ec = mFrameWriter->Start(startParams.videoParams);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  }

  // Kick off capture of audio
  if (TTV_SUCCEEDED(ec)) {
    if (audioEnabled) {
      ec = mAudioStreamer->StartCapture();
      TTV_ASSERT(TTV_SUCCEEDED(ec));
    }
  }

  // Kick off capture of video
  if (TTV_SUCCEEDED(ec)) {
    ec = mVideoStreamer->StartCapture();
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  }

  if (TTV_SUCCEEDED(ec)) {
    // schedule a request for the stream info
    mStreamInfoFetchTimer.Clear();
    mStreamInfoFetchTimer.ScheduleNextRetry();

    // keep a copy of the start params
    mStartParams = startParams;
    mAudioParams = audioParams;

    mBandwidthWarningTimer.Set(kBandwidthWarningIntervalMilliseconds);
  }

  if (TTV_SUCCEEDED(ec)) {
    // This atomic state change will ensure start is not entered by 2 threads at once
    std::unique_lock<std::mutex> lock(mStateMutex);
    mStreamerState = StreamerState::Started;
    mStateChangeError = TTV_EC_SUCCESS;
    NotifyStreamerStateChanged();
  } else {
    InternalStop(ec, false, nullptr);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::Streamer::GetStreamKey(std::function<void(TTV_ErrorCode ec)>&& callback) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "Streamer::GetStreamKey()");

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto complete = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  graphql::GetStreamKeyQueryInfo::InputParams inputParams;
  inputParams.authToken = oauthToken->GetToken();
  inputParams.userId = std::to_string(user->GetUserId());

  auto resultCallback = [this, complete, callback](GraphQLTask<graphql::GetStreamKeyQueryInfo>* source,
                          Result<graphql::GetStreamKeyQueryInfo::PayloadType>&& result) {
    CompleteTask(source);

    if (result.IsError()) {
      ttv::trace::Message(kLoggerName, MessageLevel::Debug, "Failed to fetch stream key");
      mStreamKey.clear();
      mChannelId = 0;

      complete(result.GetErrorCode());
      callback(result.GetErrorCode());
      return;
    }

    auto& data = result.GetResult();

    if (data.channel.HasValue()) {
      auto channel = data.channel.Value();
      if (channel.videoStreamSettings.HasValue()) {
        auto videoStreamSettings = channel.videoStreamSettings.Value();
        auto streamKeyVariant = videoStreamSettings.streamKey.streamKeyVariant;
        if (streamKeyVariant.Is<graphql::GetStreamKeyQueryInfo::StreamKey>()) {
          mStreamKey = streamKeyVariant.As<graphql::GetStreamKeyQueryInfo::StreamKey>().value;
        } else {
          mStreamKey.clear();

          CanTheyError error = [&streamKeyVariant]() -> CanTheyError {
            if (!streamKeyVariant.Is<graphql::GetStreamKeyQueryInfo::StreamKeyError>()) {
              return {};
            }

            auto streamKeyError = streamKeyVariant.As<graphql::GetStreamKeyQueryInfo::StreamKeyError>();

            CanTheyError error;
            error.code = streamKeyError.code;
            error.message = streamKeyError.message;
            error.links = streamKeyError.links;

            return error;
          }();

          // Handle Stream Key Error response.
          mListeners.Invoke(
            [this, &error](const std::shared_ptr<IListener>& listener) { listener->OnStreamKeyError(this, error); });
        }
      }
    }

    mChannelId = ttv::graphql::GQLUserIdToChannelId(data.id);

    complete(TTV_EC_SUCCESS);
    callback(TTV_EC_SUCCESS);
  };

  auto task = std::make_shared<GraphQLTask<graphql::GetStreamKeyQueryInfo>>(std::move(inputParams), resultCallback);

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_FAILED(ec)) {
    complete(ec);
  }

  return ec;
}

/*
 * Notifies clients when the streamer's state has changed. Only call while the mStateMutex is locked.
 */
void ttv::broadcast::Streamer::NotifyStreamerStateChanged() {
  StreamerState state = mStreamerState;

  if (state == mLastReportedStreamerState) {
    return;
  }

  mLastReportedStreamerState = state;

  TTV_ErrorCode stateChangeError = mStateChangeError;
  mStateChangeError = TTV_EC_SUCCESS;

  // Called on the main thread
  LambdaTask::CompleteCallback callback = [this, state, stateChangeError](LambdaTask* source, TTV_ErrorCode /*ec*/) {
    CompleteTask(source);

    mListeners.Invoke([this, state, stateChangeError](const std::shared_ptr<IListener>& listener) {
      listener->OnStreamerStateChanged(this, state, stateChangeError);
    });
  };

  auto task = std::make_shared<LambdaTask>(nullptr, callback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to start task, notify of streamer state change");
  }
}

TTV_ErrorCode ttv::broadcast::Streamer::Stop(const std::string& reason, StopCallback&& callback) {
  AutoTracer tracer("Streamer", MessageLevel::Debug, "Streamer::Stop()");

  return InternalStop(TTV_EC_SUCCESS, true, [this, callback = std::move(callback), reason](TTV_ErrorCode ec) {
    if (TTV_SUCCEEDED(ec)) {
      mTrackingContext->TrackEvent("mobile_broadcast_stream_ended", {{"reason", reason}});
    }

    if (callback != nullptr) {
      callback(ec);
    }
  });
}

TTV_ErrorCode ttv::broadcast::Streamer::InternalStop(TTV_ErrorCode ec, bool solicited, StopCallback&& callback) {
  AutoTracer tracer("Streamer", MessageLevel::Debug, "Streamer::InternalStop()");

  {
    std::unique_lock<std::mutex> lock(mStateMutex);
    if (mStreamerState == StreamerState::Stopped) {
      return TTV_EC_BROADCAST_STREAM_NOT_STARTED;
    } else if (mStreamerState == StreamerState::Stopping) {
      return TTV_EC_REQUEST_PENDING;
    } else {
      mStreamerState = StreamerState::Stopping;
      mStateChangeError = ec;
      NotifyStreamerStateChanged();
    }
  }

  LambdaTask::CompleteCallback taskCallback = [this, solicited, callback = std::move(callback)](
                                                LambdaTask* source, TTV_ErrorCode callbackEc) {
    CompleteTask(source);

    if (callback != nullptr) {
      callback(callbackEc);
    }

    // Requested by the client
    if (solicited) {
      mListeners.Invoke(
        [this, callbackEc](std::shared_ptr<IListener> listener) { listener->OnStopComplete(this, callbackEc); });
    }

    mMinuteBroadcastTrackingTimer.Clear();
  };

  // Called on a worker thread
  LambdaTask::JobCallback job = [this]() -> TTV_ErrorCode {
    if (mAudioStreamer != nullptr) {
      mAudioStreamer->Stop();
    }

    if (mVideoStreamer != nullptr) {
      TTV_ErrorCode stopEc = mVideoStreamer->Stop();
      ASSERT_ON_ERROR(stopEc);
    }

    if (mFrameWriter != nullptr) {
      mFrameWriter->Shutdown();
      mFrameWriter.reset();
    }

    if (mVideoEncoder != nullptr) {
      mVideoEncoder->SetFrameWriter(nullptr);
    }

    if (mAudioEncoder != nullptr) {
      mAudioEncoder->SetFrameWriter(nullptr);
    }

    if (mFlvMuxer != nullptr) {
      TTV_ASSERT(mFlvMuxer.unique());
      mFlvMuxer.reset();
    }

    if (mCustomMuxer != nullptr) {
      mCustomMuxer.reset();
    }

    mInitialTime = 0;
    mFirstFrameSubmitted = false;
    mBandwidthWarningState = TTV_EC_SUCCESS;
    mBandwidthWarningTimer.Clear();

    {
      std::unique_lock<std::mutex> lock(mStateMutex);
      mStreamerState = StreamerState::Stopped;
      mContext->isBusy = false;
      mStateChangeError = TTV_EC_SUCCESS;
      NotifyStreamerStateChanged();
    }

    // Clear the stream info
    mStreamInfo.reset();

    return TTV_EC_SUCCESS;
  };

  auto task = std::make_shared<LambdaTask>("StopStream", job, taskCallback);

  ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to start task, can't stop stream async");
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::Streamer::TrackStartFailure(TTV_ErrorCode ec, bool synchronous) {
  if (!mBandwidthTestMode) {
    return mTrackingContext->TrackEvent(
      "mobile_broadcast_start_failure", {{"error_code", ttv::ErrorToString(ec)}, {"synchronous", synchronous}});
  }
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::Streamer::TrackStreamFailure(TTV_ErrorCode ec) {
  if (!mBandwidthTestMode) {
    return mTrackingContext->TrackEvent("mobile_broadcast_failure", {{"error_code", ttv::ErrorToString(ec)}});
  }
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::Streamer::TrackMinuteBroadcast() {
  if (!mBandwidthTestMode) {
    uint64_t averageRecommendedKbps = 0;
    uint64_t averageEncodedKbps = 0;
    bool gotStats = TTV_SUCCEEDED(mFrameWriter->GatherTrackingStats(averageRecommendedKbps, averageEncodedKbps));

    return mTrackingContext->TrackEvent("mobile_minute_broadcast",
      {{"broadcast_id",
         ((mStreamInfo != nullptr) ? TrackingValue{std::to_string(mStreamInfo->streamId)} : TrackingValue{nullptr})},
        {"category", mContext->category}, {"stream_name", mContext->title},
        {"average_recommended_video_bitrate_kbps",
          (gotStats ? TrackingValue{static_cast<int64_t>(averageRecommendedKbps)} : TrackingValue{nullptr})},
        {"average_video_output_bitrate_kbps",
          (gotStats ? TrackingValue{static_cast<int64_t>(averageEncodedKbps)} : TrackingValue{nullptr})}});
  }
  return TTV_EC_SUCCESS;
}

uint64_t ttv::broadcast::Streamer::GetStreamTime() const {
  {
    std::unique_lock<std::mutex> lock(mStateMutex);
    // not playing
    if (mStreamerState != StreamerState::Started) {
      return std::numeric_limits<uint64_t>::max();
    }
  }

  return SystemTimeToMs(GetSystemClockTime() - mInitialTime);
}

void ttv::broadcast::Streamer::GetRequiredAuthScopes(std::vector<std::string>& scopes) {
  scopes.insert(scopes.end(), kRequiredScopes, kRequiredScopes + sizeof(kRequiredScopes) / sizeof(kRequiredScopes[0]));
}

TTV_ErrorCode ttv::broadcast::Streamer::AddVideoFrame(const std::shared_ptr<VideoFrame>& frame) {
  // ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::AddVideoFrame()");

  // Capture these here in case the stream is shutdown while submitting
  // TODO: We should guard here in a different way
  auto frameWriter = mFrameWriter;
  auto videoStreamer = mVideoStreamer;
  auto videoEncoder = mVideoEncoder;
  auto audioStreamer = mAudioStreamer;

  if (mStreamerState != StreamerState::Started || frameWriter == nullptr || videoEncoder == nullptr ||
      videoStreamer == nullptr) {
    return TTV_EC_BROADCAST_STREAM_NOT_STARTED;
  }

  TTV_ASSERT(frame != nullptr);
  TTV_RETURN_ON_NULL(frame, TTV_EC_BROADCAST_INVALID_VIDEOFRAME);

  // Check for frame backup
  UpdateBandwidthWarningState();

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  // Add the frame
  if (TTV_SUCCEEDED(ec)) {
    // Generate a timestamp if needed
    uint64_t timestamp = frame->GetTimeStamp();
    timestamp = timestamp > 0 ? timestamp : GetSystemClockTime();

    // On submission of first frame we need to set the initial timestamp and start streamer threads
    if (!mFirstFrameSubmitted) {
      mFirstFrameSubmitted = true;

      mInitialTime = timestamp;
      videoStreamer->SetInitialTime(mInitialTime);

      if (audioStreamer != nullptr) {
        audioStreamer->SetInitialTime(mInitialTime);
      }
    }

    // Adjust timestamp to be relative to stream start
    frame->SetTimeStamp(timestamp - mInitialTime);

    ec = videoStreamer->ProcessFrame(frame);
  }

  // Check the frame writer for fatal errors
  if (TTV_SUCCEEDED(ec)) {
    ec = frameWriter->GetLastError();
  }

  // Perform automatic bitrate adjustment
  if (TTV_SUCCEEDED(ec)) {
    if (mStartParams.videoParams.automaticBitRateAdjustmentEnabled && videoEncoder->SupportsBitRateAdjustment()) {
      uint32_t kbps = frameWriter->GetRecommendedBitRate() / 1000;
      videoEncoder->SetTargetBitRate(kbps);
    }
  }

  // Terminate the stream on failure
  if (TTV_FAILED(ec)) {
    TrackStreamFailure(ec);
    InternalStop(ec, false, nullptr);
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::Streamer::UpdateBandwidthWarningState() {
  // Not time to check yet
  if (!mBandwidthWarningTimer.Check(true)) {
    return mBandwidthWarningState;
  }

  // Schedule the next check
  mBandwidthWarningTimer.Set(kBandwidthWarningIntervalMilliseconds);

  TTV_ErrorCode ec;

  switch (mFrameWriter->GetDelayState()) {
    case FrameWriter::DelayState::Error: {
      ec = TTV_EC_BROADCAST_FRAME_QUEUE_TOO_LONG;
      break;
    }
    case FrameWriter::DelayState::Warning: {
      ec = TTV_EC_BROADCAST_FRAMES_QUEUEING;
      break;
    }
    case FrameWriter::DelayState::Okay: {
      // Was healthy before so nothing to do
      if (mBandwidthWarningState == TTV_EC_SUCCESS) {
        return mBandwidthWarningState;
      }

      ec = TTV_EC_SUCCESS;
      break;
    }
  }
  mBandwidthWarningState = ec;

  uint64_t queueDelayMilliseconds = mFrameWriter->GetQueueDelayInMilliseconds();

  ttv::trace::Message("Streamer", MessageLevel::Info, "Streamer::UpdateBandwidthWarningState(): %s %u",
    ErrorToString(mBandwidthWarningState), queueDelayMilliseconds);

  // Notify clients on the main thread
  TTV_ErrorCode bandwidthError = mBandwidthWarningState;
  LambdaTask::CompleteCallback callback = [this, bandwidthError, queueDelayMilliseconds](
                                            LambdaTask* source, TTV_ErrorCode /*ec*/) {
    CompleteTask(source);

    // Notify clients
    mListeners.Invoke([this, bandwidthError, queueDelayMilliseconds](const std::shared_ptr<IListener>& listener) {
      listener->OnBandwidthWarning(this, bandwidthError, static_cast<uint32_t>(queueDelayMilliseconds));
    });
  };

  auto task = std::make_shared<LambdaTask>("BandwidthWarning", nullptr, callback);

  ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    Log(MessageLevel::Error, "Failed to start task, can't notify of bandwidth warning");
  }

  return mBandwidthWarningState;
}

TTV_ErrorCode ttv::broadcast::Streamer::SetIngestServer(const IngestServer& ingestServer) {
  mSelectedIngestServer = ingestServer;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::Streamer::SetStreamName(const std::string& streamKey, bool absEnabled) {
  ttv::trace::Message("Streamer", MessageLevel::Debug, "Streamer::SetStreamName()");

  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  TTV_ASSERT(streamKey.size() > 0);
  TTV_ASSERT(!mSelectedIngestServer.serverName.empty());

  if (mSelectedIngestServer.serverName.empty()) {
    ttv::trace::Message("Streamer", MessageLevel::Error, "Inside Streamer::SetStreamName - Invalid ingest server");
    ret = TTV_EC_BROADCAST_INVALID_INGEST_SERVER;
  } else {
    mOutputStreamName = mSelectedIngestServer.serverUrl;

    const char* kStreamKeyTemplate = "{stream_key}";
    if (streamKey.size() > 0) {
      size_t replacePos = mOutputStreamName.find(kStreamKeyTemplate);
      TTV_ASSERT(replacePos != std::string::npos);
      if (replacePos == std::string::npos) {
        ttv::trace::Message("Streamer", MessageLevel::Error,
          "Inside Streamer::SetStreamName - Couldn't find {stream_key} in server URL to replace");
        return TTV_EC_BROADCAST_INVALID_INGEST_SERVER;
      }

      std::string clientId;
      GetClientId(clientId);

      mOutputStreamName.replace(replacePos, strlen(kStreamKeyTemplate), streamKey);
      mOutputStreamName.append("?client_id=");
      mOutputStreamName.append(clientId);
      mOutputStreamName.append("&sdk_version=");
      mOutputStreamName.append(VersionString());
      mOutputStreamName.append("&video_encoder=");
      mOutputStreamName.append(mVideoStreamer->GetEncoderName());

      // Add "-ABS" to the end of the video encoder tag if we're using ABS.
      if (absEnabled && mVideoEncoder->SupportsBitRateAdjustment()) {
        mOutputStreamName.append(kAbsVideoEncoderSuffix);
      }

      mOutputStreamName.append("&os=");
      mOutputStreamName.append(GetOSName());
      mOutputStreamName.append("&broadcaster=");
      mOutputStreamName.append(mBroadcasterSoftware);

      if (mForceArchiveBroadcast) {
        mOutputStreamName.append("&recorder=1");
      } else if (mForceDontArchiveBroadcast) {
        mOutputStreamName.append("&recorder=0");
      }

      if (mBandwidthTestMode) {
        mOutputStreamName.append("&bandwidthtest=true");
      }
    } else {
      ttv::trace::Message("Streamer", MessageLevel::Error, "Inside Streamer::SetStreamName - No stream key");
      ret = TTV_EC_BROADCAST_NO_STREAM_KEY;
    }
  }

  return ret;
}

TTV_ErrorCode ttv::broadcast::Streamer::GetVolume(AudioLayerId layer, float& volume) const {
  if (mAudioStreamer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  volume = mAudioStreamer->GetVolume(layer);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::Streamer::SetVolume(AudioLayerId layer, float volume) {
  if (mAudioStreamer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  assert(volume >= -1.f && volume <= 1.f);

  mAudioStreamer->SetVolume(layer, volume);
  return TTV_EC_SUCCESS;
}

void ttv::broadcast::Streamer::SetAudioCapturer(AudioLayerId layer, const std::shared_ptr<IAudioCapture>& capturer) {
  mAudioStreamer->SetCapturer(layer, capturer);
}

void ttv::broadcast::Streamer::SetAudioCapturerEnabled(AudioLayerId layer, bool enabled) {
  mAudioStreamer->SetCapturerEnabled(layer, enabled);
}

void ttv::broadcast::Streamer::GetAudioCapturers(std::vector<std::shared_ptr<IAudioCapture>>& result) const {
  mAudioStreamer->GetCapturers(result);
}

std::shared_ptr<IAudioCapture> ttv::broadcast::Streamer::GetAudioCapturer(AudioLayerId layer) const {
  return mAudioStreamer->GetCapturer(layer);
}

void ttv::broadcast::Streamer::SetVideoCapturer(std::shared_ptr<IVideoCapture> capturer) {
  mVideoCapturer = capturer;

  if (mVideoStreamer != nullptr) {
    mVideoStreamer->SetCapturer(mVideoCapturer);
  }
}

std::shared_ptr<ttv::broadcast::IVideoCapture> ttv::broadcast::Streamer::GetVideoCapturer() const {
  return mVideoCapturer;
}
