/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/task/channelinfotask.h"

#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"

// curl -H "Client-Id: $CLIENTID" -H "Accept: application/vnd.twitchtv.v5+json" -H "Authorization: OAuth $OAUTH"
// "https://api.twitch.tv/kraken/channel" | python -m json.tool

namespace {
const char* kChannelInfoURL = "https://api.twitch.tv/kraken/channel";
}

ttv::broadcast::ChannelInfoTask::ChannelInfoTask(const std::string& oauthToken, Callback callback)
    : HttpTask(oauthToken), mCallback(callback) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChannelInfoTask created");
}

void ttv::broadcast::ChannelInfoTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kChannelInfoURL;
  requestInfo.httpReqType = HTTP_GET_REQUEST;
  requestInfo.requestHeaders.emplace_back("Accept", "application/vnd.twitchtv.v5+json");
}

void ttv::broadcast::ChannelInfoTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChannelInfoTask::ProcessResponse entered");

  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    // Make sure we get at least the name of the channel and its stream key
    //
    if (jsonVal["stream_key"].empty()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "No JSON stream_key element found or it's empty");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_NO_STREAMKEY;
      return;
    }
    if (jsonVal["name"].empty()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "No JSON name element found or it's empty");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_NO_CHANNELNAME;
      return;
    }

    if (jsonVal["_id"].empty()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "No JSON name element found or it's empty");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();
    // Extract the channel info fields
    //
    std::string streamKey = jsonVal["stream_key"].asString();
    TTV_ASSERT(streamKey.size() > 0);
    mResult->streamKey = streamKey;

    std::string name = jsonVal["name"].asString();
    TTV_ASSERT(name.size() > 0);
    mResult->loginName = name;

    ParseChannelId(jsonVal["_id"], mResult->channelId);

    if (!jsonVal["url"].empty()) {
      std::string url = jsonVal["url"].asString();
      TTV_ASSERT(url.size() > 0);
      mResult->url = url;
    }

    if (!jsonVal["display_name"].empty()) {
      std::string displayName = jsonVal["display_name"].asString();
      TTV_ASSERT(displayName.size() > 0);
      mResult->displayName = displayName;
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::broadcast::ChannelInfoTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
