/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/task/gamestreamstask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"

namespace {
const char* kStreamsURL = "https://api.twitch.tv/kraken/streams";
}

// curl -H "Client-Id: $CLIENTID" -H "Accept: application/vnd.twitchtv.v5+json" -H "Authorization: OAuth $OAUTH"
// "https://api.twitch.tv/kraken/streams" | python -m json.tool

ttv::broadcast::GameStreamsTask::GameStreamsTask(
  const std::string& gameName, uint limit, uint offset, Callback callback)
    : mCallback(callback), mGameName(gameName), mLimit(limit), mOffset(offset) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "GameStreamsTask created");
}

void ttv::broadcast::GameStreamsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  Uri url;
  url.SetUrl(kStreamsURL);
  url.SetParam("game", mGameName);
  url.SetParam("limit", mLimit);
  url.SetParam("offset", mOffset);

  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_GET_REQUEST;
  requestInfo.requestHeaders.emplace_back("Accept", "application/vnd.twitchtv.v5+json");
}

void ttv::broadcast::GameStreamsTask::ProcessResponse(uint /*status*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (jsonVal["streams"].empty() || !jsonVal["streams"].isArray()) {
      // No live streams for the game
      mTaskStatus = TTV_EC_SUCCESS;
      return;
    }

    mResult = std::make_shared<Result>();

    // Get the number of live streams
    auto streamCount = jsonVal["streams"].size();

    // Read the streams and add them to the result list
    for (size_t i = 0; i < streamCount; ++i) {
      if (!jsonVal["streams"][i].empty()) {
        LiveGameStreamInfo streamInfo;
        if (!jsonVal["streams"][i]["channel"].empty()) {
          streamInfo.channelUrl = jsonVal["streams"][i]["channel"]["url"].asString();
          // We must at least have a valid (non-empty) URL for the channel
          if (!streamInfo.channelUrl.empty()) {
            streamInfo.streamTitle = jsonVal["streams"][i]["channel"]["status"].asString();
            streamInfo.channelDisplayName = jsonVal["streams"][i]["channel"]["display_name"].asString();

            streamInfo.previewUrlTemplate = jsonVal["streams"][i]["preview"].asString();
            streamInfo.viewerCount = static_cast<unsigned int>(jsonVal["streams"][i]["viewers"].asUInt());

            mResult->streams.push_back(streamInfo);
          }
        }
      }
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::broadcast::GameStreamsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
