/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#if 0

#include "twitchsdk/broadcast/internal/task/implicitgrantlistenertask.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/winsocket.h"

#if TTV_PLATFORM_WIN32
#include <Shellapi.h>
#endif

// TODO: Implement for non windows platforms

namespace
{
    const char* kAuthURL = "https://api.twitch.tv/kraken/oauth2/authorize";
}

//-----------------------------------------------------------------------
ttv::broadcast::ImplicitGrantListenerTask::ImplicitGrantListenerTask(TTV_TaskCallback clientCallback,
                                                          void* userData,
                                                          const std::string& clientId,
                                                          const std::string& port,
                                                          const std::string& successRedirect,
                                                          const std::string& failureRedirect,
                                                          const std::vector<std::string>& scopes,
                                                          TwitchAPI::AuthToken* authToken)
:   Task(clientCallback, userData)
,   mClientId(clientId)
,   mPort(port)
,   mSuccessRedirect(successRedirect)
,   mFailureRedirect(failureRedirect)
,   mResult(TTV_EC_SUCCESS)
,   mResultAuthToken(authToken)
{
    assert (mResultAuthToken);

    for (auto const& scope: scopes)
    {
        mScopes += scope + "+";
    }
    mScopes.pop_back();
}


using urlParameter_t = std::pair<std::string, std::string>;
using urlParameters_t = std::vector<urlParameter_t>;

void ParseHeader(const uint8_t* header,
                 size_t headerSize,
                 std::string& verb,
                 std::string& url,
                 urlParameters_t& params)
{
    using std::begin;
    using std::end;
    using std::find;
    using std::find_first_of;
    using std::string;

    const uint8_t urlDelims[] = {' ', '?'};
    const uint8_t newlineDelims[] = {'\n', '\r'};

    const uint8_t* pos = header;
    const uint8_t* endOfLine = find_first_of(pos, pos + headerSize, begin(newlineDelims), end(newlineDelims));

    if (endOfLine != header+headerSize)
    {
        verb = string(pos, find(pos, endOfLine, ' '));
        pos += verb.length();

        if (*pos++ == ' ')
        {
            url = string(pos, find_first_of(pos, endOfLine, begin(urlDelims), end(urlDelims)));
            pos += url.length();

            if (*pos++ == '?')
            {
                const uint8_t* endOfParams = find (pos, endOfLine, ' ');
                if (endOfParams != pos)
                {
                    const uint8_t* delim = nullptr;
                    do
                    {
                        delim = find(pos, endOfParams, '&');
                        const uint8_t* equalSign = find(pos, delim, '=');
                        string paramName(pos, equalSign);
                        string paramValue;
                        if (equalSign != delim)
                        {
                            paramValue = string (equalSign+1, delim);
                        }

                        params.push_back( urlParameter_t(paramName, paramValue) );

                        pos = delim+1;
                    } while (delim != endOfParams);
                }
            }
        }
    }
}

static std::string CreateHTTPHeader(const std::string& content)
{
    std::stringstream header;
    header << ""
        "HTTP/1.1 200 OK\r\n"
        "Server: twitchSDK/0.0.1\r\n"
        "Content-Type: text/html; charset=UTF-8\r\n"
        "Content-Length: ";

    header << content.length();
    header << "\r\n""\r\n";

    return header.str();
}

void ttv::broadcast::ImplicitGrantListenerTask::Run()
{
#if TTV_PLATFORM_WIN32
    const std::string kHttpError404 = ""
        "HTTP/1.1 404 Not Found\r\n"
        "Server: twitchSDK/0.0.1\r\n"
        "Content-Type: text/html; charset=UTF-8\r\n"
        "Content-Length: 0";

    const std::string kFragmentToParams = ""
        "<script>\r\n"
        "var hash = document.location.hash.slice(1);\r\n"
        "if (hash.length > 0) {\r\n"
            "window.location.replace(window.location.href.split('#')[0]+'?'+hash);\r\n"
        "}else{\r\n"
            "window.location.replace(window.location+\"?error='No Parameters Passed'\")\r\n"
        "}</script>\r\n";

    std::stringstream urlBuilder;
    urlBuilder << kAuthURL << "?response_type=token";
    urlBuilder << "&client_id=" << mClientId;
    urlBuilder << "&redirect_uri=http://localhost:" << mPort;
    urlBuilder << "&scope=" << mScopes;

    TTV_ErrorCode ret = TTV_EC_SUCCESS;
    auto shellExecuteResult = ShellExecuteA(NULL, "open", urlBuilder.str().c_str(), NULL, NULL, SW_SHOWNORMAL);
    if (shellExecuteResult != ERROR_SUCCESS)
    {
        uint32_t port = 0;

        if (!ParseNum(mPort, port))
        {
            ret = TTV_EC_INVALID_ARG;
        }

        // TODO: This is broken until we determine if we want to support incoming connections
        if (TTV_SUCCEEDED(ret))
        {
            //std::string uri = BuildUri("", "127.0.0.1", port);
            //ret = CreateSocket(uri, mLoginListener);
        }

        if (TTV_SUCCEEDED(ret))
        {
            //mLoginListener->TCPListen("127.0.0.1", mPort.c_str());
        }

        // TODO: This should not be platform specific
        std::shared_ptr<WinSocket> winSocket = std::static_pointer_cast<WinSocket>(mLoginListener);

        while (TTV_SUCCEEDED(ret))
        {
            BufferedSocket socket;
            uint8_t buffer[1000];
            size_t read = 0;

            std::shared_ptr<ISocket> connectedSocket;
            ret = winSocket->AcceptConnection(connectedSocket);

            if TTV_SUCCEEDED(ret)
            {
                socket.Bind(winSocket);
                socket.SetBlockingMode(true);

                ret = socket.Recv(buffer, 999, read, 10000);
            }

            if TTV_SUCCEEDED(ret)
            {
                buffer[read] = 0;

                std::string verb;
                std::string url;
                urlParameters_t params;
                ParseHeader(buffer, read, verb, url, params);

                if (verb != "GET" || url != "/")
                {
                    socket.Send(((uint8_t*)kHttpError404.c_str()), kHttpError404.length(), false);
                }
                else if (params.size() == 0)
                {
                    std::string header = CreateHTTPHeader(kFragmentToParams);
                    socket.Send(((uint8_t*)header.c_str()), header.length(), false);
                    socket.Send(((uint8_t*)kFragmentToParams.c_str()), kFragmentToParams.length(), false);
                }
                else
                {
                    for (auto const& param: params)
                    {
                        if (param.first == "access_token")
                        {
                            strcpy (mResultAuthToken->data, param.second.c_str());
                        }
                        else if (param.first == "error")
                        {
                            mHttpError = param.second;
                            mResult = TTV_EC_UNKNOWN_ERROR;
                        }
                        else if (param.first == "error_description")
                        {
                            mHttpErrorDescription = param.second;
                        }
                    }

                    std::stringstream redirectScript;

                    redirectScript << ""
                        "<script>\r\n"
                        "window.location.replace('";

                    if (TTV_SUCCEEDED(mResult))
                    {
                        redirectScript << mSuccessRedirect;
                    }
                    else
                    {
                        redirectScript << mFailureRedirect;
                    }

                    bool firstParam = mFailureRedirect.find("?") == std::string::npos;

                    for (auto const& param: params)
                    {
                        redirectScript << (firstParam ? "?" : "&");
                        redirectScript << param.first;
                        if (param.second.length())
                        {
                            redirectScript << "=" << param.second;
                        }
                        firstParam = false;
                    }

                    redirectScript << "');\r\n"
                        "</script>\r\n";


                    std::string redirectString = redirectScript.str();
                    std::string header = CreateHTTPHeader(redirectString);
                    socket.Send(((uint8_t*)header.c_str()), header.length(), false);
                    socket.Send(((uint8_t*)redirectString.c_str()), redirectString.length(), false);

                    break;
                }
            }
        }
    }
#endif  // TTV_PLATFORM_WIN32
}

void ttv::broadcast::ImplicitGrantListenerTask::OnComplete()
{
    //TODO if error string is no t empty ttv::trace it
    if (mCallback)
    {
        if (mAborted)
        {
            mResult = TTV_EC_REQUEST_ABORTED;
        }

        mCallback(mResult, mUserData);
    }
}

#endif
