/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/task/ingestlisttask.h"

#include "twitchsdk/core/json/reader.h"

#include <cstring>
#include <string>

// curl -X GET -H "Client-Id: $CLIENTID" -H "Accept: application/vnd.twitchtv.v5+json"
// "https://ingest.twitch.tv/api/v2/ingests" | python -m json.tool

namespace {
const char* kIngestListURL = "https://ingest.twitch.tv/api/v2/ingests";
}  // namespace

ttv::broadcast::IngestListTask::IngestListTask(ChannelId channelId, Callback callback)
    : mChannelId(channelId), mCallback(callback) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "IngestListTask created");
}

void ttv::broadcast::IngestListTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kIngestListURL;
  requestInfo.httpReqType = HTTP_GET_REQUEST;
  requestInfo.requestHeaders.emplace_back("Accept", "application/vnd.twitchtv.v5+json");
  requestInfo.requestHeaders.emplace_back("X-Channel-ID", std::to_string(mChannelId));
}

void ttv::broadcast::IngestListTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (jsonVal["ingests"].empty()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "ingests is null or empty");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_NO_INGESTS;
      return;
    }

    mResult = std::make_shared<Result>();

    // Get the number of ingest servers
    const auto& jIngests = jsonVal["ingests"];

    // Read the ingest servers and add them to the result list
    for (const auto& jServer : jIngests) {
      IngestServer server;
      server.serverId = static_cast<uint32_t>(jServer["_id"].asUInt());
      server.serverName = jServer["name"].asString();
      server.priority = static_cast<uint32_t>(jServer["priority"].asUInt());

      server.serverUrl = jServer["url_template"].asString();

      mResult->ingestList.push_back(server);
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::broadcast::IngestListTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
