/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/task/matchgamenamestask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"

// https://github.com/justintv/Twitch-API/blob/master/v2_resources/search.md#get-searchgames

namespace {
const char* kGamesURL = "https://api.twitch.tv/kraken/search/games";
}

// curl -H "Client-Id: $CLIENTID" -H "Accept: application/vnd.twitchtv.v5+json" -H "Authorization: OAuth $OAUTH"
// "https://api.twitch.tv/kraken/search/games?query=overwatch&type=suggest" | python -m json.tool

ttv::broadcast::MatchGameNamesTask::Result::Result() {}

ttv::broadcast::MatchGameNamesTask::MatchGameNamesTask(const std::string& str, Callback /*callback*/) : mString(str) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "MatchGameNamesTask created");
}

void ttv::broadcast::MatchGameNamesTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  Uri url;
  url.SetUrl(kGamesURL);
  url.SetParam("query", mString);
  url.SetParam("type", "suggest");

  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_GET_REQUEST;
  requestInfo.requestHeaders.emplace_back("Accept", "application/vnd.twitchtv.v5+json");
}

void ttv::broadcast::MatchGameNamesTask::ProcessResponse(uint /*status*/, const std::vector<char>& response) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "MatchGameNamesTask::ProcessResponse entered");

  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonRoot;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonRoot);

    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    json::Value jsonGames = jsonRoot.get("games", json::Value::null);
    if (!jsonGames.isArray()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "Invalid response, missing/invalid games element");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    mResult = std::make_shared<Result>();

    GameInfoList& result = mResult->games;
    result.games.resize(static_cast<size_t>(jsonGames.size()));

    for (size_t i = 0; i < jsonGames.size(); ++i) {
      json::Value jsonGame = jsonGames[i];

      if (!jsonGame.isObject()) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Info, "Invalid response, invalid game element");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        return;
      }

      json::Value jsonName = jsonGame.get("name", json::Value::null);
      json::Value jsonId = jsonGame.get("_id", 0);
      json::Value jsonPopularity = jsonGame.get("popularity", 0);

      if (jsonName.isNull()) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Info, "Invalid response, invalid data in game element");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        return;
      }

      result.games[i].name = jsonName.asString();
      result.games[i].gameId = static_cast<uint32_t>(jsonId.asInt());
      result.games[i].popularity = static_cast<uint32_t>(jsonPopularity.asInt());
    }
  }
}

void ttv::broadcast::MatchGameNamesTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
