/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/task/metadatatask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/systemclock.h"

#include <sstream>

// https://github.com/justintv/sdk/wiki/Metadata-API-v1
// https://github.com/justintv/sdk/wiki/Metadata-API-v1-Event-Schema

namespace {
const char* kMetadataURL = "https://api.twitch.tv/metadata/events";
}

ttv::broadcast::MetaDataTask::MetaDataTask(const std::string& oauthToken, uint64_t streamId, EventClass eventClass,
  const std::string& name, int64_t streamTime, unsigned long sequenceNumber, const std::string& humanDescription,
  json::Value&& metaData, Callback callback)
    : HttpTask(oauthToken),
      mCallback(callback),
      mStreamId(streamId),
      mEventClass(eventClass),
      mName(name),
      mStreamTime(streamTime),
      mSequenceNumber(sequenceNumber),
      mHumanDescription(humanDescription),
      mMetaData(metaData) {
  ttv::trace::Message("MetaDataTask", MessageLevel::Info, "MetaDataTask created");
}

void ttv::broadcast::MetaDataTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  Uri url(kMetadataURL);
  url.SetParam("broadcast_id", mStreamId);

  requestInfo.url = kMetadataURL + std::to_string(mStreamId);
  requestInfo.httpReqType = HTTP_POST_REQUEST;

  std::string type;
  std::string sequenceId;

  switch (mEventClass) {
    case EventClass::Action:
      type = "action";
      break;
    case EventClass::Start:
      type = "start";
      sequenceId = std::to_string(mSequenceNumber);
      break;
    case EventClass::End:
      type = "end";
      sequenceId = std::to_string(mSequenceNumber);
      break;
  }

  ttv::json::Value data;
  data["event"]["type"] = type;
  data["event"]["name"] = mName;
  data["event"]["time_offset"] = SystemTimeToMs(static_cast<uint64_t>(mStreamTime));  // TODO: This is very suspect
  data["event"]["description"] = mHumanDescription;
  data["event"]["data"] = mMetaData.toStyledString();

  if (sequenceId.size() > 0) {
    data["event"]["sequence_id"] = sequenceId;
  }

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(data);

  requestInfo.requestHeaders.emplace_back("Accept", "application/vnd.twitchtv.v1+json");
}

void ttv::broadcast::MetaDataTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
