/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/task/recordingstatustask.h"

#include "twitchsdk/core/json/reader.h"

namespace {
#define CLIENTID_TEMPLATE "{client_id}"
const char* kClientIdTemplate = CLIENTID_TEMPLATE;
const char* kRecordingStatusURL = "https://api.twitch.tv/kraken/clients/" CLIENTID_TEMPLATE "/video_status";
}  // namespace

// curl -H "Accept: application/vnd.twitchtv.v5+json" -H "Authorization: OAuth $OAUTH"
// "https://api.twitch.tv/kraken/clients/$CLIENTID/video_status" | python -m json.tool

// TODO: This endpoint does not seem to be working as it used to

namespace ttv {
TTV_ErrorCode GetClientId(std::string& clientId);
}

ttv::broadcast::RecordingStatusTask::Result::Result() {
  state.recordingEnabled = false;
}

ttv::broadcast::RecordingStatusTask::RecordingStatusTask(const std::string& oauthToken, Callback callback)
    : HttpTask(oauthToken), mCallback(callback) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "RecordingStatusTask created");
}

void ttv::broadcast::RecordingStatusTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::string clientId;
  GetClientId(clientId);

  requestInfo.url = kRecordingStatusURL;
  requestInfo.url.replace(requestInfo.url.find(kClientIdTemplate), strlen(kClientIdTemplate), clientId);

  requestInfo.httpReqType = HTTP_GET_REQUEST;
  requestInfo.requestHeaders.emplace_back("Accept", "application/vnd.twitchtv.v4+json");
}

void ttv::broadcast::RecordingStatusTask::ProcessResponse(uint /*status*/, const std::vector<char>& response) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "RecordingStatusTask::ProcessResponse entered");

  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }
    if (jsonVal["recording"].empty()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Info, "No JSON 'recording' element found or it's empty");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_NO_RECORDING_STATUS;
      return;
    }

    mResult = std::make_shared<Result>();
    mResult->state.recordingEnabled = jsonVal["recording"].asBool();

    if (!mResult->state.recordingEnabled) {
      if (jsonVal["cure_url"].isString()) {
        mResult->state.cureUrl = jsonVal["cure_url"].asString();
      }
    }
  }
}

void ttv::broadcast::RecordingStatusTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
