/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/twitchapi.h"

#include "twitchsdk/broadcast/internal/streamer.h"
#include "twitchsdk/broadcast/internal/task/channelinfotask.h"
#include "twitchsdk/broadcast/internal/task/gamestreamstask.h"
#include "twitchsdk/broadcast/internal/task/ingestlisttask.h"
#include "twitchsdk/broadcast/internal/task/matchgamenamestask.h"
#include "twitchsdk/broadcast/internal/task/metadatatask.h"
#include "twitchsdk/broadcast/internal/task/recordingstatustask.h"
#include "twitchsdk/broadcast/internal/task/runcommercialtask.h"
#include "twitchsdk/broadcast/internal/task/setstreaminfotask.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/task/getstreamtask.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/task/validateoauthtask.h"
#include "twitchsdk/core/user/user.h"

namespace {
const char* kLogger = "TwitchAPI";
}

ttv::broadcast::TwitchAPI::TwitchAPI() : Component() {
  ttv::trace::Message(kLogger, MessageLevel::Info, "TwitchAPI created with no user");
}

ttv::broadcast::TwitchAPI::TwitchAPI(const std::shared_ptr<User>& user) : Component(), mUser(user) {
  ttv::trace::Message(kLogger, MessageLevel::Info, "TwitchAPI created for user");
}

std::string ttv::broadcast::TwitchAPI::GetLoggerName() const {
  return kLogger;
}

const char* ttv::broadcast::TwitchAPI::CurrentApiVersionString() {
  return "application/vnd.twitchtv.v1+json";
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::GetChannelInfo(GetChannelInfoCallback&& callback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  ChannelInfoTask::Callback taskCallback = [this, callback, cleanup](ChannelInfoTask* source, TTV_ErrorCode ec,
                                             const std::shared_ptr<ChannelInfoTask::Result>& result) {
    CompleteTask(source);

    ChannelInfo channelInfo;

    if (TTV_SUCCEEDED(ec)) {
      channelInfo.streamKey = result->streamKey;
      channelInfo.loginName = result->loginName;
      channelInfo.displayName = result->displayName;
      channelInfo.url = result->url;
    }

    cleanup(ec);

    callback(this, ec, std::move(channelInfo));
  };

  auto task = std::make_shared<ChannelInfoTask>(oauthToken->GetToken(), taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't get channel info");
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::SetStreamInfo(
  ChannelId channelId, const std::string& streamTitle, const std::string& gameName, SetStreamInfoCallback&& callback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  SetStreamInfoTask::Callback taskCallback = [this, callback, cleanup](SetStreamInfoTask* source, TTV_ErrorCode ec,
                                               const std::shared_ptr<SetStreamInfoTask::Result>& /*result*/) {
    CompleteTask(source);

    cleanup(ec);

    callback(this, ec);
  };

  auto task =
    std::make_shared<SetStreamInfoTask>(channelId, oauthToken->GetToken(), streamTitle, gameName, taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't set stream info");
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::GetStreamInfo(ChannelId channelId, GetStreamInfoCallback&& callback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  GetStreamTask::Callback taskCallback = [this, callback, cleanup](GetStreamTask* source, TTV_ErrorCode ec,
                                           const std::shared_ptr<GetStreamTask::Result>& result) {
    CompleteTask(source);

    std::shared_ptr<StreamInfo> streamInfo;

    if (TTV_SUCCEEDED(ec)) {
      streamInfo = std::move(result->streamInfo);
    }

    cleanup(ec);

    callback(this, ec, streamInfo);
  };

  auto task = std::make_shared<GetStreamTask>(channelId, oauthToken->GetToken(), taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't get stream info");
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::GetChannelRecordingStatus(GetChannelRecordingStatusCallback&& callback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  RecordingStatusTask::Callback taskCallback = [this, callback, cleanup](RecordingStatusTask* source, TTV_ErrorCode ec,
                                                 const std::shared_ptr<RecordingStatusTask::Result>& result) {
    CompleteTask(source);

    std::shared_ptr<ArchivingState> state;

    if (TTV_SUCCEEDED(ec)) {
      state = std::make_shared<ArchivingState>();
      *state = result->state;
    }

    cleanup(ec);

    callback(this, ec, state);
  };

  auto task = std::make_shared<RecordingStatusTask>(oauthToken->GetToken(), taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't archiving state");
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::RunCommercial(
  ChannelId channelId, uint32_t /*timeBreakSeconds*/, RunCommercialCallback&& callback) {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  auto oauthToken = user->GetOAuthToken();

  auto cleanup = [user, oauthToken](TTV_ErrorCode ec) {
    if (ec == TTV_EC_AUTHENTICATION) {
      user->ReportOAuthTokenInvalid(oauthToken, ec);
    }
  };

  RunCommercialTask::Callback taskCallback = [this, callback, cleanup](RunCommercialTask* source, TTV_ErrorCode ec,
                                               const std::shared_ptr<RunCommercialTask::Result>& /*result*/) {
    CompleteTask(source);

    cleanup(ec);

    callback(this, ec);
  };

  auto task = std::make_shared<RunCommercialTask>(channelId, oauthToken->GetToken(), taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't trigger commercial");
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::GetGameNameList(const std::string& str, GetGameNameListCallback&& callback) {
  std::string trimmed;
  Trim(trimmed);

  if (trimmed == "") {
    return TTV_EC_INVALID_ARG;
  }

  // Wait until the current task finishes
  if (mOutstandingMatchGameNamesTask != nullptr) {
    mNextGameNamesQuery = str;
    mNextGameNamesCallback = std::move(callback);

    return TTV_EC_SUCCESS;
  }

  MatchGameNamesTask::Callback taskCallback = [this, str, callback](MatchGameNamesTask* source, TTV_ErrorCode ec,
                                                const std::shared_ptr<MatchGameNamesTask::Result>& result) {
    CompleteTask(source);

    std::shared_ptr<GameInfoList> games;

    if (TTV_SUCCEEDED(ec)) {
      games = std::make_shared<GameInfoList>();
      *games = result->games;
    }

    callback(this, str, ec, games);

    // Kick off another request if the string has changed
    if (mNextGameNamesQuery != "") {
      if (mNextGameNamesQuery != str) {
        GetGameNameList(mNextGameNamesQuery, std::move(mNextGameNamesCallback));
      }

      mNextGameNamesQuery = "";
      mNextGameNamesCallback = nullptr;
    }
  };

  mOutstandingMatchGameNamesTask = std::make_shared<MatchGameNamesTask>(trimmed, taskCallback);

  TTV_ErrorCode ec = StartTask(mOutstandingMatchGameNamesTask);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't get game names");
    mOutstandingMatchGameNamesTask.reset();
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::TwitchAPI::GetIngestServerList(GetIngestServerListCallback&& callback) {
  IngestListTask::Callback taskCallback = [this, callback](IngestListTask* source, TTV_ErrorCode ec,
                                            const std::shared_ptr<IngestListTask::Result>& result) {
    CompleteTask(source);

    if (callback != nullptr) {
      if (TTV_SUCCEEDED(ec)) {
        TTV_ASSERT(result != nullptr);
        callback(this, ec, std::move(result->ingestList));
      } else {
        callback(this, ec, {});
      }
    }
  };

  ChannelId channelId = 0;
  auto user = mUser.lock();
  if (user != nullptr) {
    channelId = static_cast<ChannelId>(user->GetUserId());
  }

  auto task = std::make_shared<IngestListTask>(channelId, taskCallback);

  TTV_ErrorCode ec = StartTask(task);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(kLogger, MessageLevel::Error, "Failed to start task, can't fetch ingest server list");
  }

  return ec;
}
