/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/videoframequeue.h"

#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

namespace {
const int kKeyFrameFrequency = 2000;  // How often we need a keyframe in MS
}

ttv::broadcast::VideoFrameQueue::VideoFrameQueue()
    : mNumFramesDropped(0),
      mNumFramesProcessed(0),
      mFrameDuration(0),
      mTimeOfNextFrame(0),
      mTimeOfNextKeyFrame(0),
      mTimeOfLastSubmittedFrame(0),
      mAllowFrameDroppingAndReuse(true) {
  ttv::trace::Message("VideoFrameQueue", MessageLevel::Info, "VideoFrameQueue created");
  TTV_ErrorCode ret = CreateMutex(mMutex, "VideoFrameQueue");
  ASSERT_ON_ERROR(ret);
}

void ttv::broadcast::VideoFrameQueue::SetFPS(uint fps) {
  TTV_ASSERT(fps > 0);
  mFrameDuration = GetSystemClockFrequency() / fps;
}

TTV_ErrorCode ttv::broadcast::VideoFrameQueue::AddFrame(const std::shared_ptr<VideoFrame>& frame) {
  TTV_ASSERT(frame != nullptr);
  TTV_ASSERT(mFrameDuration > 0);

  AutoMutex lock(mMutex.get());

  // We want at most one frame in the queue
  if (mAllowFrameDroppingAndReuse) {
    if (!mQueuedVideoFrames.empty()) {
      auto droppedFrame = mQueuedVideoFrames.front();

      ttv::trace::Message("VideoFrameQueue", MessageLevel::Warning,
        "DROP! dropped frame time = %lld, new frame time = %lld (total dropped = %d)",
        SystemTimeToMs(droppedFrame->GetTimeStamp()), SystemTimeToMs(frame->GetTimeStamp()), ++mNumFramesDropped);

      mQueuedVideoFrames.clear();
    }
  }

  mQueuedVideoFrames.push_back(frame);

  return TTV_EC_SUCCESS;
}

std::shared_ptr<ttv::broadcast::VideoFrame> ttv::broadcast::VideoFrameQueue::GetNextFrame(uint64_t currentTime) {
  std::shared_ptr<VideoFrame> ret;

  if (mAllowFrameDroppingAndReuse) {
    bool canProcessNewFrame = (currentTime >= mTimeOfNextFrame);
    bool duplicateFrameIfNeeded = (currentTime >= mTimeOfNextFrame + mFrameDuration);
    bool needKeyFrame = (currentTime + mFrameDuration > mTimeOfNextKeyFrame);

    if (canProcessNewFrame) {
      bool freshFrame;

      // Scope for mutex
      {
        AutoMutex lock(mMutex.get());
        if (!mQueuedVideoFrames.empty()) {
          ret = mQueuedVideoFrames.front();
          mQueuedVideoFrames.pop_front();
          freshFrame = true;

          mLastProcessedFrame = ret;
        } else {
          ret = mLastProcessedFrame;
          freshFrame = false;
        }
      }

      // It is possible we do not have a frame here,
      // that would mean that no frame was ever submitted
      // to us
      if (ret != nullptr) {
        if (!freshFrame) {
          if (duplicateFrameIfNeeded) {
            ret->SetTimeStamp(currentTime);
          } else {
            ttv::trace::Message(
              "VideoFrameQueue", MessageLevel::Info, "No fresh frame! CT %llu NFT %llu", currentTime, mTimeOfNextFrame);
            ret.reset();
          }
        }

        if (ret != nullptr) {
          ttv::trace::Message("VideoFrameQueue", MessageLevel::Info, "%s frame   [PR %8u FT %16llu CT %16llu",
            freshFrame ? "New" : "Old", ++mNumFramesProcessed, SystemTimeToMs(ret->GetTimeStamp()),
            SystemTimeToMs(currentTime));
        }
      }

      if (ret != nullptr) {
        if (ret->GetTimeStamp() <= (mTimeOfLastSubmittedFrame + MsToSystemTime(2))) {
          ttv::trace::Message("VideoFrameQueue", MessageLevel::Warning,
            "DROPPED! ret->GetTimeStamp() %llu mTimeOfLastSubmittedFrame %llu \n", ret->GetTimeStamp(),
            mTimeOfLastSubmittedFrame);
          ret.reset();
        } else {
          mTimeOfLastSubmittedFrame = ret->GetTimeStamp();
          ret->SetIsKeyFrame(needKeyFrame);

          if (needKeyFrame) {
            ttv::trace::Message(
              "VideoFrameQueue", MessageLevel::Info, "Keyframe at time %lld", SystemTimeToMs(ret->GetTimeStamp()));
            mTimeOfNextKeyFrame = currentTime + MsToSystemTime(kKeyFrameFrequency);
          }
        }

        // Determine when we expect the next frame to be submitted
        if (mTimeOfNextFrame + (mFrameDuration * 4) < currentTime) {
          mTimeOfNextFrame = currentTime;
        }
        mTimeOfNextFrame += mFrameDuration;
      }
    }
  } else {
    // Try and get a new frame
    {
      AutoMutex lock(mMutex.get());
      if (!mQueuedVideoFrames.empty()) {
        ret = mQueuedVideoFrames.front();
        mQueuedVideoFrames.pop_front();
      }
    }
  }

  return ret;
}

void ttv::broadcast::VideoFrameQueue::Shutdown() {
  AutoMutex lock(mMutex.get());
  mQueuedVideoFrames.clear();
  mLastProcessedFrame.reset();
}
