/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/internal/videostreamer.h"

#include "twitchsdk/broadcast/iframewriter.h"
#include "twitchsdk/broadcast/internal/videoframequeue.h"
#include "twitchsdk/broadcast/ipreencodedvideoframereceiver.h"
#include "twitchsdk/broadcast/ivideocapture.h"
#include "twitchsdk/broadcast/ivideoencoder.h"
#include "twitchsdk/broadcast/videoframe.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

namespace {
const uint32_t kVideoStreamIndex = 0;
}

ttv::broadcast::VideoStreamer::VideoStreamer() : mInitialTime(0), mDoProcessing(false), mFrameQueue(nullptr) {
  ttv::trace::Message("VideoStreamer", MessageLevel::Info, "VideoStreamer created");
}

ttv::broadcast::VideoStreamer::~VideoStreamer() {
  TTV_ASSERT(mFrameQueueThread == nullptr);
  TTV_ASSERT(!mDoProcessing);
  TTV_ASSERT(mFrameQueue == nullptr);

  ttv::trace::Message("VideoStreamer", MessageLevel::Info, "VideoStreamer destroyed");
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::SetEncoder(const std::shared_ptr<IVideoEncoder>& encoder) {
  // TODO: Make sure not streaming
  mVideoEncoder = encoder;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::SetCapturer(const std::shared_ptr<IVideoCapture>& capturer) {
  // TODO: Make sure not streaming
  mVideoCapturer = capturer;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::ValidateVideoParams(const VideoParams& videoParams) const {
  if (videoParams.targetFramesPerSecond > kMaxFramesPerSecond ||
      videoParams.targetFramesPerSecond < kMinFramesPerSecond) {
    return TTV_EC_BROADCAST_INVALID_FPS;
  }

  if (videoParams.initialKbps > kMaxBitRate || videoParams.initialKbps < kMinBitRate) {
    return TTV_EC_BROADCAST_INVALID_BITRATE;
  }

  if (videoParams.outputHeight > kMaxFrameHeight || videoParams.outputWidth > kMaxFrameWidth) {
    return TTV_EC_BROADCAST_INVALID_RESOLUTION;
  }

  if (mVideoEncoder == nullptr) {
    return TTV_EC_BROADCAST_INVALID_ENCODER;
  }

  return mVideoEncoder->ValidateVideoParams(videoParams);
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::Initialize(const VideoParams& videoParams) {
  AutoTracer tracer("VideoStreamer", MessageLevel::Debug, "VideoStreamer::Initialize()");

  if (mVideoEncoder == nullptr) {
    ttv::trace::Message("VideoStreamer", MessageLevel::Debug, "VideoStreamer::Initialize() - No video encoder set");
    return TTV_EC_BROADCAST_INVALID_ENCODER;
  } else if (mVideoCapturer == nullptr) {
    ttv::trace::Message("VideoStreamer", MessageLevel::Debug, "VideoStreamer::Initialize() - No video capturer set");
    return TTV_EC_BROADCAST_INVALID_VIDEO_CAPTURER;
  }

  TTV_ErrorCode ec = ValidateVideoParams(videoParams);
  if (TTV_FAILED(ec)) {
    ttv::trace::Message(
      "VideoStreamer", MessageLevel::Debug, "VideoStreamer::Initialize() - Video parameter validation failed");
    return ec;
  }

  mFrameQueue = std::make_unique<VideoFrameQueue>();

  // NOTE: This is a bit of a hack to prevent corruption of pre-encoded data
  // This really should be queried from the capturer/encoder itself
  if (mVideoEncoder->SupportsReceiverProtocol(IPreEncodedVideoFrameReceiver::GetReceiverTypeId())) {
    mFrameQueue->SetAllowFrameDroppingAndReuse(false);
  }

  mVideoParams = videoParams;

  // Set the target FPS for the frame limiter
  mFrameQueue->SetFPS(videoParams.targetFramesPerSecond);

  ec = mVideoEncoder->Start(kVideoStreamIndex, videoParams);

  if (TTV_FAILED(ec)) {
    ttv::trace::Message("VideoStreamer", MessageLevel::Error,
      "VideoStreamer::StartCapture() - Video encoder initialization failed: %s", ErrorToString(ec));
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::StartCapture() {
  AutoTracer tracer("VideoStreamer", MessageLevel::Debug, "VideoStreamer::StartCapture()");

  if (mVideoCapturer == nullptr) {
    ttv::trace::Message("VideoStreamer", MessageLevel::Debug, "VideoStreamer::StartCapture() - No video capturer set");
    return TTV_EC_BROADCAST_INVALID_VIDEO_CAPTURER;
  }

  TTV_ErrorCode ec = mVideoCapturer->SetVideoEncoder(mVideoEncoder);
  ASSERT_ON_ERROR(ec);

  ec = mVideoCapturer->Start(mVideoParams);
  ASSERT_ON_ERROR(ec);

  return ec;
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::Stop() {
  AutoTracer tracer("VideoStreamer", MessageLevel::Debug, "VideoStreamer::Stop()");

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (mFrameQueueThread != nullptr) {
    mDoProcessing = false;

    if (mFrameQueueThread->Joinable()) {
      mFrameQueueThread->Join();
    }

    mFrameQueueThread.reset();
  }

  if (mFrameQueue != nullptr) {
    mFrameQueue->Shutdown();
  }

  if (mVideoEncoder != nullptr) {
    ec = mVideoEncoder->Stop();
    TTV_ASSERT(TTV_SUCCEEDED(ec));

    mVideoEncoder->SetFrameWriter(nullptr);
  }

  mFrameQueue.reset();

  mInitialTime = 0;

  return ec;
}

TTV_ErrorCode ttv::broadcast::VideoStreamer::ProcessFrame(const std::shared_ptr<VideoFrame>& videoFrame) {
  // ttv::trace::Message("VideoStreamer", MessageLevel::Debug, "VideoStreamer::ProcessFrame()");

  TTV_ASSERT(videoFrame != nullptr);
  TTV_ASSERT(mInitialTime != 0);

  TTV_RETURN_ON_NULL(videoFrame, TTV_EC_BROADCAST_INVALID_VIDEOFRAME);

  TTV_ASSERT(mVideoEncoder);
  TTV_ErrorCode ret = mVideoEncoder->ValidateFrame(videoFrame);

  if (TTV_SUCCEEDED(ret)) {
    ret = mFrameQueue->AddFrame(videoFrame);
    if (TTV_FAILED(ret)) {
      ttv::trace::Message("VideoStreamer", MessageLevel::Info, "Frame queue full");
      ret = TTV_EC_BROADCAST_FRAME_QUEUE_FULL;
    }
  }

  return ret;
}

void ttv::broadcast::VideoStreamer::ProcessFrameQueue() {
  ttv::trace::Message("VideoStreamer", MessageLevel::Debug, "VideoStreamer::ProcessFrameQueue()");

  uint64_t start = GetSystemClockTime();
  uint numFrames = 0;

  while (mDoProcessing) {
    const uint64_t pts = GetSystemClockTime() - mInitialTime;

    std::shared_ptr<VideoFrame> frame = mFrameQueue->GetNextFrame(pts);
    if (frame != nullptr) {
      // TODO: Video Encoder errors are lost - This is very bad!
      const TTV_ErrorCode ret = mVideoEncoder->SubmitFrame(frame);
      if (TTV_FAILED(ret)) {
        /*
         * Note: No assert on error here, because submission of frames might fail due to
         * compression session getting invalidated when camera is interrupted.
         */
        ttv::trace::Message(
          "VideoStreamer", MessageLevel::Info, "Error while submitting frame to video encoder %s", ErrorToString(ret));
      }

      if ((numFrames++ % 60) == 0) {
        uint64_t now = GetSystemClockTime();
        float durationInSeconds = static_cast<float>(SystemTimeToMs(now - start)) / 1000.f;
        float fps = static_cast<float>(numFrames) / durationInSeconds;
        ttv::trace::Message("VideoStreamer", MessageLevel::Info, "Processed %u frames in %.2f secs = %f fps \n",
          numFrames, static_cast<double>(durationInSeconds), static_cast<double>(fps));
      }
    } else {
      // TODO: This needs some review.  If the app is submitting too many frames then we might be encoding and muxing
      // more than the fps
      ttv::Sleep(1);
    }
  }
}

void ttv::broadcast::VideoStreamer::SetInitialTime(uint64_t initialTime) {
  mDoProcessing = true;
  TTV_ASSERT(nullptr == mFrameQueueThread);

  TTV_ErrorCode ec = ttv::CreateThread(
    std::bind(&VideoStreamer::ProcessFrameQueue, this), "ttv::broadcast::VideoStreamer::FrameQueue", mFrameQueueThread);
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mFrameQueueThread != nullptr);

  mFrameQueueThread->Run();

  mInitialTime = initialTime;
}

std::string ttv::broadcast::VideoStreamer::GetEncoderName() const {
  return mVideoEncoder ? mVideoEncoder->GetName() : "";
}
