/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "amf0.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/types/coretypes.h"

#include <vector>

namespace ttv {
class ISocket;

namespace test {
class TestSocket;
}

namespace broadcast {
namespace test {
namespace rtmp {
enum class RtmpState {
  Uninitialized,
  VersionSent,
  WaitingForC1,
  WaitingForC2,
  HandshakeDone,
  WaitingForConnect,
  WaitingForCreateStream,
  WaitingForPublish,
  Broadcasting
};

enum class ChunkMessageHeaderType : uint8_t {
  Large = 0,   // 11 bytes long
  Medium = 1,  // 7 bytes long
  Small = 2,   // 3 bytes long
  Empty = 3    // No message header
};

enum class Channel : uint8_t { Invalid = 0, Network = 2, System, Source, NetStream, Video = 8, Audio };

enum class MessageTypeId : uint8_t {
  Invalid = 0x00,  ///< Not a real value according to the standard
  ChunkSize = 0x01,
  AbortMsg = 0x02,
  BytesRead = 0x03,
  Control = 0x04,  ///< See RTMP_CTL_* below for possible values
  WinAckSize = 0x05,
  PeerBw = 0x06,

  EdgeOrigin = 0x07,  ///< No payloads actually documented...

  Audio = 0x08,
  Video = 0x09,

  Amf3Data = 0x0F,                 ///< amf3 metadata body
  Amf3SharedObjectCommand = 0x10,  ///< amf3 shared object command (don't support)
  Amf3 = 0x11,                     ///< cmd-name, trans-id, amf3 body

  Amf0Data = 0x12,                 ///< amf0 metadata body
  Amf0SharedObjectCommand = 0x13,  ///< amf0 shared object command (don't support)
  Amf0 = 0x14,                     ///< cmd-name, trans-id, amf0 body

  Aggregate = 0x16  ///< needs to be chopped up
};

enum class ControlType : uint8_t {
  StreamBegin = 0,  ///< Data: stream-id (4B)
  StreamEof,        ///< Data: stream-id (4B)
  StreamDry,        ///< Data: stream-id (4B)
  SetBufferLength,  ///< Data: stream-id, buffer-length in milliseconds (*** 8B ***)
  StreamRecorded,   ///< Server telling client it's not live, Data: stream-id (4B)
  Ping = 6,         ///< Data: timestamp (4B)
  Pong              ///< Responds with the ping's timestamp, so RTT can be calculated (4B)
};

struct BasicHeader;
struct ChunkHeader;
struct TagHeader;
struct TagFooter;

class ChunkParser;
class ChunkWriter;
class RtmpServer;

struct ChunkBasicHeader {
  ChunkBasicHeader() : chunkStreamId(0), chunkMessageHeaderType(ChunkMessageHeaderType::Large) {}

  uint32_t chunkStreamId;
  ChunkMessageHeaderType chunkMessageHeaderType;
};

struct MessageHeader {
  MessageHeader() : timestamp(0), messageLength(0), messageTypeId(MessageTypeId::Invalid), messageStreamId(0) {}

  uint32_t timestamp;
  uint32_t messageLength;
  MessageTypeId messageTypeId;
  uint32_t messageStreamId;
};

struct ChunkHeader {
  ChunkHeader() : extendedTimestamp(0) {}
  ChunkBasicHeader basicHeader;
  MessageHeader messageHeader;
  uint32_t extendedTimestamp;
};

struct ConnectCommand {
  ConnectCommand();
  ~ConnectCommand();

  uint32_t transactionId;
  std::shared_ptr<amf0::Amf0Object> commandArguments;
  std::shared_ptr<amf0::Amf0Object> userArguments;
};

struct CreateStreamCommand {
  CreateStreamCommand();
  ~CreateStreamCommand();

  uint32_t transactionId;
  std::shared_ptr<amf0::Amf0Object> commandArguments;
};

class IRtmpListener;
class RtmpListenerProxy;
}  // namespace rtmp
}  // namespace test
}  // namespace broadcast
}  // namespace ttv

class ttv::broadcast::test::rtmp::IRtmpListener {
 public:
  virtual void OnStateChanged(RtmpState state) = 0;
  virtual void OnConnectCommandReceived(const std::shared_ptr<ConnectCommand>& cmd) = 0;
  virtual void OnCreateStreamCommandReceived(const std::shared_ptr<CreateStreamCommand>& cmd) = 0;
};

/**
 * A lambda proxy for IRtmpListener.
 */
class ttv::broadcast::test::rtmp::RtmpListenerProxy : public IRtmpListener {
 public:
  using OnStateChangedFunc = std::function<void(RtmpState state)>;
  using OnConnectCommandReceivedFunc = std::function<void(const std::shared_ptr<ConnectCommand>& cmd)>;
  using OnCreateStreamCommandReceivedFunc = std::function<void(const std::shared_ptr<CreateStreamCommand>& cmd)>;

 public:
  // IRtmpListener implementation
  virtual void OnStateChanged(RtmpState state) override {
    if (mOnStateChangedFunc != nullptr) {
      mOnStateChangedFunc(state);
    }
  }

  virtual void OnConnectCommandReceived(const std::shared_ptr<ConnectCommand>& cmd) override {
    if (mOnConnectCommandReceivedFunc != nullptr) {
      mOnConnectCommandReceivedFunc(cmd);
    }
  }

  virtual void OnCreateStreamCommandReceived(const std::shared_ptr<CreateStreamCommand>& cmd) override {
    if (mOnConnectCommandReceivedFunc != nullptr) {
      mOnCreateStreamCommandReceived(cmd);
    }
  }

 public:
  OnStateChangedFunc mOnStateChangedFunc;
  OnConnectCommandReceivedFunc mOnConnectCommandReceivedFunc;
  OnCreateStreamCommandReceivedFunc mOnCreateStreamCommandReceived;
};

class ttv::broadcast::test::rtmp::ChunkParser {
 public:
  ChunkParser();

  void AddListener(const std::shared_ptr<IRtmpListener>& listener);
  void RemoveListener(const std::shared_ptr<IRtmpListener>& listener);

  /**
   * Reads from the buffer and populates the chunk.
   * Returns true when the parsing has completed and the chunk is ready.
   */
  bool Parse(std::vector<uint8_t>& buffer);
  void Reset();

 private:
  enum class State { ReadingChunkMessageHeader, WaitingForChunkBody };

  bool ParseChunkHeader(std::vector<uint8_t>& buffer);
  bool ParseChunkBody(std::vector<uint8_t>& buffer);
  bool ParseChunkBody_Amf0(std::vector<uint8_t>& buffer);

  EventSource<IRtmpListener> mListeners;
  ChunkHeader mChunkHeader;
  State mState;
};

class ttv::broadcast::test::rtmp::ChunkWriter {
 public:
  ChunkWriter();

  void SetChunkStreamId(uint32_t streamId);
  void SetTimestamp(uint32_t timestamp);
  void SetMessageTypeId(MessageTypeId type);
  void SetMessageStreamId(uint32_t streamId);

  void WriteBody(const amf0::Amf0Value& value);
  void WriteBody(const std::vector<uint8_t>& bytes);

  void GetChunk(std::vector<uint8_t>& buffer);
  void Reset();

 private:
  ChunkHeader mChunkHeader;
  std::vector<uint8_t> mBodyBuffer;  //!< The chunk body buffer
};

class ttv::broadcast::test::rtmp::RtmpServer {
 public:
  RtmpServer();
  ~RtmpServer();

  void AddListener(const std::shared_ptr<IRtmpListener>& listener);
  void RemoveListener(const std::shared_ptr<IRtmpListener>& listener);

  RtmpState GetState() const { return mState; }

  /**
   * Sets an already connected socket.
   */
  void SetSocket(const std::shared_ptr<ttv::test::TestSocket>& socket);
  void SetBandwidthTest(bool bandwidthTest) { mBandwidthTest = bandwidthTest; }

  void Initialize();
  void Update();
  void Shutdown();

 private:
  void SetState(RtmpState state);

  ChunkParser mChunkParser;
  std::shared_ptr<RtmpListenerProxy> mChunkListener;
  EventSource<IRtmpListener> mListeners;
  std::shared_ptr<ttv::test::TestSocket> mSocket;
  std::vector<uint8_t> mReceiveBuffer;  // The received bytes which have not yet been processed
  RtmpState mState;
  bool mBandwidthTest;
};
