/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "amf0.h"

#include "buffer.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::broadcast::test;

// https://en.wikipedia.org/wiki/Action_Message_Format#AMF0

ttv::broadcast::test::amf0::Amf0Value::Amf0Value() : type(Amf0Type::Null), number(0), boolean(0) {}

ttv::broadcast::test::amf0::Amf0Value::Amf0Value(bool b) : type(Amf0Type::Boolean), number(0), boolean(b) {}

ttv::broadcast::test::amf0::Amf0Value::Amf0Value(double n) : type(Amf0Type::Number), number(n), boolean(0) {}

ttv::broadcast::test::amf0::Amf0Value::Amf0Value(uint32_t n)
    : type(Amf0Type::Number), number(static_cast<double>(n)), boolean(0) {}

ttv::broadcast::test::amf0::Amf0Value::Amf0Value(const char* str)
    : type(Amf0Type::String), str(str), number(0), boolean(0) {}

ttv::broadcast::test::amf0::Amf0Value::Amf0Value(const std::string& str)
    : type(Amf0Type::String), str(str), number(0), boolean(0) {}

ttv::broadcast::test::amf0::Amf0Value::Amf0Value(const std::shared_ptr<Amf0Object>& obj)
    : type(Amf0Type::Object), object(obj), number(0), boolean(0) {}

ttv::broadcast::test::amf0::Amf0Value::~Amf0Value() {}

bool ttv::broadcast::test::amf0::ParseAmf0String(std::vector<uint8_t>& buffer, Amf0Value& result) {
  result.type = Amf0Type::String;
  size_t length = ReadUInt16(buffer);
  result.str = ReadString(buffer, length);

  return true;
}

bool ttv::broadcast::test::amf0::ParseAmf0Value(std::vector<uint8_t>& buffer, Amf0Value& result) {
  // https://en.wikipedia.org/wiki/Action_Message_Format#AMF0
  result.type = ReadByte<Amf0Type>(buffer);

  switch (result.type) {
    case Amf0Type::String: {
      ParseAmf0String(buffer, result);

      break;
    }
    case Amf0Type::Boolean: {
      result.boolean = ReadByte<char>(buffer) != 0;

      break;
    }
    case Amf0Type::Number: {
      result.number = ReadDouble(buffer);

      break;
    }
    case Amf0Type::Object: {
      result.object = std::make_shared<Amf0Object>();
      ParseAmf0Object(buffer, *result.object);

      break;
    }
    case Amf0Type::ObjectEnd: {
      // Handled at a higher level
      EXPECT_FALSE(true);
      break;
    }
    default: {
      // TODO: Implement as needed
      EXPECT_FALSE(true);
      break;
    }
  }

  return true;
}

bool ttv::broadcast::test::amf0::ParseAmf0Object(std::vector<uint8_t>& buffer, Amf0Object& result) {
  for (;;) {
    // Peek at the type to see if the end of the pairs
    if (ReadByte<char>(buffer, 0, false) == 0 && ReadByte<char>(buffer, 1, false) == 0) {
      ReadByte<char>(buffer, 0);
      ReadByte<char>(buffer, 0);
      Amf0Type type = ReadByte<Amf0Type>(buffer);
      EXPECT_EQ(type, Amf0Type::ObjectEnd);
      break;
    }

    std::shared_ptr<Amf0Value> key = std::make_shared<Amf0Value>();
    std::shared_ptr<Amf0Value> value = std::make_shared<Amf0Value>();

    ParseAmf0String(buffer, *key);
    ParseAmf0Value(buffer, *value);

    result.properties.emplace_back(key->str, value);
  }

  return true;
}

void ttv::broadcast::test::amf0::WriteAmf0Object(std::vector<uint8_t>& buffer, const Amf0Object& value) {
  for (const auto& prop : value.properties) {
    WriteString(buffer, prop.first);
    WriteByte(buffer, prop.second->type);
    WriteAmf0Value(buffer, *prop.second);
  }
  WriteByte(buffer, 0);
  WriteByte(buffer, 0);
}

void ttv::broadcast::test::amf0::WriteAmf0String(std::vector<uint8_t>& buffer, const std::string& value) {
  WriteUInt16(buffer, static_cast<uint16_t>(value.size()));
  WriteString(buffer, value);
}

void ttv::broadcast::test::amf0::WriteAmf0String(std::vector<uint8_t>& buffer, const Amf0Value& value) {
  WriteAmf0String(buffer, value.str);
}

void ttv::broadcast::test::amf0::WriteAmf0Number(std::vector<uint8_t>& buffer, const Amf0Value& value) {
  WriteDouble(buffer, value.number);
}

void ttv::broadcast::test::amf0::WriteAmf0Boolean(std::vector<uint8_t>& buffer, const Amf0Value& value) {
  WriteByte(buffer, value.boolean ? 1 : 0);
}

void ttv::broadcast::test::amf0::WriteAmf0Value(std::vector<uint8_t>& buffer, const Amf0Value& value, bool writeType) {
  if (writeType) {
    WriteByte(buffer, value.type);
  }

  switch (value.type) {
    case Amf0Type::String: {
      WriteAmf0String(buffer, value);
      break;
    }
    case Amf0Type::Boolean: {
      WriteAmf0Boolean(buffer, value);
      break;
    }
    case Amf0Type::Number: {
      WriteAmf0Number(buffer, value);
      break;
    }
    case Amf0Type::Object: {
      WriteAmf0Object(buffer, *value.object);

      if (writeType) {
        WriteByte(buffer, Amf0Type::ObjectEnd);
      }

      break;
    }
    case Amf0Type::Null: {
      // Null has no other data
      break;
    }
    case Amf0Type::ObjectEnd: {
      // Handled at a higher level
      EXPECT_FALSE(true);
      break;
    }
    default: {
      // TODO: Implement as needed
      EXPECT_FALSE(true);
      break;
    }
  }
}
