/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "broadcasttestutility.h"
#include "fixtures/broadcastbasetest.h"
#include "pubsubtestutility.h"
#include "testutilities.h"
#include "twitchsdk/broadcast/broadcastapi.h"
#include "twitchsdk/broadcast/broadcastlistener.h"
#include "twitchsdk/broadcast/internal/ingesttester.h"
#include "twitchsdk/broadcast/passthroughvideoencoder.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"
#include "usertestutility.h"

#include <fstream>

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::test;
using namespace ttv::broadcast;
using namespace ttv::broadcast::test;

namespace {
class BroadcastApiListener : public ttv::broadcast::IBroadcastAPIListener {
 public:
  virtual void ModuleStateChanged(IModule* /*source*/, IModule::State /*state*/, TTV_ErrorCode /*ec*/) override {}

  virtual void BroadcastStateChanged(TTV_ErrorCode /*ec*/, BroadcastState /*state*/) override {}

  virtual void BroadcastBandwidthWarning(TTV_ErrorCode /*ec*/, uint32_t /*backupMilliseconds*/) override {}

  virtual void BroadcastFrameSubmissionIssue(TTV_ErrorCode /*ec*/) override {}

  virtual void StreamInfoFetched(TTV_ErrorCode /*ec*/, const StreamInfo& /*streamInfo*/) override {}

  virtual void StreamKeyError(const CanTheyError& /*canTheyError*/) override {}
};

class IngestTesterListener : public ttv::broadcast::IIngestTesterListener {
 public:
  virtual void BroadcastIngestTesterStateChanged(IIngestTester* /*source*/) override {}
};

class BroadcastApiTest : public BroadcastBaseTest {
 public:
  struct TestParams {
    std::string oauth;
    UserInfo userInfo;
  };

 public:
  BroadcastApiTest()
      : mCoreApi(std::make_shared<CoreAPI>()),
        mBroadcastAPI(std::make_shared<BroadcastAPI>()),
        mBroadcastApiListener(std::make_shared<BroadcastApiListener>()),
        mIngestTesterListener(std::make_shared<IngestTesterListener>()) {}

  void SetUpComponents() {
    BroadcastBaseTest::SetUpComponents();

    ttv::test::InitializeModule(mCoreApi);

    mBroadcastAPI->SetCoreApi(mCoreApi);
    mBroadcastAPI->SetListener(mBroadcastApiListener);

    AddModule(mCoreApi);
    AddModule(mBroadcastAPI);
  }

  TTV_ErrorCode LogIn(const TestParams& testParams) {
    return ttv::test::SdkBaseTest::LogIn(mCoreApi, testParams.oauth, testParams.userInfo);
  }

  void TestBroadcastApiInitializationCallback() {
    bool receivedCallback = false;
    TTV_ErrorCode ec = mBroadcastAPI->Initialize([&receivedCallback](TTV_ErrorCode ec) {
      EXPECT_EQ(TTV_EC_SUCCESS, ec);
      receivedCallback = true;
    });
    EXPECT_EQ(TTV_EC_SUCCESS, ec);

    std::function<bool()> chatApiResults = [this, &receivedCallback]() {
      return receivedCallback && (mBroadcastAPI->GetState() == BroadcastAPI::State::Initialized);
    };

    WaitUntilResult(1000, chatApiResults);
  }

  void TestBroadcastApiShutdownCallback() {
    std::function<bool()> chatApiResults = [this]() {
      return mBroadcastAPI->GetState() == BroadcastAPI::State::Uninitialized;
    };

    WaitUntilResult(3000, chatApiResults);
  }

 protected:
  std::shared_ptr<ttv::CoreAPI> mCoreApi;
  std::shared_ptr<ttv::broadcast::BroadcastAPI> mBroadcastAPI;
  std::shared_ptr<BroadcastApiListener> mBroadcastApiListener;
  std::shared_ptr<IngestTesterListener> mIngestTesterListener;
};
}  // namespace

TEST_F(BroadcastApiTest, CreateIngestTester) {
  InitializeModule(mBroadcastAPI);

  TestParams params;
  params.oauth = "valid_oauth";
  params.userInfo.userId = 9001;
  params.userInfo.userName = "valid_login";
  params.userInfo.displayName = "Display Name";
  LogIn(params);

  std::ifstream sampleDataFile("broadcast/TestVideoData.data", std::ios_base::binary);
  ASSERT_TRUE(sampleDataFile.is_open());

  std::vector<uint8_t> rawData;
  rawData.reserve(100000);
  rawData.assign((std::istreambuf_iterator<char>{sampleDataFile}), (std::istreambuf_iterator<char>{}));

  // Create an ingest tester instance
  std::shared_ptr<IIngestTester> ingestTester;
  TTV_ErrorCode ec = mBroadcastAPI->CreateIngestTester(
    params.userInfo.userId, mIngestTesterListener, rawData.data(), static_cast<uint32_t>(rawData.size()), ingestTester);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  // Release the reference
  ingestTester.reset();

  TestBroadcastApiShutdownCallback();
}
