/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/tasktest.h"
#include "testutilities.h"
#include "twitchsdk/broadcast/internal/task/channelinfotask.h"
#include "twitchsdk/broadcast/internal/task/ingestlisttask.h"

#include <algorithm>
#include <iostream>
#include <numeric>

using namespace ttv;
using namespace ttv::test;
using namespace ttv::broadcast;

TEST_F(TaskTest, BroadcastIngestListTask_Success) {
  auto response = mHttpRequest->AddResponse("https://ingest.twitch.tv/api/v2/ingests")
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v5+json")
                    .SetResponseBodyFromFile("broadcast/ingestlisttask_200.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<IngestListTask> task;

  IngestListTask::Callback callback = [this, &callbackCalled, &task](IngestListTask* source, TTV_ErrorCode ec,
                                        std::shared_ptr<IngestListTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    // Sanity check a few values
    ASSERT_EQ(result->ingestList.size(), 40);

    ASSERT_EQ(result->ingestList[0].serverName, "US West: San Francisco, CA");
    ASSERT_EQ(result->ingestList[0].priority, 0);
    ASSERT_EQ(result->ingestList[0].serverUrl, "rtmp://live.twitch.tv/app/{stream_key}");
    ASSERT_EQ(result->ingestList[0].serverId, 0);

    ASSERT_EQ(result->ingestList[25].serverName, "EU: Prague, CZ");
    ASSERT_EQ(result->ingestList[25].priority, 25);
    ASSERT_EQ(result->ingestList[25].serverUrl, "rtmp://live-prg.twitch.tv/app/{stream_key}");
    ASSERT_EQ(result->ingestList[25].serverId, 25);
  };

  auto checkFunction = [this, &callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<IngestListTask>(1234, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, BroadcastChannelInfoTask_Success) {
  const char* kOAuthToken = "test_oauth_token";

  auto response = mHttpRequest->AddResponse("https://api.twitch.tv/kraken/channel")
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v5+json")
                    .AddRequestHeader("Authorization", std::string("OAuth ") + kOAuthToken)
                    .SetResponseBodyFromFile("broadcast/channelinfotask_200.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<ChannelInfoTask> task;

  ChannelInfoTask::Callback callback = [this, &callbackCalled, &task](ChannelInfoTask* source, TTV_ErrorCode ec,
                                         std::shared_ptr<ChannelInfoTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(result->streamKey, "my_private_stream_key");
    ASSERT_EQ(result->displayName, "twitch");
    ASSERT_EQ(result->loginName, "twitch");
    ASSERT_EQ(result->url, "https://www.twitch.tv/twitch");
  };

  auto checkFunction = [this, &callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<ChannelInfoTask>(kOAuthToken, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}
