/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "buffer.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::broadcast::test;

uint32_t ttv::broadcast::test::EndianSwap(uint32_t source) {
  uint32_t target = 0;

  auto pSource = reinterpret_cast<uint8_t*>(&source);
  auto pTarget = reinterpret_cast<uint8_t*>(&target);
  pTarget[0] = pSource[3];
  pTarget[1] = pSource[2];
  pTarget[2] = pSource[1];
  pTarget[3] = pSource[0];

  return target;
}

uint32_t ttv::broadcast::test::EndianSwap(uint24_t source) {
  uint32_t target = 0;

  auto pTarget = reinterpret_cast<uint8_t*>(&target);
  pTarget[0] = source[2];
  pTarget[1] = source[1];
  pTarget[2] = source[0];

  return target;
}

uint32_t ttv::broadcast::test::ReadUInt16(std::vector<uint8_t>& buffer, size_t offset, bool consume) {
  uint32_t result = (buffer[offset + 0] << 8) | (buffer[offset + 1] << 0);

  if (consume) {
    buffer.erase(buffer.begin(), buffer.begin() + 2);
  }

  return result;
}

uint32_t ttv::broadcast::test::ReadUInt24(std::vector<uint8_t>& buffer, size_t offset, bool consume) {
  uint32_t result = (buffer[offset + 0] << 16) | (buffer[offset + 1] << 8) | (buffer[offset + 2] << 0);

  if (consume) {
    buffer.erase(buffer.begin(), buffer.begin() + 3);
  }

  return result;
}

uint32_t ttv::broadcast::test::ReadUInt32(std::vector<uint8_t>& buffer, size_t offset, bool consume) {
  uint32_t result =
    (buffer[offset + 0] << 24) | (buffer[offset + 1] << 16) | (buffer[offset + 2] << 8) | (buffer[offset + 3] << 0);

  if (consume) {
    buffer.erase(buffer.begin(), buffer.begin() + 4);
  }

  return result;
}

double ttv::broadcast::test::ReadDouble(std::vector<uint8_t>& buffer, size_t offset, bool consume) {
  uint64_t result =
    (static_cast<uint64_t>(buffer[offset + 0]) << 56) | (static_cast<uint64_t>(buffer[offset + 1]) << 48) |
    (static_cast<uint64_t>(buffer[offset + 2]) << 40) | (static_cast<uint64_t>(buffer[offset + 3]) << 32) |
    (static_cast<uint64_t>(buffer[offset + 4]) << 24) | (static_cast<uint64_t>(buffer[offset + 5]) << 16) |
    (static_cast<uint64_t>(buffer[offset + 6]) << 8) | (static_cast<uint64_t>(buffer[offset + 7]) << 0);

  if (consume) {
    buffer.erase(buffer.begin(), buffer.begin() + 8);
  }

  uint8_t* p = reinterpret_cast<uint8_t*>(&result);
  return *reinterpret_cast<double*>(p);
}

std::string ttv::broadcast::test::ReadString(std::vector<uint8_t>& buffer, size_t length, size_t offset, bool consume) {
  const char* p = reinterpret_cast<const char*>(buffer.data() + offset);
  std::string result(p, length);

  if (consume) {
    buffer.erase(buffer.begin(), buffer.begin() + length);
  }

  return result;
}

void ttv::broadcast::test::WriteUInt16(std::vector<uint8_t>& buffer, uint16_t value) {
  const uint8_t* p = reinterpret_cast<const uint8_t*>(&value);

  buffer.push_back(p[1]);
  buffer.push_back(p[0]);
}

void ttv::broadcast::test::WriteUInt24(std::vector<uint8_t>& buffer, uint32_t value) {
  const uint8_t* p = reinterpret_cast<const uint8_t*>(&value);

  buffer.push_back(p[2]);
  buffer.push_back(p[1]);
  buffer.push_back(p[0]);
}

void ttv::broadcast::test::WriteUInt32(std::vector<uint8_t>& buffer, uint32_t value) {
  const uint8_t* p = reinterpret_cast<const uint8_t*>(&value);

  buffer.push_back(p[3]);
  buffer.push_back(p[2]);
  buffer.push_back(p[1]);
  buffer.push_back(p[0]);
}

void ttv::broadcast::test::WriteDouble(std::vector<uint8_t>& buffer, double value) {
  const uint8_t* p = reinterpret_cast<const uint8_t*>(&value);

  for (size_t i = 0; i < 8; ++i) {
    buffer.push_back(p[7 - i]);
  }
}

void ttv::broadcast::test::WriteString(std::vector<uint8_t>& buffer, const std::string& str, bool writeNull) {
  for (size_t i = 0; i < str.size(); ++i) {
    buffer.push_back(str[i]);
  }

  if (writeNull) {
    buffer.push_back(0);
  }
}
