/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "broadcasttestutility.h"
#include "fixtures/broadcastbasetest.h"
#include "pubsubtestutility.h"
#include "rtmp.h"
#include "testsocket.h"
#include "testsocketfactory.h"
#include "testutilities.h"
#include "testvideoencoder.h"
#include "twitchsdk/broadcast/broadcastlistener.h"
#include "twitchsdk/broadcast/internal/ingestsampledata.h"
#include "twitchsdk/broadcast/internal/ingesttester.h"
#include "twitchsdk/broadcast/internal/streamer.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"
#include "usertestutility.h"

#include <fstream>

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::test;
using namespace ttv::broadcast;
using namespace ttv::broadcast::test;
using namespace ttv::broadcast::test::rtmp;

namespace {
class IngestTesterTest : public BroadcastBaseTest {
 protected:
  virtual void SetUpStubs() override {
    // BroadcastBaseTest::SetUpStubs();

    mPubSubTestUtility.SetUpStubs();
  }

  virtual void SetUpComponents() override {
    BroadcastBaseTest::SetUpComponents();

    auto userRepository = CreateUserRepository();
    InitializeComponent(userRepository);

    UserInfo userInfo;
    userInfo.userName = "test_user_name";
    userInfo.displayName = "test_display_name";
    userInfo.userId = 9001;
    mUser = GetUserRepository()->RegisterUser(userInfo.userId);
    mUser->SetUserInfo(userInfo);
    mUser->SetOAuthToken(std::make_shared<OAuthToken>("u83bi514zrc1w3f6bcjx6ca5eg7kc1"));

    mPubSubTestUtility.SetUpComponents(CreateTaskRunner(), mUser);
    InitializeComponent(mPubSubTestUtility.GetPubSubClient());

    mTestVideoEncoder = std::make_shared<TestVideoEncoder>();
    mIngestTesterListener = std::make_shared<IngestTesterListenerProxy>();

    mIngestTester = std::make_shared<ttv::broadcast::IngestTester>(mUser, std::make_shared<StreamerContext>());
    mIngestTester->AddListener(mIngestTesterListener);
    mIngestTester->SetTaskRunner(CreateTaskRunner());
    AddComponent(mIngestTester);

    mRtmpListener = std::make_shared<RtmpListenerProxy>();
  }

  virtual void TearDownComponents() override {
    mPubSubTestUtility.TearDownComponents();

    BroadcastBaseTest::TearDownComponents();
  }

  virtual void TearDownStubs() override {
    mPubSubTestUtility.TearDownStubs();

    BroadcastBaseTest::TearDownStubs();
  }

  void RegisterChannelInfoRequest(
    const std::string& /*userName*/, const std::string& /*displayName*/, const std::string& streamKey) {
    auto json = BroadcastTestUtility::JsonForChannelInfo(mUser->GetUserName(), mUser->GetDisplayName(), streamKey);
    mHttpRequest->AddResponse("https://api.twitch.tv/kraken/channel")
      .AddRequestHeader("Accept", "application/vnd.twitchtv.v1+json")
      .SetResponseBody(json)
      .Done();
  }

  std::shared_ptr<TestSocket> RegisterRtmpSocket(const std::string& host) {
    std::string ingestUrl = host + ":" + "1935";

    std::shared_ptr<TestSocket> testSocket = std::make_shared<TestSocket>();
    mTestSocketFactory->SetSocket(testSocket, ingestUrl);

    mRtmpServer.SetBandwidthTest(true);
    mRtmpServer.SetSocket(testSocket);
    mRtmpServer.AddListener(mRtmpListener);

    return testSocket;
  }

 protected:
  PubSubTestUtility mPubSubTestUtility;
  RtmpServer mRtmpServer;
  std::shared_ptr<User> mUser;
  std::shared_ptr<ttv::broadcast::IngestTester> mIngestTester;
  std::shared_ptr<IngestTesterListenerProxy> mIngestTesterListener;
  std::shared_ptr<RtmpListenerProxy> mRtmpListener;
  std::shared_ptr<TestVideoEncoder> mTestVideoEncoder;
};

void ValidateStateChange(IIngestTester::TestState prevState, IIngestTester::TestState newState) {
  switch (newState) {
    case IIngestTester::TestState::Stopped:
      EXPECT_FALSE(true);
      break;
    case IIngestTester::TestState::Connecting:
      EXPECT_TRUE(prevState == IIngestTester::TestState::Stopped);
      break;
    case IIngestTester::TestState::Testing:
      EXPECT_TRUE(prevState == IIngestTester::TestState::Connecting);
      break;
    case IIngestTester::TestState::Disconnecting:
      EXPECT_TRUE(prevState == IIngestTester::TestState::Testing || prevState == IIngestTester::TestState::Connecting);
      break;
    case IIngestTester::TestState::Finished:
      EXPECT_TRUE(prevState == IIngestTester::TestState::Disconnecting);
      break;
    case IIngestTester::TestState::Failed:
      EXPECT_TRUE(
        prevState == IIngestTester::TestState::Connecting || prevState == IIngestTester::TestState::Disconnecting);
      break;
  }
}
}  // namespace

TEST(IngestTesterTest, HydrateTestVideoData) {
  std::ifstream sampleDataFile("broadcast/TestVideoData.data", std::ios_base::binary);
  ASSERT_TRUE(sampleDataFile.is_open());

  std::vector<uint8_t> rawData;
  rawData.reserve(100000);
  rawData.assign(std::istreambuf_iterator<char>(sampleDataFile), std::istreambuf_iterator<char>());

  IngestSampleData sampleData;
  ASSERT_TRUE(TTV_SUCCEEDED(sampleData.Parse(rawData.data(), rawData.size())));

  ASSERT_EQ(sampleData.spsData.size(), 24);
  ASSERT_EQ(sampleData.ppsData.size(), 8);
  ASSERT_EQ(sampleData.frames.size(), 4);

  ASSERT_TRUE(sampleData.frames[0].keyFrame);
  ASSERT_EQ(sampleData.frames[0].data.size(), 76885);

  ASSERT_FALSE(sampleData.frames[1].keyFrame);
  ASSERT_EQ(sampleData.frames[1].data.size(), 94);

  ASSERT_TRUE(sampleData.frames[2].keyFrame);
  ASSERT_EQ(sampleData.frames[2].data.size(), 3066);

  ASSERT_FALSE(sampleData.frames[3].keyFrame);
  ASSERT_EQ(sampleData.frames[3].data.size(), 93);
}

TEST_F(IngestTesterTest, DISABLED_EnsureRequiredPropertiesAreSet) {
  mIngestTesterListener->mOnStateChangedFunc = [this](IIngestTester* source) {
    EXPECT_EQ(source, mIngestTester.get());
    EXPECT_FALSE(true);
  };

  // Don't set any required params
  InitializeComponent(mIngestTester, 100, TTV_EC_BROADCAST_INVALID_ENCODER);

  // Remove the component so it doesn't get shutdown automatically
  RemoveComponent(mIngestTester);
}

TEST_F(IngestTesterTest, DISABLED_InitializeAndShutdown) {
  mIngestTesterListener->mOnStateChangedFunc = [this](IIngestTester* source) {
    EXPECT_EQ(source, mIngestTester.get());
    EXPECT_FALSE(true);
  };

  InitializeComponent(mIngestTester);
}

TEST_F(IngestTesterTest, DISABLED_SingleServer) {
  TTV_ErrorCode ec = mTestVideoEncoder->Initialize();
  EXPECT_TRUE(TTV_SUCCEEDED(ec));

  InitializeComponent(mIngestTester);

  const std::string kStreamKey = "valid_stream_key";
  const std::string kIngestHost = "live-syd.twitch.tv";

  IngestServer server;
  server.serverName = "Fake Ingest Server";
  server.serverUrl = "rtmp://" + kIngestHost + "/app/{stream_key}";
  server.priority = 0;

  RegisterChannelInfoRequest(mUser->GetUserName(), mUser->GetDisplayName(), kStreamKey);

  // Create the test socket
  auto testSocket = RegisterRtmpSocket(kIngestHost);

  // Configure the IngestTester
  IIngestTester::TestState testState = IIngestTester::TestState::Stopped;
  ec = mIngestTester->GetTestState(testState);
  EXPECT_TRUE(TTV_SUCCEEDED(ec));
  EXPECT_TRUE(testState == IIngestTester::TestState::Stopped);

  const uint64_t kTestDuration = 1000;
  const uint64_t kMaxConnectTime = 1000000;
  ec = mIngestTester->SetTestDurationMilliseconds(kTestDuration);
  EXPECT_TRUE(TTV_SUCCEEDED(ec));

  // Start the test
  mIngestTesterListener->mOnStateChangedFunc = [this, &testState](IIngestTester* source) {
    IIngestTester::TestState prevState = testState;

    EXPECT_EQ(source, mIngestTester.get());
    TTV_ErrorCode ec = source->GetTestState(testState);
    EXPECT_TRUE(TTV_SUCCEEDED(ec));

    ValidateStateChange(prevState, testState);
  };

  mRtmpListener->mOnStateChangedFunc = [this, &testState](RtmpState /*state*/) {};

  mRtmpListener->mOnConnectCommandReceivedFunc = [this, &testState](const std::shared_ptr<ConnectCommand>& /*cmd*/) {};

  ec = mIngestTester->Start(server);
  EXPECT_TRUE(TTV_SUCCEEDED(ec));

  std::function<void()> updateFunc = [this]() { GetDefaultUpdateFunc()(); };

  std::function<bool()> checkFunc = [this, &testState]() {
    return testState == IIngestTester::TestState::Finished || testState == IIngestTester::TestState::Failed;
  };

  // Wait for the test to complete
  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(kMaxConnectTime + kTestDuration, checkFunc, updateFunc));
}

TEST_F(IngestTesterTest, DISABLED_TestIngestTesterLoop) {
  const uint64_t kTestDuration = 1000;
  mIngestTester->SetTestDurationMilliseconds(kTestDuration);

  const int kNumTests = 6;
  int currentTest = 0;

  IngestServer server;
  server.serverName = "SFO";
  server.serverUrl = "rtmp://live.twitch.tv/app/{stream_key}";

  auto pump = [this, kNumTests, &currentTest, &server]() {
    IIngestTester::TestState state;
    mIngestTester->GetTestState(state);

    if (state == IIngestTester::TestState::Finished || state == IIngestTester::TestState::Failed ||
        state == IIngestTester::TestState::Stopped) {
      currentTest++;

      if (currentTest < kNumTests) {
        mIngestTester->Start(server);
      }
    }
  };

  mIngestTesterListener->mOnStateChangedFunc = [this, pump](IIngestTester* /*source*/) {
    IIngestTester::TestState state;
    mIngestTester->GetTestState(state);

    pump();
  };

  TTV_ErrorCode ec = mTestVideoEncoder->Initialize();
  EXPECT_TRUE(TTV_SUCCEEDED(ec));

  InitializeComponent(mIngestTester);

  pump();

  auto checkFunc = [this, kNumTests, &currentTest]() { return kNumTests <= currentTest; };

  EXPECT_TRUE(WaitUntilResultWithPollTask(kNumTests * kTestDuration * 2, checkFunc, GetDefaultUpdateFunc()));
}
