/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "internal/pch.h"

#include "internal/bindings/csharp/csharp_broadcast.h"

#include "twitchsdk.h"

using namespace ttv;

namespace ttv {
ManagedBroadcastAPIListener gManagedBroadcastAPIListener;
ManagedStatsListener gManagedStatsListener;
bool gBroadcastInitialized = false;
}  // namespace ttv

#if TTV_PLATFORM_MAC
#define EXPORT_API __attribute__((visibility("default")))
#endif

extern "C" void TTV_SetEncoders(TTV_VideoEncoder vidEncoder, TTV_AudioEncoder audEncoder);

extern "C" EXPORT_API void TTV_CSharp_Broadcast_SetEncoders(TTV_VideoEncoder vidEncoder, TTV_AudioEncoder audEncoder) {
  return TTV_SetEncoders(vidEncoder, audEncoder);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetDefaultParams(TTV_VideoParams* videoParams) {
  return TTV_GetDefaultParams(videoParams);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_Init(const ManagedBroadcastAPIListener* listener) {
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, false, TTV_EC_ALREADY_INITIALIZED);

  gBroadcastInitialized = true;
  gManagedBroadcastAPIListener = *listener;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_Shutdown() {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  gBroadcastInitialized = false;
  memset(&gManagedBroadcastAPIListener, 0, sizeof(gManagedBroadcastAPIListener));

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetMaxResolution(
  uint maxKbps, uint frameRate, float bitsPerPixel, float aspectRatio, uint* width, uint* height) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_GetMaxResolution(maxKbps, frameRate, bitsPerPixel, aspectRatio, width, height);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_PollTasks() {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_PollTasks();
}

namespace {
void HandleRequestAuthTokenCallback(TTV_ErrorCode ec, void* userData) {
  TTV_AuthToken* result = reinterpret_cast<TTV_AuthToken*>(userData);

  gManagedBroadcastAPIListener.FireRequestAuthTokenCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_RequestAuthToken(
  const TTV_AuthParams* authParams, uint32_t flags) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_AuthToken* result = new TTV_AuthToken();
  memset(result, 0, sizeof(*result));

  TTV_ErrorCode ret = TTV_RequestAuthToken(authParams, flags, &HandleRequestAuthTokenCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

namespace {
void HandleLoginCallback(TTV_ErrorCode ec, void* userData) {
  TTV_ChannelInfo* result = reinterpret_cast<TTV_ChannelInfo*>(userData);

  gManagedBroadcastAPIListener.FireLoginCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_Login(const TTV_AuthToken* authToken) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_ChannelInfo* result = new TTV_ChannelInfo();
  memset(result, 0, sizeof(*result));
  result->size = sizeof(*result);

  TTV_ErrorCode ret = TTV_Login(authToken, &HandleLoginCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

namespace {
void HandleGetIngestServersCallback(TTV_ErrorCode ec, void* userData) {
  TTV_IngestList* result = reinterpret_cast<TTV_IngestList*>(userData);

  gManagedBroadcastAPIListener.FireGetIngestListCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetIngestServers(const TTV_AuthToken* authToken) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_IngestList* result = new TTV_IngestList();
  memset(result, 0, sizeof(*result));

  TTV_ErrorCode ret = TTV_GetIngestServers(authToken, &HandleGetIngestServersCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeIngestList(TTV_IngestList* ingestList) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_FreeIngestList(ingestList);
}

namespace {
void HandleStartCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireStartCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_Start(const TTV_VideoParams* videoParams,
  const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags, bool async) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec;

  if (async) {
    ec = TTV_Start(videoParams, audioParams, ingestServer, flags, &HandleStartCallback, nullptr);
  } else {
    ec = TTV_Start(videoParams, audioParams, ingestServer, flags, nullptr, nullptr);
  }

  return ec;
}

namespace {
void HandleStopCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireStopCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_Stop(bool async) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec;

  if (async) {
    ec = TTV_Stop(&HandleStopCallback, nullptr);
  } else {
    ec = TTV_Stop(nullptr, nullptr);
  }

  return ec;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_PauseVideo() {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_PauseVideo();
}

namespace {
void BufferUnlockCallback(const uint8_t* buffer, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireBufferUnlockCallback(buffer);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SubmitVideoFrame(const uint8_t* frameBuffer) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
  return TTV_SubmitVideoFrame(frameBuffer, &BufferUnlockCallback, nullptr);
#pragma clang diagnostic pop
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SubmitAudioSamples(
  const int16_t* samplesBuffer, uint numSamples) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_SubmitAudioSamples(samplesBuffer, numSamples);
}

namespace {
void HandleSendActionMetaDataCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireSendActionMetaDataCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SendActionMetaData(const TTV_AuthToken* authToken,
  const char* name, uint64_t streamTime, const char* humanDescription, const char* data) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_SendActionMetaData(
    authToken, name, streamTime, humanDescription, data, &HandleSendActionMetaDataCallback, nullptr);
}

namespace {
void HandleSendStartSpanMetaDataCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireSendStartSpanMetaDataCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SendStartSpanMetaData(const TTV_AuthToken* authToken,
  const char* name, uint64_t streamTime, unsigned long* sequenceId, const char* humanDescription, const char* data) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_SendStartSpanMetaData(
    authToken, name, streamTime, sequenceId, humanDescription, data, &HandleSendStartSpanMetaDataCallback, nullptr);
}

namespace {
void HandleSendEndSpanMetaDataCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireSendEndSpanMetaDataCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SendEndSpanMetaData(const TTV_AuthToken* authToken,
  const char* name, uint64_t streamTime, unsigned long sequenceId, const char* humanDescription, const char* data) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_SendEndSpanMetaData(
    authToken, name, streamTime, sequenceId, humanDescription, data, &HandleSendEndSpanMetaDataCallback, nullptr);
}

namespace {
void HandleSetStreamInfoCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireSetStreamInfoCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SetStreamInfo(
  const TTV_AuthToken* authToken, const char* channel, const TTV_StreamInfoForSetting* streamInfoToSet) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_SetStreamInfo(authToken, channel, streamInfoToSet, &HandleSetStreamInfoCallback, nullptr);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetVolume(TTV_AudioDeviceType device, float* volume) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_GetVolume(device, volume);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_SetVolume(TTV_AudioDeviceType device, float volume) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_SetVolume(device, volume);
}

namespace {
void HandleGetUserInfoCallback(TTV_ErrorCode ec, void* userData) {
  TTV_UserInfo___* result = reinterpret_cast<TTV_UserInfo___*>(userData);

  gManagedBroadcastAPIListener.FireGetUserInfoCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetUserInfo(const TTV_AuthToken* authToken) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_UserInfo___* result = new TTV_UserInfo___();
  memset(result, 0, sizeof(*result));
  result->size = sizeof(*result);

  TTV_ErrorCode ret = TTV_GetUserInfo(authToken, &HandleGetUserInfoCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

namespace {
void HandleGetStreamInfoCallback(TTV_ErrorCode ec, void* userData) {
  TTV_StreamInfo* result = reinterpret_cast<TTV_StreamInfo*>(userData);

  gManagedBroadcastAPIListener.FireGetStreamInfoCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetStreamInfo(
  const TTV_AuthToken* authToken, const char* channel) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_StreamInfo* result = new TTV_StreamInfo();
  memset(result, 0, sizeof(*result));
  result->size = sizeof(*result);

  TTV_ErrorCode ret = TTV_GetStreamInfo(authToken, &HandleGetStreamInfoCallback, result, channel, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

namespace {
void HandleGetArchivingStateCallback(TTV_ErrorCode ec, void* userData) {
  TTV_ArchivingState* result = reinterpret_cast<TTV_ArchivingState*>(userData);

  gManagedBroadcastAPIListener.FireGetArchivingStateCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetArchivingState(const TTV_AuthToken* authToken) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_ArchivingState* result = new TTV_ArchivingState();
  memset(result, 0, sizeof(*result));
  result->size = sizeof(*result);

  TTV_ErrorCode ret = TTV_GetArchivingState(authToken, &HandleGetArchivingStateCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

namespace {
void HandleRunCommercialCallback(TTV_ErrorCode ec, void* /*userData*/) {
  gManagedBroadcastAPIListener.FireRunCommercialCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_RunCommercial(const TTV_AuthToken* authToken) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ret = TTV_RunCommercial(authToken, &HandleRunCommercialCallback, nullptr);

  return ret;
}

namespace {
void HandleGetGameLiveStreamsCallback(TTV_ErrorCode ec, void* userData) {
  TTV_LiveGameStreamList* result = reinterpret_cast<TTV_LiveGameStreamList*>(userData);

  gManagedBroadcastAPIListener.FireGetGameLiveStreamsCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTVSDK_API TTV_ErrorCode TTV_CSharp_Broadcast_GetGameLiveStreams(
  const char* gameName, uint limit, uint offset) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_LiveGameStreamList* result = new TTV_LiveGameStreamList();
  memset(result, 0, sizeof(*result));

  TTV_ErrorCode ret =
    TTV_GetGameLiveStreams(gameName, limit, offset, &HandleGetGameLiveStreamsCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeGameLiveStreamList(
  TTV_LiveGameStreamList* gameStreamList) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_FreeGameLiveStreamList(gameStreamList);
}

namespace {
void HandleStatCallback(TTV_StatType type, uint64_t data) {
  gManagedStatsListener.FireStatCallback(type, data);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_RegisterStatsCallback(const ManagedStatsListener* listener) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  gManagedStatsListener = *listener;

  return TTV_RegisterStatsCallback(&HandleStatCallback);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_RemoveStatsCallback(const ManagedStatsListener* listener) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(listener, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_RemoveStatsCallback(&HandleStatCallback);

  if (TTV_SUCCEEDED(ec)) {
    memset(&gManagedStatsListener, 0, sizeof(gManagedStatsListener));
  }

  return ec;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_PollStats() {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_PollStats();
}

namespace {
void HandleGetGameNameListCallback(TTV_ErrorCode ec, void* userData) {
  TTV_GameInfoList* result = reinterpret_cast<TTV_GameInfoList*>(userData);

  gManagedBroadcastAPIListener.FireGetGameNameListCallback(ec, result);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetGameNameList(const char* str) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  TTV_GameInfoList* result = new TTV_GameInfoList();
  memset(result, 0, sizeof(*result));

  TTV_ErrorCode ret = TTV_GetGameNameList(str, &HandleGetGameNameListCallback, result, result);

  // call failed so abort
  if (TTV_FAILED(ret)) {
    delete result;
  }

  return ret;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeGameNameList(const TTV_GameInfoList* gameList) {
  TTV_RETURN_ON_NULL(gameList, TTV_EC_INVALID_ARG);

  delete[] gameList->list;
  delete gameList;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_GetStreamTime(uint64_t* timeMs) {
  TTV_RETURN_ON_DIFFERENT(gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);

  return TTV_GetStreamTime(timeMs);
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeAuthToken(const TTV_AuthToken* authToken) {
  TTV_RETURN_ON_NULL(authToken, TTV_EC_INVALID_ARG);

  delete authToken;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeChannelInfo(const TTV_ChannelInfo* channelInfo) {
  TTV_RETURN_ON_NULL(channelInfo, TTV_EC_INVALID_ARG);

  delete channelInfo;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeStreamInfo(const TTV_StreamInfo* streamInfo) {
  TTV_RETURN_ON_NULL(streamInfo, TTV_EC_INVALID_ARG);

  delete streamInfo;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeUserInfo(const TTV_UserInfo___* userInfo) {
  TTV_RETURN_ON_NULL(userInfo, TTV_EC_INVALID_ARG);

  delete userInfo;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeArchivingState(const TTV_ArchivingState* state) {
  TTV_RETURN_ON_NULL(state, TTV_EC_INVALID_ARG);

  delete state;

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_AllocateFrameBuffer(unsigned int size, void** buffer) {
  *buffer = ttv::gAllocCallback(size, 16);

  if ((*buffer) == nullptr) {
    return TTV_EC_MEMORY;
  }

#if 0
    unsigned char* pDest = static_cast<unsigned char*>(*buffer);

    for (size_t i=0; i<size; ++i)
    {
        pDest[i] = (uint8_t)rand();
    }
#endif

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_FreeFrameBuffer(void* buffer) {
  ttv::gFreeCallback(buffer);
  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Broadcast_RandomizeFrameBuffer(uint8_t* buffer, unsigned int size) {
  if (buffer == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  for (unsigned int i = 0; i < size; ++i) {
    buffer[i] = static_cast<uint8_t>(rand());
  }

  return TTV_EC_SUCCESS;
}
