/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2012-2017 Twitch Interactive, Inc.
 *********************************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"
#include "twitchsdk/broadcast/darwintestcameracapturer.h"
#include "twitchsdk/broadcast/icvpixelbuffervideoframereceiver.h"
#include "twitchsdk/broadcast/ivideoframequeue.h"
#include "twitchsdk/broadcast/ivideoencoder.h"

#include "twitchsdk/core/systemclock.h"

#include <CoreVideo/CoreVideo.h>

#import <Foundation/Foundation.h>
#import <AVFoundation/AVFoundation.h>

namespace
{
	using CapturedCallbackFunc = std::function<void(CVPixelBufferRef pixelBuffer)>;
	
	AVCaptureDevice* FindCameraWithPosition(AVCaptureDevicePosition position)
	{
		NSArray* devices = [AVCaptureDevice devicesWithMediaType:AVMediaTypeVideo];
		
		for (AVCaptureDevice* device in devices)
		{
			if ([device position] == position)
			{
				return device;
			}
		}
		
		return nil;
	}
}

typedef void (*PixelBufferReleaseCallback)(CVPixelBufferRef pixelBuffer);


@interface DarwinTestCameraVideoCapturerProxy : NSObject<AVCaptureVideoDataOutputSampleBufferDelegate>

- (instancetype)initWithCapturedCallback:(CapturedCallbackFunc)func;

- (void)captureOutput:(AVCaptureOutput *)captureOutput didOutputSampleBuffer:(CMSampleBufferRef)sampleBuffer fromConnection:(AVCaptureConnection*)connection;

@property (assign, nonatomic, readonly) CapturedCallbackFunc frameCapturedFunc;

@end

@implementation DarwinTestCameraVideoCapturerProxy

@synthesize frameCapturedFunc = _frameCapturedFunc;

- (instancetype)initWithCapturedCallback:(CapturedCallbackFunc)func
{
	if ((self = [super init]))
	{
		_frameCapturedFunc = func;
	}
	
	return self;
}

/**
 * AVCaptureVideoDataOutputSampleBufferDelegate implementation
 * The callback from AVCaptureVideoDataOutput to supply video frames.
 */
- (void)captureOutput:(AVCaptureOutput*)captureOutput didOutputSampleBuffer:(CMSampleBufferRef)sampleBuffer fromConnection:(AVCaptureConnection*)connection
{
	if (self.frameCapturedFunc != nullptr)
	{
		CVImageBufferRef pixelBuffer = CMSampleBufferGetImageBuffer(sampleBuffer);
		self.frameCapturedFunc(pixelBuffer);
	}
}

@end


struct ttv::broadcast::test::DarwinTestCameraVideoCapturerInternalData
{
	DarwinTestCameraVideoCapturerInternalData(DarwinTestCameraVideoCapturer::CameraType type)
		: captureDevice(nullptr)
		, captureSession(nullptr)
		, cameraType(type)
		, preset(AVCaptureSessionPreset640x480)
		, started(false)
	{
	}
	
	DarwinTestCameraVideoCapturerInternalData(const DarwinTestCameraVideoCapturerInternalData& other) = delete;
	
	~DarwinTestCameraVideoCapturerInternalData()
	{
		Stop();

		if (capturerProxy != nil)
		{
			capturerProxy = nil;
		}
	}
	
	void Stop()
	{
		if (captureSession != nil)
		{
			if (captureDeviceVideoInput != nil)
			{
				[captureSession removeInput:captureDeviceVideoInput];
				captureDeviceVideoInput = nil;
			}
			
			captureSession = nil;
		}
		
		started = false;
	}
	
	AVCaptureDevice* captureDevice;
	AVCaptureDeviceInput* captureDeviceVideoInput;
	AVCaptureSession* captureSession;

	std::shared_ptr<IVideoEncoder> videoEncoder;
	std::shared_ptr<IVideoFrameQueue> frameQueue;
	std::shared_ptr<ICVPixelBufferVideoFrameReceiver> receiver;
	DarwinTestCameraVideoCapturerProxy* capturerProxy;
	DarwinTestCameraVideoCapturer::CameraType cameraType;
	NSString* preset;
	bool started;
};


ttv::broadcast::test::DarwinTestCameraVideoCapturer::DarwinTestCameraVideoCapturer(CameraType type)
	: mInternalData(std::make_shared<DarwinTestCameraVideoCapturerInternalData>(type))
{
}


std::string ttv::broadcast::test::DarwinTestCameraVideoCapturer::GetName() const
{
	return "DarwinTestCameraVideoCapturer";
}
	

TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::SetVideoEncoder(const std::shared_ptr<IVideoEncoder>& encoder)
{
	mInternalData->videoEncoder = encoder;
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::SetFrameQueue(const std::shared_ptr<IVideoFrameQueue>& queue)
{
	mInternalData->frameQueue = queue;
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::Initialize()
{
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::SetPreset(const std::string& preset)
{
	mInternalData->preset = [NSString stringWithCString:preset.c_str() encoding:[NSString defaultCStringEncoding]];
	
	NSString* presets[] =
	{
		//AVCaptureSessionPreset320x240
		AVCaptureSessionPreset352x288
		, AVCaptureSessionPreset640x480
		//, AVCaptureSessionPreset960x540,
		, AVCaptureSessionPreset1280x720
#if TTV_TARGET_IOS
		, AVCaptureSessionPreset1920x1080
		, AVCaptureSessionPreset3840x2160
#endif
	};
	
	for (size_t i = 0; i < sizeof(presets) / sizeof(presets[0]); ++i)
	{
		if (mInternalData->preset == presets[i])
		{
			return TTV_EC_SUCCESS;
		}
	}
	
	mInternalData->preset = nil;
	
	return TTV_EC_INVALID_ARG;
}


TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::Start(const VideoParams& videoParams)
{
	auto& data = *mInternalData;
	
	if (mInternalData->started ||
		mInternalData->frameQueue == nullptr ||
		mInternalData->videoEncoder == nullptr)
	{
		return TTV_EC_INVALID_STATE;
	}

	// Find the desired camera
	switch (data.cameraType)
	{
		case CameraType::Front:
		{
			data.captureDevice = FindCameraWithPosition(AVCaptureDevicePositionFront);
			break;
		}
		case CameraType::Back:
		{
			data.captureDevice = FindCameraWithPosition(AVCaptureDevicePositionBack);
			break;
		}
		case CameraType::Default:
		default:
		{
			NSArray* videoDevices = [AVCaptureDevice devicesWithMediaType:AVMediaTypeVideo];
			if (videoDevices.count == 0)
			{
				// TODO: We need a better error
				return TTV_EC_INVALID_ARG;
			}
			
			data.captureDevice = videoDevices[0];
			
			break;
		}
	}
	
	if (data.captureDevice == nil)
	{
		// TODO: Return a better error code
		return TTV_EC_INVALID_ARG;
	}
	
	auto frameCapturedCallback = [data = mInternalData](CVPixelBufferRef pixelBuffer)
	{
		if (!data->started)
		{
			return;
		}
		
		std::shared_ptr<VideoFrame> videoFrame;
		TTV_ErrorCode ec = data->receiver->PackageFrame(pixelBuffer, ttv::GetSystemClockTime(), nullptr, videoFrame);
		
		if (TTV_FAILED(ec))
		{
			return;
		}
		
		ec = data->frameQueue->AddVideoFrame(videoFrame);
		if (TTV_FAILED(ec))
		{
			return;
		}
	};
	
	data.capturerProxy = [[DarwinTestCameraVideoCapturerProxy alloc] initWithCapturedCallback:frameCapturedCallback];
	
	auto receiver = data.videoEncoder->GetReceiverImplementation(ICVPixelBufferVideoFrameReceiver::GetReceiverTypeId());
	TTV_ASSERT(receiver != nullptr);
	
	data.receiver = std::static_pointer_cast<ICVPixelBufferVideoFrameReceiver>(receiver);
	
	data.captureSession = [[AVCaptureSession alloc] init];
	
	// Configure the camera
	[data.captureSession beginConfiguration];
	{
		data.captureSession.sessionPreset = data.preset;
		
		// Configure the input to the session from the camera device
		NSError* error = nil;
		data.captureDeviceVideoInput = [AVCaptureDeviceInput deviceInputWithDevice:data.captureDevice error:&error];
		if (data.captureDeviceVideoInput != nil)
		{
			[data.captureSession addInput:data.captureDeviceVideoInput];
		}
		else
		{
			// TODO: Log
			
			data.Stop();
			
			// TOO: Use a better error code
			return TTV_EC_NOT_INITIALIZED;
		}
		
		// Configure the output of the session to call our delegate with the buffers
		AVCaptureVideoDataOutput* videoOut = [[AVCaptureVideoDataOutput alloc] init];
		if (videoOut != nil)
		{
			[videoOut setAlwaysDiscardsLateVideoFrames:YES];
			[videoOut setVideoSettings:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:kCVPixelFormatType_32BGRA] forKey:id(kCVPixelBufferPixelFormatTypeKey)]]; // BGRA is necessary for manual preview
		}
		else
		{
			// TODO: Log
			
			data.Stop();
			
			// TOO: Use a better error code
			return TTV_EC_NOT_INITIALIZED;
		}
		
		// Create the queue for frames to be submitted to us by the camera
		dispatch_queue_t captureQueue = dispatch_queue_create("twitch.DarwinTestCameraVideoCapturer", nullptr);
		
		if ([data.captureSession canAddOutput:videoOut])
		{
			[data.captureSession addOutput:videoOut];
		}
		else
		{
			// TODO: Log
			
			data.Stop();
			
			// TOO: Use a better error code
			return TTV_EC_NOT_INITIALIZED;
		}
		
		// Set the delegate for capturing the frames
		[videoOut setSampleBufferDelegate:data.capturerProxy queue:captureQueue];
	}
	[data.captureSession commitConfiguration];

	// Start capturing from the camera
	[data.captureSession startRunning];
	
	mInternalData->started = true;
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::Stop()
{
	auto& data = *mInternalData;
	
	if (!mInternalData->started)
	{
		return TTV_EC_INVALID_STATE;
	}

	// Stop capturing
	[data.captureSession stopRunning];
	
	data.Stop();

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::broadcast::test::DarwinTestCameraVideoCapturer::Shutdown()
{
	return TTV_EC_SUCCESS;
}
