/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/appleaacencoder.h"

#include "twitchsdk/broadcast/audioconstants.h"
#include "twitchsdk/broadcast/iframewriter.h"
#include "twitchsdk/broadcast/internal/pcmaudioframe.h"
#include "twitchsdk/broadcast/packet.h"

namespace {
using namespace ttv;
using namespace ttv::broadcast;

const uint32_t kNumAudioChannels = 2;  // The number of channels captured.
const uint32_t kPcmSamplesPerAacFrame = 1024;
const char* kLoggerName = "AppleAacEncoder";

OSStatus AudioConverterInputCallback(AudioConverterRef /*inAudioConverter*/, UInt32* ioNumberDataPackets,
  AudioBufferList* ioData, AudioStreamPacketDescription** /*outDataPacketDescription*/, void* inUserData) {
  int16_t* mixedSamplesBuffer = reinterpret_cast<int16_t*>(inUserData);
  TTV_ASSERT(mixedSamplesBuffer);

  UInt32 requestedPackets = *ioNumberDataPackets;

  ioData->mNumberBuffers = 1;
  ioData->mBuffers[0].mData = mixedSamplesBuffer;
  ioData->mBuffers[0].mNumberChannels = kNumAudioChannels;
  ioData->mBuffers[0].mDataByteSize = requestedPackets * ioData->mBuffers[0].mNumberChannels * sizeof(int16_t);

  *ioNumberDataPackets = requestedPackets;

  return noErr;
}
}  // namespace

struct ttv::broadcast::AppleAacEncoderInternalData {
  AppleAacEncoderInternalData()
      : mAudioConverter(nullptr), mStreamIndex(0), mTotalSamplesWritten(0), mOutputPacketSize(0) {
    mPcmReceiver = std::make_shared<PcmAudioReceiver>(kPcmSamplesPerAacFrame);
  }

  std::shared_ptr<PcmAudioReceiver> mPcmReceiver;

  AudioConverterRef mAudioConverter;
  std::shared_ptr<IFrameWriter> mFrameWriter;
  uint32_t mStreamIndex;
  uint64_t mTotalSamplesWritten;
  UInt32 mOutputPacketSize;
};

ttv::broadcast::AppleAacEncoder::AppleAacEncoder() : mInternalData(std::make_unique<AppleAacEncoderInternalData>()) {}

ttv::broadcast::AppleAacEncoder::~AppleAacEncoder() {
  Stop();
}

std::string ttv::broadcast::AppleAacEncoder::GetName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::SetFrameWriter(const std::shared_ptr<IFrameWriter>& frameWriter) {
  // TODO: Make sure not encoding

  mInternalData->mFrameWriter = frameWriter;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::GetAudioEncodingFormat(AudioFormat& result) {
  result = AudioFormat::AAC;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::Initialize() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "AppleAacEncoder::Initialize()");

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::Shutdown() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "AppleAacEncoder::Shutdown()");

  return Stop();
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::Start(uint32_t streamIndex, const AudioParams& /*audioParams*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "AppleAacEncoder::Start()");

  auto& data = *mInternalData;

  TTV_ASSERT(data.mFrameWriter != nullptr);
  if (data.mFrameWriter == nullptr) {
    return TTV_EC_INVALID_STATE;
  }

  data.mStreamIndex = streamIndex;
  data.mTotalSamplesWritten = 0;

  // Set up the recording format
  AudioStreamBasicDescription recordFormat;
  memset(&recordFormat, 0, sizeof(recordFormat));
  recordFormat.mFormatID = kAudioFormatLinearPCM;
  recordFormat.mChannelsPerFrame = kNumAudioChannels;
  recordFormat.mFormatFlags = kLinearPCMFormatFlagIsPacked | kLinearPCMFormatFlagIsSignedInteger;
  recordFormat.mFramesPerPacket = 1;
  recordFormat.mBitsPerChannel = sizeof(int16_t) * 8;
  recordFormat.mBytesPerFrame = recordFormat.mChannelsPerFrame * recordFormat.mBitsPerChannel / 8;
  recordFormat.mBytesPerPacket = recordFormat.mBytesPerFrame * recordFormat.mFramesPerPacket;
  recordFormat.mSampleRate = kAudioEncodeRate;

  // Set up the output format
  AudioStreamBasicDescription outputFormat;
  memset(&outputFormat, 0, sizeof(outputFormat));
  outputFormat.mFormatID = kAudioFormatMPEG4AAC;
  outputFormat.mChannelsPerFrame = kNumAudioChannels;
  outputFormat.mFormatFlags = kMPEG4Object_AAC_LC;
  outputFormat.mFramesPerPacket = kPcmSamplesPerAacFrame;
  outputFormat.mSampleRate = kAudioEncodeRate;

  // Create the converter
  //
  OSStatus err = noErr;
#if TTV_TARGET_IOS
  // TODO - On iOS we need to use the software encoder. Using the hardware encoder generates corrupt audio.
  // This code throws an exception but we can skip it
  AudioClassDescription desc;
  desc.mType = kAudioEncoderComponentType;
  desc.mSubType = 'aac ';
  desc.mManufacturer = kAppleSoftwareAudioCodecManufacturer;
  err = AudioConverterNewSpecific(&recordFormat, &outputFormat, 1, &desc, &data.mAudioConverter);
#else
  err = AudioConverterNew(&recordFormat, &outputFormat, &data.mAudioConverter);
#endif
  TTV_ASSERT(err == noErr);

  if (err == noErr) {
    // Determine the max size of the output AAC packet
    UInt32 propSize = sizeof(data.mOutputPacketSize);
    OSStatus propErr = AudioConverterGetProperty(
      data.mAudioConverter, kAudioConverterPropertyMaximumOutputPacketSize, &propSize, &data.mOutputPacketSize);
    TTV_ASSERT(propErr == noErr);
  }

  return (err == noErr) ? TTV_EC_SUCCESS : TTV_EC_BROADCAST_APPLEAAC_FAILED_INIT;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::GetNumInputSamplesPerEncodeFrame(uint32_t& result) {
  result = kPcmSamplesPerAacFrame;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::SubmitPcmSamples(const int16_t* samples, uint numSamplesPerChannel) {
  // ttv::trace::Message(kLoggerName, MessageLevel::Debug, "AppleAacEncoder::SubmitPcmSamples()");

  auto& data = *mInternalData;

  TTV_ASSERT(data.mOutputPacketSize > 0);

  auto packet = std::make_unique<Packet>();
  packet->data.resize(data.mOutputPacketSize);

  // Create an AudioBufferList to write the converted data to
  AudioBufferList convertedData;
  convertedData.mNumberBuffers = 1;
  convertedData.mBuffers[0].mNumberChannels = kNumAudioChannels;
  convertedData.mBuffers[0].mDataByteSize = data.mOutputPacketSize;
  convertedData.mBuffers[0].mData = packet->data.data();

  // Now call the converter (we only generate one output packet at a time)
  const UInt32 numOutputDataPackets = 1;
  UInt32 ioOutputDataPackets = numOutputDataPackets;
  AudioStreamPacketDescription aspdList[numOutputDataPackets];
  OSStatus err = AudioConverterFillComplexBuffer(data.mAudioConverter, AudioConverterInputCallback,
    const_cast<int16_t*>(samples), &ioOutputDataPackets, &convertedData, aspdList);
  TTV_ASSERT(err == noErr);

  if (err != noErr) {
    return TTV_EC_BROADCAST_APPLEAAC_FAILED_ENCODING;
  }

  packet->data.resize(convertedData.mBuffers[0].mDataByteSize);
  packet->keyframe = true;
  packet->streamIndex = data.mStreamIndex;
  data.mTotalSamplesWritten += numSamplesPerChannel;
  packet->timestamp = data.mTotalSamplesWritten * 1000 / kAudioEncodeRate;

  return data.mFrameWriter->WritePacket(std::move(packet));
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::Stop() {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "AppleAacEncoder::Stop()");

  auto& data = *mInternalData;

  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  if (data.mAudioConverter == nullptr) {
    OSStatus err = AudioConverterDispose(data.mAudioConverter);
    data.mAudioConverter = nullptr;

    if (err != noErr) {
      ret = TTV_EC_BROADCAST_APPLEAAC_FAILED_SHUTDOWN;
    }
  }

  return ret;
}

TTV_ErrorCode ttv::broadcast::AppleAacEncoder::SubmitFrame(const std::shared_ptr<AudioFrame>& audioFrame) {
  if (audioFrame == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  // We only handle PCM frames
  TTV_ASSERT(audioFrame->GetReceiverTypeId() == IPcmAudioFrameReceiver::GetReceiverTypeId());
  if (audioFrame->GetReceiverTypeId() != IPcmAudioFrameReceiver::GetReceiverTypeId()) {
    return TTV_EC_BROADCAST_INVALID_SUBMISSION_METHOD;
  }

  auto pcmFrame = std::static_pointer_cast<PcmAudioFrame>(audioFrame);

  // TODO: Add support for other than signed 16-bit audio samples
  TTV_ASSERT(pcmFrame->GetAudioSampleFormat() == AudioSampleFormat::TTV_ASF_PCM_S16);
  if (pcmFrame->GetAudioSampleFormat() != AudioSampleFormat::TTV_ASF_PCM_S16) {
    return TTV_EC_UNIMPLEMENTED;
  }

  // TODO: We only handle interleved audio right now
  TTV_ASSERT(pcmFrame->GetInterleaved());
  if (!pcmFrame->GetInterleaved()) {
    return TTV_EC_UNIMPLEMENTED;
  }

  TTV_ASSERT(pcmFrame->GetNumChannels() == kNumAudioChannels);

  const int16_t* sampleBuffer = reinterpret_cast<const int16_t*>(pcmFrame->GetSampleBuffer().data());

  return SubmitPcmSamples(sampleBuffer, pcmFrame->GetNumSamplesPerChannel());
}

bool ttv::broadcast::AppleAacEncoder::SupportsReceiverProtocol(IAudioFrameReceiver::ReceiverTypeId typeId) const {
  return typeId == IPcmAudioFrameReceiver::GetReceiverTypeId();
}

std::shared_ptr<ttv::broadcast::IAudioFrameReceiver> ttv::broadcast::AppleAacEncoder::GetReceiverImplementation(
  IAudioFrameReceiver::ReceiverTypeId typeId) {
  if (typeId == IPcmAudioFrameReceiver::GetReceiverTypeId()) {
    return mInternalData->mPcmReceiver;
  }

  return nullptr;
}
