/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/appleaudiocapture.h"

#include "twitchsdk/broadcast/audioconstants.h"
#include "twitchsdk/broadcast/audioframe.h"
#include "twitchsdk/broadcast/iaudiomixer.h"
#include "twitchsdk/broadcast/ipcmaudioframereceiver.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/threadsync.h"

#include <AudioToolbox/AudioToolbox.h>

#include <atomic>

namespace {
const bool kPcmInterleaved = true;
const uint32_t kNumAudioChannels = 2;

// TODO move to non-ios folder and remove from ios project
#if !TTV_TARGET_IOS

AudioDeviceID GetDefaultInputDevice() {
  AudioDeviceID inputDevice = 0;
  UInt32 size = sizeof(AudioDeviceID);

  AudioObjectPropertyAddress address = {
    kAudioHardwarePropertyDefaultInputDevice, kAudioObjectPropertyScopeGlobal, kAudioObjectPropertyElementMaster};

  OSStatus err = AudioObjectGetPropertyData(kAudioObjectSystemObject, &address, 0, nullptr, &size, &inputDevice);
  TTV_ASSERT(err == noErr);

  return inputDevice;
}

AudioDeviceID GetSoundflowerDevice() {
  AudioDeviceID sfDevice = 0;

  const unsigned int kMaxDevices = 64;
  AudioDeviceID devices[kMaxDevices];
  memset(devices, 0, sizeof(devices));

  // Get the list of all audio devices
  AudioObjectPropertyAddress thePropertyAddress = {
    kAudioHardwarePropertyDevices, kAudioObjectPropertyScopeInput, kAudioObjectPropertyElementMaster};

  UInt32 arraySize = sizeof(devices);
  OSStatus err =
    AudioObjectGetPropertyData(kAudioObjectSystemObject, &thePropertyAddress, 0, nullptr, &arraySize, &devices);
  TTV_ASSERT(err == noErr);

  if (err == noErr) {
    unsigned int deviceCount = arraySize / sizeof(AudioDeviceID);

    for (unsigned int i = 0; i < deviceCount; ++i) {
      // Determine if the device is an input device (it is an input device if it has input channels)
      //
      UInt32 size = 0;
      thePropertyAddress.mSelector = kAudioDevicePropertyStreamConfiguration;
      err = AudioObjectGetPropertyDataSize(devices[i], &thePropertyAddress, 0, nullptr, &size);
      TTV_ASSERT(err == kAudioHardwareNoError);
      if (err == kAudioHardwareNoError) {
        std::unique_ptr<AudioBufferList> bufferList(reinterpret_cast<AudioBufferList*>(new uint8_t[size]));
        TTV_ASSERT(bufferList);

        err = AudioObjectGetPropertyData(devices[i], &thePropertyAddress, 0, nullptr, &size, bufferList.get());
        TTV_ASSERT(err == kAudioHardwareNoError);
        if (err == kAudioHardwareNoError && bufferList->mNumberBuffers > 0) {
          // This is an input device. Get its name and check if it's a SoundFlower device
          CFStringRef deviceName;
          size = sizeof(deviceName);
          thePropertyAddress.mSelector = kAudioDevicePropertyDeviceNameCFString;

          err = AudioObjectGetPropertyData(devices[i], &thePropertyAddress, 0, nullptr, &size, &deviceName);
          TTV_ASSERT(err == kAudioHardwareNoError);

          CFRange range = CFStringFind(deviceName, CFSTR("soundflower"), kCFCompareCaseInsensitive);
          if (range.location >= 0) {
            // Found a SoundFlower input device
            sfDevice = devices[i];
            break;
          }
        }
      }
    }
  }

  return sfDevice;
}

CFStringRef GetDeviceUID(AudioDeviceID device) {
  CFStringRef deviceUID = nullptr;

  AudioObjectPropertyAddress thePropertyAddress = {
    kAudioDevicePropertyDeviceUID, kAudioObjectPropertyScopeInput, kAudioObjectPropertyElementMaster};
  UInt32 size = sizeof(deviceUID);

  OSStatus err = AudioObjectGetPropertyData(device, &thePropertyAddress, 0, nullptr, &size, &deviceUID);
  TTV_ASSERT(err == kAudioHardwareNoError);

  return deviceUID;
}

#endif  // !TTV_TARGET_IOS
}  // namespace

class ttv::broadcast::AppleAudioCaptureInternalData {
 public:
  AppleAudioCaptureInternalData(AppleAudioCapture::CaptureType typeArg)
      : audioQueue(nullptr),
        captureBufferWritePos(0),
        captureBufferReadPos(0),
        sampleTime(0),
        type(typeArg),
        initializationError(TTV_EC_NOT_INITIALIZED) {}

  ~AppleAudioCaptureInternalData() { Stop(); }

  static void AQInputCallback(void* inUserData, AudioQueueRef inQueue, AudioQueueBufferRef inBuffer,
    const AudioTimeStamp* /*inStartTime*/, UInt32 inNumberPackets,
    const AudioStreamPacketDescription* /*inPacketDesc*/) {
    AppleAudioCaptureInternalData* data = reinterpret_cast<AppleAudioCaptureInternalData*>(inUserData);
    TTV_ASSERT(data != nullptr);

    if (data == nullptr) {
      return;
    }

    AutoMutex lock(data->captureMutex.get());

    TTV_ASSERT(data->captureBufferWritePos < data->captureBuffer.size());
    unsigned int samplesToWrite = inNumberPackets * kNumAudioChannels;
    unsigned int preWrapSamplesToWrite =
      std::min(samplesToWrite, static_cast<unsigned int>(data->captureBuffer.size() - data->captureBufferWritePos));
    TTV_ASSERT(samplesToWrite >= preWrapSamplesToWrite);
    unsigned int postWrapSamplesToWrite =
      samplesToWrite <= preWrapSamplesToWrite ? 0 : samplesToWrite - preWrapSamplesToWrite;
    TTV_ASSERT(postWrapSamplesToWrite <= data->captureBuffer.size());

    memcpy(&data->captureBuffer.data()[data->captureBufferWritePos], inBuffer->mAudioData,
      preWrapSamplesToWrite * sizeof(int16_t));
    if (postWrapSamplesToWrite > 0) {
      memcpy(data->captureBuffer.data(), reinterpret_cast<int16_t*>(inBuffer->mAudioData) + preWrapSamplesToWrite,
        postWrapSamplesToWrite * sizeof(int16_t));
    }
    data->captureBufferWritePos += preWrapSamplesToWrite;
    data->captureBufferWritePos %= data->captureBuffer.size();
    data->captureBufferWritePos += postWrapSamplesToWrite;

    OSStatus err = AudioQueueEnqueueBuffer(inQueue, inBuffer, 0, nullptr);

    // We get kAudioQueueErr_EnqueueDuringReset when we call Enqueue while we're shutting down and QueueStop has been
    // called
    if (err != noErr && err != kAudioQueueErr_EnqueueDuringReset) {
      ttv::trace::Message("AppleAudioCapture", MessageLevel::Error,
        "AppleAudioCapture::AQInputCallback - AudioQueueEnqueueBuffer failed with %d", err);
    }
  }

  void Stop() {
    if (audioQueue != nullptr) {
      AudioQueueStop(audioQueue, true);
      AudioQueueDispose(audioQueue, true);
      audioQueue = nullptr;
    }

    receiver.reset();
  }

 public:
  std::shared_ptr<IPcmAudioFrameReceiver> receiver;  // The receiver to use to send the audio data to the mixer.
  AudioQueueRef audioQueue;
  std::vector<int16_t> captureBuffer;
  uint32_t captureBufferWritePos;
  uint32_t captureBufferReadPos;
  std::unique_ptr<IMutex> captureMutex;
  uint64_t sampleTime;
  AppleAudioCapture::CaptureType type;
  TTV_ErrorCode initializationError;
};

ttv::broadcast::AppleAudioCapture::AppleAudioCapture(CaptureType type)
    : mInternalData(std::make_unique<AppleAudioCaptureInternalData>(type)) {
  const unsigned int kCaptureBufferDuration = 3;
  mInternalData->captureBuffer.resize(kCaptureBufferDuration * kAudioEncodeRate * kNumAudioChannels);

  TTV_ErrorCode ret = ttv::CreateMutex(mInternalData->captureMutex, "AppleAudioCapture");
  ASSERT_ON_ERROR(ret);

  ttv::trace::Message("AppleAudioCapture", MessageLevel::Info, "AppleAudioCapture created");
}

ttv::broadcast::AppleAudioCapture::~AppleAudioCapture() {
  ttv::trace::Message("AppleAudioCapture", MessageLevel::Info, "AppleAudioCapture destroyed");
}

std::string ttv::broadcast::AppleAudioCapture::GetName() const {
  if (mInternalData->type == CaptureType::System) {
    return "AppleAudioCapture - System";
  } else {
    return "AppleAudioCapture - Microphone";
  }
}

uint32_t ttv::broadcast::AppleAudioCapture::GetNumChannels() const {
  return kNumAudioChannels;
}

TTV_ErrorCode ttv::broadcast::AppleAudioCapture::Start() {
  TTV_ErrorCode ret = AudioCaptureBase::Start();

  if (TTV_SUCCEEDED(ret)) {
#if !TTV_TARGET_IOS
    AudioDeviceID device = 0;
    switch (mInternalData->type) {
      case CaptureType::System:
        device = GetSoundflowerDevice();
        if (device == 0) {
          ttv::trace::Message(
            "AppleAudioCapture", MessageLevel::Warning, "AppleAudioCapture::Start - SoundFlower not installed");
          return TTV_EC_BROADCAST_SOUNDFLOWER_NOT_INSTALLED;
        }
        break;
      case CaptureType::Microphone:
        device = GetDefaultInputDevice();
        break;
      default:
        TTV_ASSERT(false);
        break;
    }

    if (device == 0) {
      ttv::trace::Message(
        "AppleAudioCapture", MessageLevel::Warning, "AppleAudioCapture::Start - Audio device not found");
      return TTV_EC_BROADCAST_AUDIO_DEVICE_INIT_FAILED;
    }
#endif  // !TTV_TARGET_IOS

    // Set up the recording format
    //
    AudioStreamBasicDescription recordFormat;
    memset(&recordFormat, 0, sizeof(recordFormat));
    recordFormat.mFormatID = kAudioFormatLinearPCM;
    recordFormat.mChannelsPerFrame = kNumAudioChannels;
    recordFormat.mFormatFlags = kLinearPCMFormatFlagIsPacked | kLinearPCMFormatFlagIsSignedInteger;
    recordFormat.mFramesPerPacket = 1;
    recordFormat.mBitsPerChannel = sizeof(int16_t) * 8;
    recordFormat.mBytesPerFrame = recordFormat.mChannelsPerFrame * recordFormat.mBitsPerChannel / 8;
    recordFormat.mBytesPerPacket = recordFormat.mBytesPerFrame * recordFormat.mFramesPerPacket;
    recordFormat.mSampleRate = kAudioEncodeRate;

    // Set up a recording queue
    //
    OSStatus err = AudioQueueNewInput(&recordFormat, AppleAudioCaptureInternalData::AQInputCallback,
      static_cast<void*>(mInternalData.get()), nullptr, nullptr, 0, &mInternalData->audioQueue);

    if (err != noErr) {
      ttv::trace::Message(
        "AppleAudioCapture", MessageLevel::Error, "AppleAudioCapture::Start - AudioQueueNewInput failed with %d", err);
    }

#if !TTV_TARGET_IOS
    // TODO - On iOS we currently only capture the default input device (i.e. microphone)
    // therefore we don't need to set the device property of the queue. On Mac we need
    // to set it since it may not be the default input device. Also note that the device ID
    // stuff is not available on iOS, it seems
    //
    if (err == noErr) {
      CFStringRef deviceUID = GetDeviceUID(device);
      TTV_ASSERT(deviceUID);
      err = AudioQueueSetProperty(
        mInternalData->audioQueue, kAudioQueueProperty_CurrentDevice, &deviceUID, sizeof(deviceUID));
      TTV_ASSERT(err == noErr);

      if (err != noErr) {
        ttv::trace::Message("AppleAudioCapture", MessageLevel::Error,
          "AppleAudioCapture::Start - AudioQueueSetProperty failed with %d", err);
      }
    }
#endif

    if (err != noErr) {
      ret = TTV_EC_BROADCAST_MAC_INPUT_Q_SETUP_FAILED;
    } else {
      // Set up buffers for the queue
      //
      const unsigned int kBufferTimeInSec = 1;
      const unsigned int kBufferSize =
        static_cast<unsigned int>(kBufferTimeInSec * recordFormat.mSampleRate * recordFormat.mChannelsPerFrame *
                                  recordFormat.mBitsPerChannel / 8);

      const unsigned int kNumRecordBuffers = 3;
      for (unsigned int i = 0; i < kNumRecordBuffers && err == noErr; ++i) {
        AudioQueueBufferRef buffer;
        err = AudioQueueAllocateBuffer(mInternalData->audioQueue, kBufferSize, &buffer);
        if (err != noErr) {
          ttv::trace::Message("AppleAudioCapture", MessageLevel::Error,
            "AppleAudioCapture::Start - AudioQueueAllocateBuffer failed with %d", err);
        } else {
          err = AudioQueueEnqueueBuffer(mInternalData->audioQueue, buffer, 0, nullptr);

          if (err != noErr) {
            ttv::trace::Message("AppleAudioCapture", MessageLevel::Error,
              "AppleAudioCapture::Start - AudioQueueEnqueueBuffer failed with %d", err);
          }
        }

        if (err != noErr) {
          ret = TTV_EC_BROADCAST_MAC_INPUT_Q_BUFFER_SETUP_FAILED;
          break;
        }
      }
    }

    mInternalData->initializationError = ret;
  }

  return ret;
}

TTV_ErrorCode ttv::broadcast::AppleAudioCapture::Stop() {
  TTV_ErrorCode ec = AudioCaptureBase::Stop();

  if (TTV_SUCCEEDED(ec)) {
    mInternalData->Stop();
  }

  return ec;
}

void ttv::broadcast::AppleAudioCapture::SetInitialTime(uint64_t initialTime) {
  AudioCaptureBase::SetInitialTime(initialTime);

  if (TTV_SUCCEEDED(mInternalData->initializationError)) {
    // Start the queue
    OSStatus err = AudioQueueStart(mInternalData->audioQueue, nullptr);
    if (err != noErr) {
      ttv::trace::Message("AppleAudioCapture", MessageLevel::Error,
        "AppleAudioCapture::SetInitialTime - AudioQueueStart failed with %d", err);

      mInternalData->initializationError = TTV_EC_BROADCAST_MAC_INPUT_Q_START_FAILED;
    }
  }
}

TTV_ErrorCode ttv::broadcast::AppleAudioCapture::DoSubmitFrame(uint32_t samplesPerChannel) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (samplesPerChannel > 0) {
    auto& data = *mInternalData;

    const int16_t* samples = &data.captureBuffer[data.captureBufferReadPos];

    std::shared_ptr<AudioFrame> audioFrame;
    ec = data.receiver->PackageFrame(reinterpret_cast<const uint8_t*>(samples), samplesPerChannel, kNumAudioChannels,
      kPcmInterleaved, AudioSampleFormat::TTV_ASF_PCM_S16, data.sampleTime, audioFrame);

    if (TTV_SUCCEEDED(ec) && audioFrame != nullptr) {
      ec = mAudioMixer->SubmitFrame(mAudioLayer, audioFrame);

      data.sampleTime += samplesPerChannel;
      data.captureBufferReadPos += samplesPerChannel * kNumAudioChannels;
      data.captureBufferReadPos %= data.captureBuffer.size();
    }
  }

  return ec;
}

TTV_ErrorCode ttv::broadcast::AppleAudioCapture::Process(
  const std::shared_ptr<IAudioMixer>& mixer, uint64_t& lastSampleTime) {
  TTV_ASSERT(mixer != nullptr);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto& data = *mInternalData;

  AutoMutex lock(data.captureMutex.get());

  if (TTV_FAILED(data.initializationError)) {
    return data.initializationError;
  }

  // Note: The mixer expects "samples per channel" whereas when reading from our capture buffers we
  // deal with total samples (for all channels)
  //

  if (data.receiver == nullptr) {
    data.receiver = std::static_pointer_cast<IPcmAudioFrameReceiver>(
      mixer->GetReceiverImplementation(IPcmAudioFrameReceiver::GetReceiverTypeId()));
  }

  TTV_ASSERT(data.captureBufferReadPos < data.captureBuffer.size());

  // One contiguous buffer of samples
  if (data.captureBufferReadPos <= data.captureBufferWritePos) {
    uint32_t samplesToRead = data.captureBufferWritePos - data.captureBufferReadPos;
    if (samplesToRead > 0) {
      uint32_t samplesPerChannel = static_cast<uint32_t>(samplesToRead) / kNumAudioChannels;
      ec = DoSubmitFrame(samplesPerChannel);
    }
  }
  // Buffer wrapped around
  else {
    // Write the buffer from the read position to the end and then from the beginning to the write position
    //
    uint32_t samplesToRead = static_cast<uint32_t>(data.captureBuffer.size()) - data.captureBufferReadPos;
    uint32_t samplesPerChannel = samplesToRead / kNumAudioChannels;

    ec = DoSubmitFrame(samplesPerChannel);

    if (TTV_SUCCEEDED(ec)) {
      samplesPerChannel = data.captureBufferWritePos / kNumAudioChannels;

      ec = DoSubmitFrame(samplesPerChannel);
    }
  }

  lastSampleTime = data.sampleTime;

  return ec;
}
