/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

///////////////////////////////
//
// Some sample code for using VTCompressionSession:
//  https://github.com/galad87/HandBrake-QuickSync-Mac/blob/master/libhb/platform/macosx/encvt_h264.c
//  https://github.com/jgh-/VideoCore/blob/master/transforms/Apple/H264Encode.mm
//
///////////////////////////////

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/applevideoencoder.h"

#include "twitchsdk/broadcast/icvpixelbuffervideoframereceiver.h"
#include "twitchsdk/broadcast/iframewriter.h"
#include "twitchsdk/broadcast/irawvideoframereceiver.h"
#include "twitchsdk/broadcast/packet.h"
#include "twitchsdk/broadcast/videoframe.h"
#include "twitchsdk/core/systemclock.h"

#include <inttypes.h>

#if TTV_PLATFORM_DARWIN && _x86_64_
#include "cpuid.h"

#include <tmmintrin.h>
#endif

#define USE_ACCELERATE_FRAMEWORK 1

#if USE_ACCELERATE_FRAMEWORK
#include <Accelerate/Accelerate.h>
#endif

namespace {
using namespace ttv::broadcast;

const uint32_t kKeyFrameIntervalInSeconds = 2;
const OSType kSafeNativePixelFormat = kCVPixelFormatType_32BGRA;

void SwizzleToBGRA(const uint8_t* pixelData, uint32_t pixelWidth, uint32_t pixelHeight, int32_t sourcePitch,
  int32_t outputPitch, uint32_t bgraMask, uint8_t* output) {
  TTV_ASSERT(pixelData != nullptr);
  TTV_ASSERT(output != nullptr);

#if _x86_64_

  // Since we use the _mm_shuffle_epi8 intrinsic, we need to check if the CPU supports the SSSE3 (Supplemental SSE3)
  // instructions. If it does not support SSSE3 we use the slow swizzle. TODO: See if we can use SSE with other
  // intrinsics that don't need SSSE3.
  //
  unsigned int cpuInfo[4];
  __get_cpuid(1, &cpuInfo[0], &cpuInfo[1], &cpuInfo[2], &cpuInfo[3]);
  bool ssse3Enabled = (cpuInfo[2] & 0x200) || false;

  if (ssse3Enabled) {
    const __m128i* pixelData_128 = reinterpret_cast<const __m128i*>(
      pixelData);  // lint !e826 Suspicious pointer-to-pointer conversion (area too small)
    __m128i* output_128 =
      reinterpret_cast<__m128i*>(output);  // lint !e826 Suspicious pointer-to-pointer conversion (area too small)

    const int8_t* const swizzleMask = reinterpret_cast<int8_t*>(&bgraMask);

    // convert the pitches from bytes to 128 bit blocks
    sourcePitch /= sizeof(__m128i);
    outputPitch /= sizeof(__m128i);

    // The mask is based off of BGRA (0x00010203), so for example, RBGA would be 0x02010003. Note that since the mask is
    // a little-endian 32-bit integer the 4 bytes of it are in reverse order. We use each byte of the mask to shuffle
    // the R,G,B values into the right position for each of the 4 pixels in each __m128i
    //
    // lint -save
    // lint -e911 Implicit expression promotion from signed char to int
    // lint -e917 Prototype coercion (___) ___ to ___
    const __m128i bgraSwizzle = _mm_setr_epi8(swizzleMask[3], swizzleMask[2], swizzleMask[1], swizzleMask[0],
      swizzleMask[3] + 4, swizzleMask[2] + 4, swizzleMask[1] + 4, swizzleMask[0] + 4, swizzleMask[3] + 8,
      swizzleMask[2] + 8, swizzleMask[1] + 8, swizzleMask[0] + 8, swizzleMask[3] + 12, swizzleMask[2] + 12,
      swizzleMask[1] + 12, swizzleMask[0] + 12);
    // lint -restore

    const uint32_t sourceElementsPerLine = pixelWidth / sizeof(uint32_t);

    for (uint32_t y = 0; y < pixelHeight; ++y) {
      for (uint32_t i = 0; i < sourceElementsPerLine; ++i) {
        output_128[i] = _mm_shuffle_epi8(pixelData_128[i], bgraSwizzle);
      }

      pixelData_128 += sourcePitch;
      output_128 += outputPitch;
    }
  } else
#endif
  {
#if USE_ACCELERATE_FRAMEWORK

    // package the parameters
    vImage_Buffer srcBuffer = {const_cast<void*>(reinterpret_cast<const void*>(pixelData)),
      static_cast<vImagePixelCount>(pixelHeight), static_cast<vImagePixelCount>(pixelWidth),
      static_cast<size_t>(sourcePitch)};
    vImage_Buffer destBuffer = {output, srcBuffer.height, srcBuffer.width, static_cast<size_t>(outputPitch)};

    uint8_t mask[4];
    mask[0] = reinterpret_cast<uint8_t*>(&bgraMask)[3];
    mask[1] = reinterpret_cast<uint8_t*>(&bgraMask)[2];
    mask[2] = reinterpret_cast<uint8_t*>(&bgraMask)[1];
    mask[3] = reinterpret_cast<uint8_t*>(&bgraMask)[0];

    // try using the Accelerate framework
    vImage_Error ret = vImagePermuteChannels_ARGB8888(&srcBuffer, &destBuffer, mask, kvImageNoFlags);

    // failed for some reason so default to the slow version
    if (ret != kvImageNoError)
#endif
    {
      uint8_t* maskIndexes = reinterpret_cast<uint8_t*>(&bgraMask);

      const uint8_t* sourceRowStart = pixelData;
      uint8_t* outputRowStart = output;

      for (uint32_t y = 0; y < pixelHeight; ++y) {
        const uint32_t* src = reinterpret_cast<const uint32_t*>(sourceRowStart);
        uint8_t* dest = outputRowStart;

        for (uint32_t x = 0; x < pixelWidth; ++x) {
          // copy the source since the source and dest buffers might be the same
          uint32_t copy = *src;
          uint8_t* t = reinterpret_cast<uint8_t*>(&copy);

          // Note that since the mask is little-endian we need to read the position of each byte in reverse order
          dest[0] = t[maskIndexes[3]];
          dest[1] = t[maskIndexes[2]];
          dest[2] = t[maskIndexes[1]];
          dest[3] = t[maskIndexes[0]];

          // advance the pixel pointers
          src++;
          dest += sizeof(uint32_t);
        }

        // advance the row pointers
        sourceRowStart += sourcePitch;
        outputRowStart += outputPitch;
      }
    }
  }
}

bool IsKeyFrame(const CMSampleBufferRef sampleBuffer) {
  bool isKeyframe = false;
  CFArrayRef attachments = CMSampleBufferGetSampleAttachmentsArray(sampleBuffer, false);
  if (attachments != nullptr) {
    CFDictionaryRef attachment;
    CFBooleanRef dependsOnOthers;
    attachment = static_cast<CFDictionaryRef>(CFArrayGetValueAtIndex(attachments, 0));
    dependsOnOthers =
      static_cast<CFBooleanRef>(CFDictionaryGetValue(attachment, kCMSampleAttachmentKey_DependsOnOthers));
    isKeyframe = !CFBooleanGetValue(dependsOnOthers);
  }

  return isKeyframe;
}

void PixelBufferReleaseCallback(void* /*releaseRefCon*/, const void* baseAddress) {
  delete[] static_cast<uint8_t*>(const_cast<void*>(baseAddress));
}

OSType SdkToNativePixelFormat(PixelFormat pixelFormat) {
  switch (pixelFormat) {
    case PixelFormat::TTV_PF_BGRA:
      return kCVPixelFormatType_32BGRA;
    case PixelFormat::TTV_PF_ABGR:
      return kCVPixelFormatType_32ABGR;
    case PixelFormat::TTV_PF_RGBA:
      return kCVPixelFormatType_32RGBA;
    case PixelFormat::TTV_PF_ARGB:
      return kCVPixelFormatType_32ARGB;
    default:
      //#pragma clang diagnostic push
      //#pragma clang diagnostic ignored "-Wunreachable-code"
      TTV_ASSERT(false);
      return kCVPixelFormatType_32BGRA;
      //#pragma clang diagnostic push
  }
}

/**
 * Returns true if the hardware natively supports the given format, false if a swizzle to kCVPixelFormatType_32BGRA is
 * required before loading into a CVPixelBuffer.
 */
bool CheckNativePixelFormatSupport(OSType pixelFormat) {
  // check and see if we can create a buffer of the given format
  const int tempBufferDim = 16;
  std::vector<uint8_t> buffer;
  buffer.resize(tempBufferDim * tempBufferDim * 4);

  CVPixelBufferRef pixelBufferRef = nullptr;
  CVReturn cvRet = CVPixelBufferCreateWithBytes(kCFAllocatorDefault, tempBufferDim, tempBufferDim, pixelFormat,
    buffer.data(), tempBufferDim * 4, nullptr, nullptr, nullptr, &pixelBufferRef);

  // native support for the buffer
  if (cvRet == kCVReturnSuccess) {
    if (pixelBufferRef != nullptr) {
      CFRelease(pixelBufferRef);
      return true;
    }

    TTV_ASSERT(false);
  }

  // need to swizzle to BGRA
  return false;
}

/**
 * The video frame required to pack raw buffers.
 */
class RawVideoFrame : public VideoFrame {
 public:
  RawVideoFrame(const uint8_t* frameBuffer, PixelFormat pixelFormat, bool verticalFlip, uint64_t timeStamp,
    IRawVideoFrameReceiver::UnlockFunc unlockCallback)
      : VideoFrame(IRawVideoFrameReceiver::GetReceiverTypeId()), mFrameBuffer(frameBuffer), mPixelFormat(pixelFormat) {
    SetVerticalFlip(verticalFlip);
    SetTimeStamp(timeStamp);

    SetUnlockCallback([unlockCallback, frameBuffer]() { unlockCallback(frameBuffer); });
  }

  const uint8_t* GetFrameBuffer() const { return mFrameBuffer; }
  PixelFormat GetPixelFormat() const { return mPixelFormat; }

 private:
  const uint8_t* mFrameBuffer;
  PixelFormat mPixelFormat;
};

class RawReceiver : public IRawVideoFrameReceiver {
 public:
  virtual TTV_ErrorCode PackageFrame(const uint8_t* frameBuffer, PixelFormat pixelFormat, bool verticalFlip,
    uint64_t timeStamp, UnlockFunc unlockCallback, std::shared_ptr<VideoFrame>& result) override {
    result = std::make_shared<RawVideoFrame>(frameBuffer, pixelFormat, verticalFlip, timeStamp, unlockCallback);

    return TTV_EC_SUCCESS;
  }
};

/**
 * The video frame required to pack CVPixelBuffers.
 */
class CVPixelBufferFrame : public VideoFrame {
 public:
  CVPixelBufferFrame(
    const CVPixelBufferRef buffer, uint64_t timeStamp, ICVPixelBufferVideoFrameReceiver::UnlockFunc unlockCallback)
      : VideoFrame(ICVPixelBufferVideoFrameReceiver::GetReceiverTypeId()), mCVPixelBuffer(buffer) {
    CFRetain(buffer);

    SetTimeStamp(timeStamp);

    if (unlockCallback != nullptr) {
      SetUnlockCallback([unlockCallback, buffer]() { unlockCallback(buffer); });
    }
  }

  virtual ~CVPixelBufferFrame() {
    if (mCVPixelBuffer != nullptr) {
      CFRelease(mCVPixelBuffer);
    }
  }

  CVPixelBufferRef GetCVPixelBuffer() const { return mCVPixelBuffer; }

 private:
  const CVPixelBufferRef mCVPixelBuffer;
};

class CVPixelBufferReceiver : public ICVPixelBufferVideoFrameReceiver {
 public:
  virtual TTV_ErrorCode PackageFrame(const CVPixelBufferRef buffer, uint64_t timestamp, UnlockFunc unlockCallback,
    std::shared_ptr<VideoFrame>& result) override {
    result = std::make_shared<CVPixelBufferFrame>(buffer, timestamp, unlockCallback);

    return TTV_EC_SUCCESS;
  }
};
}  // namespace

class ttv::broadcast::AppleVideoEncoderInternalData {
 public:
  AppleVideoEncoderInternalData(bool enableBFrames)
      : compressionSession(nullptr),
        streamIndex(0),
        currentBitRateKbps(0),
        requestedBitRateKbps(0),
        encoderError(TTV_EC_SUCCESS),
        enableBFrames(enableBFrames),
        scratchPixelBufferRef(nullptr) {}

  ~AppleVideoEncoderInternalData() { DestroyScratchCVPixelBuffer(); }

  OSStatus ApplyBitRate(uint32_t kbps) {
    OSStatus err = noErr;

    if (kbps != currentBitRateKbps) {
      currentBitRateKbps = kbps;

      // Set the average bit rate in bits per second
      const int32_t bitsPerSecond = static_cast<int32_t>(kbps * 1000);
      CFNumberRef num = CFNumberCreate(nullptr, kCFNumberSInt32Type, &bitsPerSecond);
      err = VTSessionSetProperty(compressionSession, kVTCompressionPropertyKey_AverageBitRate, num);
      CFRelease(num);

      // Set the data rate limit in bytes per second
      if (err == noErr) {
        if (!enableBFrames) {
          const int32_t bytesPerSecond = bitsPerSecond / 8;
          const int32_t one = 1;
          CFNumberRef numBytes = CFNumberCreate(nullptr, kCFNumberSInt32Type, &bytesPerSecond);
          CFNumberRef numSeconds = CFNumberCreate(nullptr, kCFNumberSInt32Type, &one);

          CFTypeRef arr[2] = {numBytes, numSeconds};

          CFArrayRef prop = CFArrayCreate(nullptr, reinterpret_cast<const void**>(arr), 2, &kCFTypeArrayCallBacks);
          err = VTSessionSetProperty(compressionSession, kVTCompressionPropertyKey_DataRateLimits, prop);

          CFRelease(prop);
          CFRelease(numBytes);
          CFRelease(numSeconds);
        }

        if (err != noErr) {
          ttv::trace::Message(
            "AppleVideoEncoder", MessageLevel::Error, "AppleVideoEncoder::ApplyBitRate - Error setting data rate");

          return TTV_EC_BROADCAST_VIDEO_ENCODER_INIT_FAILED;
        }
      } else {
        ttv::trace::Message("AppleVideoEncoder", MessageLevel::Error,
          "AppleVideoEncoder::ApplyBitRate - Error setting bit rate to %" PRId32 " kbps", kbps);

        return TTV_EC_BROADCAST_VIDEO_ENCODER_INIT_FAILED;
      }

      VTCompressionSessionCompleteFrames(compressionSession, kCMTimeInvalid);

      ttv::trace::Message("AppleVideoEncoder", MessageLevel::Debug,
        "AppleVideoEncoder::ApplyBitRate - Set bit rate to %" PRId32 " kbps", kbps);
    }

    return TTV_EC_SUCCESS;
  }

  /**
   * If not already created, creates a buffer of format kCVPixelFormatType_32BGRA with the same dimensions of the video
   * frames.
   */
  CVPixelBufferRef GetScratchCVPixelBuffer() {
    if (scratchPixelBufferRef != nullptr) {
      return scratchPixelBufferRef;
    }

    CVReturn cvRet = CVPixelBufferCreate(kCFAllocatorDefault, videoParams.outputWidth, videoParams.outputHeight,
      kSafeNativePixelFormat, nullptr, &scratchPixelBufferRef);
    if (cvRet == kCVReturnSuccess) {
      return scratchPixelBufferRef;
    } else {
      TTV_ASSERT(scratchPixelBufferRef == nullptr);
      return nullptr;
    }
  }

  void DestroyScratchCVPixelBuffer() {
    if (scratchPixelBufferRef != nullptr) {
      CFRelease(scratchPixelBufferRef);
      scratchPixelBufferRef = nullptr;
    }
  }

 public:
  std::shared_ptr<IFrameWriter> frameWriter;
  VideoParams videoParams;

  VTCompressionSessionRef compressionSession;
  std::vector<uint8_t> sps;
  std::vector<uint8_t> pps;

  std::shared_ptr<CVPixelBufferReceiver> cvPixelBufferReceiver;
  std::shared_ptr<RawReceiver> rawReceiver;
  std::map<OSType, bool>
    pixelFormatSupport;  //!< Map of pixel format to whether or not there is native CVPixelBuffer support for it.

  uint32_t streamIndex;
  uint32_t currentBitRateKbps;
  uint32_t requestedBitRateKbps;

  TTV_ErrorCode encoderError;  //!< The error reported by the native encoder asynchronously.

 private:
  bool enableBFrames;
  CVPixelBufferRef scratchPixelBufferRef;  //!< The pixel buffer to use for flipping and swizzling.
};

ttv::broadcast::AppleVideoEncoder::AppleVideoEncoder(bool enableBFrames) {
  ttv::trace::Message("AppleVideoEncoder", MessageLevel::Info, "AppleVideoEncoder created");

  mEnableBFrames = enableBFrames;
  mInternalData = std::make_unique<AppleVideoEncoderInternalData>(enableBFrames);
}

ttv::broadcast::AppleVideoEncoder::~AppleVideoEncoder() {
  InternalStop();

  ttv::trace::Message("AppleVideoEncoder", MessageLevel::Info, "AppleVideoEncoder destroyed");
}

std::string ttv::broadcast::AppleVideoEncoder::GetName() const {
  return "AppleVideoEncoder";
}

bool ttv::broadcast::AppleVideoEncoder::SupportsBitRateAdjustment() const {
  return true;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::SetFrameWriter(const std::shared_ptr<IFrameWriter>& frameWriter) {
  // TODO: Make sure not started
  mInternalData->frameWriter = frameWriter;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::ValidateVideoParams(const VideoParams& /*videoParams*/) const {
  // No special requirements

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::Initialize() {
  AppleVideoEncoderInternalData& data = *mInternalData;

  // Determine which pixel formats are supported natively
  data.pixelFormatSupport[kCVPixelFormatType_32BGRA] = CheckNativePixelFormatSupport(kCVPixelFormatType_32BGRA);
  data.pixelFormatSupport[kCVPixelFormatType_32ABGR] = CheckNativePixelFormatSupport(kCVPixelFormatType_32ABGR);
  data.pixelFormatSupport[kCVPixelFormatType_32RGBA] = CheckNativePixelFormatSupport(kCVPixelFormatType_32RGBA);
  data.pixelFormatSupport[kCVPixelFormatType_32ARGB] = CheckNativePixelFormatSupport(kCVPixelFormatType_32ARGB);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::Shutdown() {
  return Stop();
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::Start(uint32_t streamIndex, const VideoParams& videoParams) {
  AppleVideoEncoderInternalData& data = *mInternalData;

  if (data.frameWriter == nullptr) {
    ttv::trace::Message(
      "AppleVideoEncoder", MessageLevel::Error, "Inside AppleVideoEncoder::Start - frameWriter pointer is null");
    TTV_ASSERT(false && "frameWriter != nullptr");
    return TTV_EC_INVALID_ARG;
  }

  data.streamIndex = streamIndex;
  data.videoParams = videoParams;

  OSStatus err = SetupMagicCookie();

  if (err == noErr) {
    err = SetupVTCompressionSession(&FrameEncodedCallback);
  }

  TTV_ErrorCode ret = (err == noErr) ? TTV_EC_SUCCESS : TTV_EC_BROADCAST_VIDEO_ENCODER_INIT_FAILED;
  data.encoderError = ret;

  return ret;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::Stop() {
  InternalStop();

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::SubmitFrame(const std::shared_ptr<VideoFrame>& videoFrame) {
  AppleVideoEncoderInternalData& data = *mInternalData;

  // Report an async encoder error
  if (TTV_FAILED(data.encoderError)) {
    return data.encoderError;
  }

  data.ApplyBitRate(data.requestedBitRateKbps);

  CVReturn cvRet = kCVReturnSuccess;
  CVPixelBufferRef pixelBufferRef = nullptr;
  bool needToReleasePixelBufferRef = false;

  // Received raw input buffer
  if (videoFrame->GetReceiverTypeId() == IRawVideoFrameReceiver::GetReceiverTypeId()) {
    auto rawFrame = std::static_pointer_cast<RawVideoFrame>(videoFrame);

    OSType format = SdkToNativePixelFormat(rawFrame->GetPixelFormat());

    // NOTE: CVPixelBufferCreateWithBytes wraps the given buffer and does not create a copy

    bool nativeSupport = data.pixelFormatSupport[format];

    // We need to manually process the input buffer
    if (!nativeSupport || videoFrame->GetVerticalFlip()) {
      pixelBufferRef = data.GetScratchCVPixelBuffer();
      if (pixelBufferRef == nullptr) {
        return TTV_EC_BROADCAST_BUFFER_ALLOCATION_FAILED;
      }

      cvRet = CVPixelBufferLockBaseAddress(pixelBufferRef, 0);
      TTV_ASSERT(cvRet == kCVReturnSuccess);

      if (cvRet == kCVReturnSuccess) {
        uint8_t* dst = static_cast<uint8_t*>(CVPixelBufferGetBaseAddress(pixelBufferRef));
        TTV_ASSERT(dst != nullptr);

        const uint8_t* src = static_cast<const uint8_t*>(rawFrame->GetFrameBuffer());
        int32_t sourcePitch = static_cast<int32_t>(data.videoParams.outputWidth * 4);

        // setup the pitch to be negative so the source is scanned in reverse
        if (videoFrame->GetVerticalFlip()) {
          sourcePitch = -sourcePitch;
          src = src + data.videoParams.outputWidth * (data.videoParams.outputHeight - 1) * 4;
        }

        SwizzleToBGRA(src, data.videoParams.outputWidth, data.videoParams.outputHeight, sourcePitch,
          static_cast<int32_t>(data.videoParams.outputWidth * 4), static_cast<uint32_t>(rawFrame->GetPixelFormat()),
          dst);

        CVPixelBufferUnlockBaseAddress(pixelBufferRef, 0);
      }
    }
    // Just wrap the given input buffer with a CVPixelBuffer since we have native support
    else {
      void* unsafePointerToBuffer = const_cast<void*>(reinterpret_cast<const void*>(rawFrame->GetFrameBuffer()));
      cvRet =
        CVPixelBufferCreateWithBytes(kCFAllocatorDefault, data.videoParams.outputWidth, data.videoParams.outputHeight,
          format, unsafePointerToBuffer, data.videoParams.outputWidth * 4, nullptr, nullptr, nullptr, &pixelBufferRef);
      TTV_ASSERT(cvRet == kCVReturnSuccess);
      needToReleasePixelBufferRef = true;
    }
  }
  // Received a CVPixelBuffer
  else if (videoFrame->GetReceiverTypeId() == ICVPixelBufferVideoFrameReceiver::GetReceiverTypeId()) {
    auto cvpFrame = std::static_pointer_cast<CVPixelBufferFrame>(videoFrame);
    pixelBufferRef = cvpFrame->GetCVPixelBuffer();
  }
  // Unsupported video frame type
  else {
    TTV_ASSERT(false);
    return TTV_EC_BROADCAST_INVALID_SUBMISSION_METHOD;
  }

  TTV_ASSERT(pixelBufferRef != nullptr);
  TTV_ASSERT(data.compressionSession != nullptr);

  CMTime pts =
    CMTimeMake(static_cast<int64_t>(videoFrame->GetTimeStamp()), static_cast<int32_t>(GetSystemClockFrequency()));

  OSStatus err = VTCompressionSessionEncodeFrame(
    data.compressionSession, pixelBufferRef, pts, kCMTimeInvalid, nullptr, nullptr, nullptr);

  if (pixelBufferRef != nullptr && needToReleasePixelBufferRef) {
    CVPixelBufferRelease(pixelBufferRef);
  }

  if (err == noErr) {
    return TTV_EC_SUCCESS;
  } else {
    ttv::trace::Message("AppleVideoEncoder", MessageLevel::Error,
      "AppleVideoEncoder::SubmitFrame - VTCompressionSessionEncodeFrame failed with error code %d", err);

    return TTV_EC_BROADCAST_VIDEO_FRAME_SUBMISSION_FAILED;
  }
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::GetSpsPps(std::vector<uint8_t>& sps, std::vector<uint8_t>& pps) {
  AppleVideoEncoderInternalData& data = *mInternalData;

  TTV_ASSERT((data.sps.size() == 0 && data.pps.size() == 0) || (data.sps.size() > 0 && data.pps.size() > 0));

  TTV_ErrorCode ret = TTV_EC_BROADCAST_NO_SPSPPS;
  if (data.sps.size() > 0 && data.pps.size() > 0) {
    sps = data.sps;
    pps = data.pps;
    ret = TTV_EC_SUCCESS;
  }

  return ret;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::ValidateFrame(const std::shared_ptr<VideoFrame>& /*videoframe*/) {
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::AppleVideoEncoder::SetTargetBitRate(uint32_t kbps) {
  if (mInternalData == nullptr || mInternalData->compressionSession == nullptr) {
    return TTV_EC_INVALID_STATE;
  }

  AppleVideoEncoderInternalData& data = *mInternalData;

  data.requestedBitRateKbps = kbps;

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::AppleVideoEncoder::WriteEncodedFrame(std::unique_ptr<Packet>&& packet) {
  AppleVideoEncoderInternalData& data = *mInternalData;

  packet->streamIndex = data.streamIndex;
  data.frameWriter->WritePacket(std::move(packet));
}

void ttv::broadcast::AppleVideoEncoder::InternalStop() {
  AppleVideoEncoderInternalData& data = *mInternalData;

  if (data.compressionSession) {
    // Flush any frames currently being processed by the encoder
    VTCompressionSessionCompleteFrames(data.compressionSession, kCMTimeInvalid);

    VTCompressionSessionInvalidate(data.compressionSession);
    CFRelease(data.compressionSession);
    data.compressionSession = nullptr;
  }

  data.DestroyScratchCVPixelBuffer();

  data.frameWriter = nullptr;
  data.encoderError = TTV_EC_SUCCESS;
}

OSStatus ttv::broadcast::AppleVideoEncoder::SetupVTCompressionSession(
  VTCompressionOutputCallback encodedFrameCallback) {
  AppleVideoEncoderInternalData& data = *mInternalData;

  TTV_ASSERT(data.compressionSession == nullptr);

  CFMutableDictionaryRef encoderSpecs = nullptr;

#ifndef TTV_TARGET_IOS
  // On iOS devices hardware accelaration is always used, so it's not necessary to set these parameters.
  //
  const int kEncoderSpecKeyCount = 3;
  CFStringRef encSpecKeys[kEncoderSpecKeyCount];
  CFTypeRef encSpecValues[kEncoderSpecKeyCount];

  encSpecKeys[0] = kVTVideoEncoderSpecification_EncoderID;
  encSpecValues[0] = CFSTR("com.apple.videotoolbox.videoencoder.h264.gva");

  encSpecKeys[1] = CFSTR("EnableHardwareAcceleratedVideoEncoder");
  encSpecValues[1] = kCFBooleanTrue;

  encSpecKeys[2] = CFSTR("RequireHardwareAcceleratedVideoEncoder");
  encSpecValues[2] = kCFBooleanTrue;

  encoderSpecs = CFDictionaryCreateMutable(
    kCFAllocatorDefault, kEncoderSpecKeyCount, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);

  CFDictionaryAddValue(encoderSpecs, encSpecKeys[0], encSpecValues[0]);
  CFDictionaryAddValue(encoderSpecs, encSpecKeys[1], encSpecValues[1]);
  CFDictionaryAddValue(encoderSpecs, encSpecKeys[2], encSpecValues[2]);
#endif

  OSStatus err = VTCompressionSessionCreate(kCFAllocatorDefault, static_cast<int32_t>(data.videoParams.outputWidth),
    static_cast<int32_t>(data.videoParams.outputHeight), kCMVideoCodecType_H264, encoderSpecs, nullptr, nullptr,
    encodedFrameCallback, this, &data.compressionSession);

  if (encoderSpecs) {
    CFRelease(encoderSpecs);
  }

  if (err == noErr) {
    data.currentBitRateKbps = 0;
    data.requestedBitRateKbps = data.currentBitRateKbps;
    data.ApplyBitRate(data.videoParams.minimumKbps);
  }

  if (err == noErr) {
    // Insert a keyframe every 2 seconds.
    const int32_t keyframeInterval =
      static_cast<int32_t>(data.videoParams.targetFramesPerSecond * kKeyFrameIntervalInSeconds);
    CFNumberRef keyframeIntervalRef = CFNumberCreate(nullptr, kCFNumberSInt32Type, &keyframeInterval);
    err =
      VTSessionSetProperty(data.compressionSession, kVTCompressionPropertyKey_MaxKeyFrameInterval, keyframeIntervalRef);
    CFRelease(keyframeIntervalRef);
  }

  if (err == noErr) {
    const int32_t targetFps = static_cast<int32_t>(data.videoParams.targetFramesPerSecond);
    CFNumberRef targetFpsRef = CFNumberCreate(nullptr, kCFNumberSInt32Type, &targetFps);
    err = VTSessionSetProperty(data.compressionSession, kVTCompressionPropertyKey_ExpectedFrameRate, targetFpsRef);
    CFRelease(targetFpsRef);
  }
  if (err == noErr) {
    err = VTSessionSetProperty(
      data.compressionSession, kVTCompressionPropertyKey_RealTime, mEnableBFrames ? kCFBooleanFalse : kCFBooleanTrue);
  }

  if (err == noErr) {
    err = VTSessionSetProperty(data.compressionSession, kVTCompressionPropertyKey_AllowFrameReordering,
      mEnableBFrames ? kCFBooleanTrue : kCFBooleanFalse);
  }

  if (err == noErr) {
    // TODO: Should we consider setting this to a specific H.264 Level (e.g. 3.1)? The only concern is that if AutoLevel
    // picks a high level then the source video may be unplayable on some devices. However, Main (with any level) should
    // be fine for the vast majority of devices
    err = VTSessionSetProperty(
      data.compressionSession, kVTCompressionPropertyKey_ProfileLevel, kVTProfileLevel_H264_Main_AutoLevel);
    if (err == noErr) {
      err = VTSessionSetProperty(
        data.compressionSession, kVTCompressionPropertyKey_H264EntropyMode, kVTH264EntropyMode_CABAC);
    }
    if (err == noErr) {
      err = VTCompressionSessionPrepareToEncodeFrames(data.compressionSession);
    }
  }

  return err;
}

OSStatus ttv::broadcast::AppleVideoEncoder::SetupMagicCookie() {
  OSStatus err = SetupVTCompressionSession(&MagicCookieEncodedCallback);
  if (err != noErr) {
    return TTV_EC_BROADCAST_GRAPHICS_API_ERROR;
  }

  AppleVideoEncoderInternalData& data = *mInternalData;

  // Create a random pixel buffer and encode it to get the magic cookie. The output callback will be called
  // with the encoded frame and we can extract the SPS and PPS from it
  //
  size_t bgraBufferSize =
    sizeof(uint8_t) * 4 * mInternalData->videoParams.outputWidth * mInternalData->videoParams.outputHeight;
  uint8_t* bgraBuffer = new uint8_t[bgraBufferSize];

  CVPixelBufferRef pixelBuffer = nullptr;
  CVReturn cvErr = CVPixelBufferCreateWithBytes(kCFAllocatorDefault, data.videoParams.outputWidth,
    data.videoParams.outputHeight, kSafeNativePixelFormat, bgraBuffer, data.videoParams.outputWidth * 4,
    &PixelBufferReleaseCallback, nullptr, nullptr, &pixelBuffer);

  if (cvErr != noErr) {
    return static_cast<OSStatus>(cvErr);
  }

  TTV_ASSERT(data.compressionSession);
  CMTime pts = CMTimeMake(0, 1000);
  err = VTCompressionSessionEncodeFrame(
    data.compressionSession, pixelBuffer, pts, kCMTimeInvalid, nullptr, pixelBuffer, nullptr);
  if (err != noErr) {
    return err;
  }

  err = VTCompressionSessionCompleteFrames(data.compressionSession, CMTimeMake(0, 1000));
  if (err != noErr) {
    return err;
  }

  TTV_ASSERT(data.sps.size() > 0 && data.pps.size() > 0);

  VTCompressionSessionInvalidate(data.compressionSession);
  CFRelease(data.compressionSession);
  data.compressionSession = nullptr;

  CVPixelBufferRelease(pixelBuffer);
  return err;
}

void ttv::broadcast::AppleVideoEncoder::MagicCookieEncodedCallback(void* outputCallbackRefCon,
  void* /*sourceFrameRefCon*/, OSStatus status, VTEncodeInfoFlags /*infoFlags*/, CMSampleBufferRef sampleBuffer) {
  TTV_ASSERT(status == noErr);
  if (status != noErr) {
    return;
  }

  AppleVideoEncoder* vidEncoder = static_cast<ttv::broadcast::AppleVideoEncoder*>(outputCallbackRefCon);
  TTV_ASSERT(vidEncoder);
  if (!vidEncoder) {
    return;
  }

  AppleVideoEncoderInternalData& data = *vidEncoder->mInternalData;

  bool isKeyframe = IsKeyFrame(sampleBuffer);
  TTV_ASSERT(isKeyframe);
  if (!isKeyframe) {
    return;
  }

  data.sps.clear();
  data.pps.clear();

  CMFormatDescriptionRef format = CMSampleBufferGetFormatDescription(sampleBuffer);
  if (!format) {
    return;
  }

  size_t spsSize = 0;
  size_t ppsSize = 0;
  size_t parmCount = 0;
  const uint8_t* sps = nullptr;
  const uint8_t* pps = nullptr;
  int spsHeaderLength = 0;
  int ppsHeaderLength = 0;
  CMVideoFormatDescriptionGetH264ParameterSetAtIndex(format, 0, &sps, &spsSize, &parmCount, &spsHeaderLength);
  CMVideoFormatDescriptionGetH264ParameterSetAtIndex(format, 1, &pps, &ppsSize, &parmCount, &ppsHeaderLength);

  TTV_ASSERT(spsHeaderLength == 4);
  TTV_ASSERT(ppsHeaderLength == 4);
  const uint8_t header[4] = {0x0, 0x0, 0x0, 0x1};
  data.sps.insert(data.sps.begin(), header, header + 4);
  data.pps.insert(data.pps.begin(), header, header + 4);

  data.sps.insert(data.sps.begin() + 4, sps, sps + spsSize);
  data.pps.insert(data.pps.begin() + 4, pps, pps + ppsSize);
}

void ttv::broadcast::AppleVideoEncoder::FrameEncodedCallback(void* outputCallbackRefCon, void* /*sourceFrameRefCon*/,
  OSStatus /*status*/, VTEncodeInfoFlags /*infoFlags*/, CMSampleBufferRef sampleBuffer) {
  AppleVideoEncoder* vidEncoder = static_cast<ttv::broadcast::AppleVideoEncoder*>(outputCallbackRefCon);
  TTV_ASSERT(vidEncoder);
  if (!vidEncoder) {
    return;
  }

  char* frameData = nullptr;
  size_t frameDataSize = 0;
  CMBlockBufferRef block = CMSampleBufferGetDataBuffer(sampleBuffer);
  OSStatus frameDataErr = CMBlockBufferGetDataPointer(block, 0, nullptr, &frameDataSize, &frameData);
  if (frameDataErr == noErr && frameDataSize > 0) {
    auto packet = std::make_unique<Packet>();

    packet->keyframe = IsKeyFrame(sampleBuffer);
    packet->data.insert(packet->data.begin(), frameData, frameData + frameDataSize);

    CMTime pts = CMSampleBufferGetPresentationTimeStamp(sampleBuffer);
    uint64_t ptsMs = static_cast<uint64_t>((pts.value * 1000) / pts.timescale);

    uint64_t timestamp = 0;
    uint32_t cts = 0;
    if (vidEncoder->mEnableBFrames) {
      CMTime dts = CMSampleBufferGetDecodeTimeStamp(sampleBuffer);
      uint64_t dtsMs = static_cast<uint64_t>((dts.value * 1000) / dts.timescale);

      timestamp = dtsMs;
      cts = static_cast<uint32_t>(ptsMs - dtsMs);
    } else {
      timestamp = ptsMs;
    }

    packet->timestamp = timestamp;
    packet->cts = cts;

    static_cast<ttv::broadcast::AppleVideoEncoder*>(outputCallbackRefCon)->WriteEncodedFrame(std::move(packet));
  }
}

bool ttv::broadcast::AppleVideoEncoder::SupportsReceiverProtocol(IVideoFrameReceiver::ReceiverTypeId typeId) const {
  return typeId == IRawVideoFrameReceiver::GetReceiverTypeId() ||
         typeId == ICVPixelBufferVideoFrameReceiver::GetReceiverTypeId();
}

std::shared_ptr<ttv::broadcast::IVideoFrameReceiver> ttv::broadcast::AppleVideoEncoder::GetReceiverImplementation(
  IVideoFrameReceiver::ReceiverTypeId typeId) {
  if (typeId == IRawVideoFrameReceiver::GetReceiverTypeId()) {
    if (mInternalData->rawReceiver == nullptr) {
      mInternalData->rawReceiver = std::make_shared<RawReceiver>();
    }

    return mInternalData->rawReceiver;
  } else if (typeId == ICVPixelBufferVideoFrameReceiver::GetReceiverTypeId()) {
    if (mInternalData->cvPixelBufferReceiver == nullptr) {
      mInternalData->cvPixelBufferReceiver = std::make_shared<CVPixelBufferReceiver>();
    }

    return mInternalData->cvPixelBufferReceiver;
  } else {
    return nullptr;
  }
}
