/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

/* ****************************************************************************** *\

INTEL CORPORATION PROPRIETARY INFORMATION
This software is supplied under the terms of a license agreement or nondisclosure
agreement with Intel Corporation and may not be copied or disclosed except in
accordance with the terms of that agreement
Copyright(c) 2008-2011 Intel Corporation. All Rights Reserved.

\* ****************************************************************************** */

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/intelsysallocator.h"

#include "twitchsdk/core/memory.h"

#define MSDK_ALIGN32(X) (((mfxU32)((X) + 31)) & (~(mfxU32)31))
#define ID_BUFFER MFX_MAKEFOURCC('B', 'U', 'F', 'F')
#define ID_FRAME MFX_MAKEFOURCC('F', 'R', 'M', 'E')

#pragma warning(disable : 4100)

namespace {
struct sBuffer;
struct sFrame;

struct sBuffer {
  mfxU32 id;
  mfxU32 nbytes;
  mfxU16 type;
};

struct sFrame {
  mfxU32 id;
  mfxFrameInfo info;
};
}  // namespace

//---------------------------------------------------------------------------------------
// SysMemFrameAllocator - Handles video frame allocation requests
//---------------------------------------------------------------------------------------
ttv::broadcast::SysMemFrameAllocator::SysMemFrameAllocator() : m_pBufferAllocator(0), m_bOwnBufferAllocator(false) {
  ttv::trace::Message("IntelSysAllocator", MessageLevel::Info, "SysMemFrameAllocator created");
}

//---------------------------------------------------------------------------------------
ttv::broadcast::SysMemFrameAllocator::~SysMemFrameAllocator() {
  Close();
  ttv::trace::Message("IntelSysAllocator", MessageLevel::Info, "SysMemFrameAllocator destroyed");
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::Init(mfxAllocatorParams *pParams) {
  // check if any params passed from application
  if (pParams) {
    SysMemAllocatorParams *pSysMemParams = 0;
    pSysMemParams = dynamic_cast<SysMemAllocatorParams *>(pParams);
    if (!pSysMemParams) {
      ttv::trace::Message(
        "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::Init - Bad allocator params parameter");
      return MFX_ERR_NOT_INITIALIZED;
    }

    m_pBufferAllocator = pSysMemParams->pBufferAllocator;
    m_bOwnBufferAllocator = false;
  }

  // if buffer allocator wasn't passed from application create own
  if (!m_pBufferAllocator) {
    m_pBufferAllocator = new SysMemBufferAllocator;
    if (!m_pBufferAllocator) {
      ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
        "Inside SysMemFrameAllocator::Init - Could not create buffer allocator");
      return MFX_ERR_MEMORY_ALLOC;
    }

    m_bOwnBufferAllocator = true;
  }

  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::Close() {
  if (m_bOwnBufferAllocator) {
    delete m_pBufferAllocator;
    m_pBufferAllocator = 0;
  }

  return BaseFrameAllocator::Close();
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::LockFrame(mfxMemId mid, mfxFrameData *ptr) {
  if (!m_pBufferAllocator) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::LockFrame - No buffer allocator created");
    return MFX_ERR_NOT_INITIALIZED;
  }

  if (!ptr) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::LockFrame - Bad frame data parameter");
    return MFX_ERR_NULL_PTR;
  }

  sFrame *fs = 0;
  mfxStatus sts = m_pBufferAllocator->Lock(m_pBufferAllocator->pthis, mid, (mfxU8 **)&fs);

  if (MFX_ERR_NONE != sts) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::LockFrame - Unable to lock buffer");
    return sts;
  }

  if (ID_FRAME != fs->id) {
    m_pBufferAllocator->Unlock(m_pBufferAllocator->pthis, mid);
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::LockFrame - Unexpected ID, was not a frame ID");
    return MFX_ERR_INVALID_HANDLE;
  }

  mfxU16 Width2 = (mfxU16)MSDK_ALIGN32(fs->info.Width);
  mfxU16 Height2 = (mfxU16)MSDK_ALIGN32(fs->info.Height);
  ptr->B = ptr->Y = (mfxU8 *)fs + MSDK_ALIGN32(sizeof(sFrame));

  switch (fs->info.FourCC) {
    case MFX_FOURCC_NV12:
      ptr->U = ptr->Y + Width2 * Height2;
      ptr->V = ptr->U + 1;
      ptr->Pitch = Width2;
      break;
    case MFX_FOURCC_YV12:
      ptr->V = ptr->Y + Width2 * Height2;
      ptr->U = ptr->V + (Width2 >> 1) * (Height2 >> 1);
      ptr->Pitch = Width2;
      break;
    case MFX_FOURCC_YUY2:
      ptr->U = ptr->Y + 1;
      ptr->V = ptr->Y + 3;
      ptr->Pitch = 2 * Width2;
      break;
    case MFX_FOURCC_RGB3:
      ptr->G = ptr->B + 1;
      ptr->R = ptr->B + 2;
      ptr->Pitch = 3 * Width2;
      break;
    case MFX_FOURCC_RGB4:
      ptr->G = ptr->B + 1;
      ptr->R = ptr->B + 2;
      ptr->A = ptr->B + 3;
      ptr->Pitch = 4 * Width2;
      break;
    default:
      ttv::trace::Message(
        "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::LockFrame - Unsupported format");
      return MFX_ERR_UNSUPPORTED;
  }

  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::UnlockFrame(mfxMemId mid, mfxFrameData *ptr) {
  if (!m_pBufferAllocator) {
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::UnlockFrame - No buffer allocator created");
    return MFX_ERR_NOT_INITIALIZED;
  }

  mfxStatus sts = m_pBufferAllocator->Unlock(m_pBufferAllocator->pthis, mid);

  if (MFX_ERR_NONE != sts) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::UnlockFrame - Unable to unlock buffer");
    return sts;
  }

  if (nullptr != ptr) {
    ptr->Pitch = 0;
    ptr->Y = 0;
    ptr->U = 0;
    ptr->V = 0;
  }

  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::GetFrameHDL(mfxMemId mid, mfxHDL *handle) {
  return MFX_ERR_UNSUPPORTED;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::CheckRequestType(mfxFrameAllocRequest *request) {
  mfxStatus sts = BaseFrameAllocator::CheckRequestType(request);
  if (MFX_ERR_NONE != sts) {
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::CheckRequestType - Bad request parameter");
    return sts;
  }

  if ((request->Type & MFX_MEMTYPE_SYSTEM_MEMORY) != 0) {
    return MFX_ERR_NONE;
  } else {
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::CheckRequestType - Request type is not system memory");
    return MFX_ERR_UNSUPPORTED;
  }
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::AllocImpl(
  mfxFrameAllocRequest *request, mfxFrameAllocResponse *response) {
  if (!m_pBufferAllocator) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::AllocImpl - No buffer allocator created");
    return MFX_ERR_NOT_INITIALIZED;
  }

  if (0 == (request->Type & MFX_MEMTYPE_SYSTEM_MEMORY)) {
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::AllocImpl - Request type not system memory");
    return MFX_ERR_UNSUPPORTED;
  }

  mfxU32 numAllocated = 0;

  mfxU32 Width2 = MSDK_ALIGN32(request->Info.Width);
  mfxU32 Height2 = MSDK_ALIGN32(request->Info.Height);
  mfxU32 nbytes;
  mfxU8 nBitsPerPixel;

  switch (request->Info.FourCC) {
    case MFX_FOURCC_YV12:
    case MFX_FOURCC_NV12:
      nBitsPerPixel = 12;
      break;
    case MFX_FOURCC_RGB3:
      nBitsPerPixel = 24;
      break;
    case MFX_FOURCC_RGB4:
      nBitsPerPixel = 32;
      break;
    case MFX_FOURCC_YUY2:
      nBitsPerPixel = 16;
      break;
    default:
      return MFX_ERR_UNSUPPORTED;
  }

  nbytes = Width2 * Height2 * nBitsPerPixel / 8;

  safe_array<mfxMemId> mids(new mfxMemId[request->NumFrameSuggested]);
  if (!mids.get()) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::AllocImpl - Could not create memory IDs");
    return MFX_ERR_MEMORY_ALLOC;
  }

  // allocate frames
  for (numAllocated = 0; numAllocated < request->NumFrameSuggested; numAllocated++) {
    mfxStatus sts = m_pBufferAllocator->Alloc(
      m_pBufferAllocator->pthis, nbytes + MSDK_ALIGN32(sizeof(sFrame)), request->Type, &(mids.get()[numAllocated]));

    if (MFX_ERR_NONE != sts)
      break;

    sFrame *fs;
    sts = m_pBufferAllocator->Lock(m_pBufferAllocator->pthis, mids.get()[numAllocated], (mfxU8 **)&fs);

    if (MFX_ERR_NONE != sts)
      break;

    fs->id = ID_FRAME;
    fs->info = request->Info;
    m_pBufferAllocator->Unlock(m_pBufferAllocator->pthis, mids.get()[numAllocated]);
  }

  // check the number of allocated frames
  if (numAllocated < request->NumFrameMin) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemFrameAllocator::AllocImpl - Not enough frames allocated");
    return MFX_ERR_MEMORY_ALLOC;
  }

  response->NumFrameActual = (mfxU16)numAllocated;
  response->mids = mids.release();

  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemFrameAllocator::ReleaseResponse(mfxFrameAllocResponse *response) {
  if (!response) {
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::ReleaseResponse - Bad response parameter");
    return MFX_ERR_NULL_PTR;
  }

  if (!m_pBufferAllocator) {
    ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
      "Inside SysMemFrameAllocator::ReleaseResponse - No buffer alloctor created");
    return MFX_ERR_NOT_INITIALIZED;
  }

  mfxStatus sts = MFX_ERR_NONE;

  if (response->mids) {
    for (mfxU32 i = 0; i < response->NumFrameActual; i++) {
      if (response->mids[i]) {
        sts = m_pBufferAllocator->Free(m_pBufferAllocator->pthis, response->mids[i]);
        if (MFX_ERR_NONE != sts) {
          ttv::trace::Message("IntelSysAllocator", MessageLevel::Error,
            "Inside SysMemFrameAllocator::ReleaseResponse - Unable to free old frame");
          return sts;
        }
      }
    }
  }

  delete[] response->mids;
  response->mids = 0;

  return sts;
}

//---------------------------------------------------------------------------------------
// SysMemBufferAllocator - Handles general allocation requests (i.e. non-video frame)
//---------------------------------------------------------------------------------------
ttv::broadcast::SysMemBufferAllocator::SysMemBufferAllocator() {
  ttv::trace::Message("IntelSysAllocator", MessageLevel::Info, "SysMemBufferAllocator created");
}

//---------------------------------------------------------------------------------------
ttv::broadcast::SysMemBufferAllocator::~SysMemBufferAllocator() {
  ttv::trace::Message("IntelSysAllocator", MessageLevel::Info, "SysMemBufferAllocator destroyed");
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemBufferAllocator::AllocBuffer(mfxU32 nbytes, mfxU16 type, mfxMemId *mid) {
  if (!mid) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::AllocBuffer - Bad memory ID parameter");
    return MFX_ERR_NULL_PTR;
  }

  mfxU32 header_size = MSDK_ALIGN32(sizeof(sBuffer));
  // The buffer must be 16-byte aligned because we may perform SSE operations on it
  // (for example swizzling).
  mfxU8 *buffer_ptr = reinterpret_cast<mfxU8 *>(AlignedAlloc(header_size + nbytes, 16));
  if (!buffer_ptr) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::AllocBuffer - Could not create buffer");
    return MFX_ERR_MEMORY_ALLOC;
  }

  memset(buffer_ptr, 0, header_size + nbytes);

  sBuffer *bs = reinterpret_cast<sBuffer *>(buffer_ptr);
  bs->id = ID_BUFFER;
  bs->type = type;
  bs->nbytes = nbytes;
  *mid = (mfxHDL)bs;

  ttv::trace::Message("IntelSysAllocator", MessageLevel::Debug, "SysMemBufferAllocator::AllocBuffer 0x%x", bs);

  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemBufferAllocator::LockBuffer(mfxMemId mid, mfxU8 **ptr) {
  if (!ptr) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::LockBuffer - Bad buffer parameter");
    return MFX_ERR_NULL_PTR;
  }

  sBuffer *bs = reinterpret_cast<sBuffer *>(mid);

  if (!bs) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::LockBuffer - Bad memory ID parameter");
    return MFX_ERR_INVALID_HANDLE;
  }
  if (ID_BUFFER != bs->id) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::LockBuffer - Memory ID is not a buffer");
    return MFX_ERR_INVALID_HANDLE;
  }

  *ptr = (mfxU8 *)bs + MSDK_ALIGN32(sizeof(sBuffer));
  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemBufferAllocator::UnlockBuffer(mfxMemId mid) {
  sBuffer *bs = static_cast<sBuffer *>(mid);

  if (!bs || ID_BUFFER != bs->id) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::UnlockBuffer - Bad memory ID parameter");
    return MFX_ERR_INVALID_HANDLE;
  }

  return MFX_ERR_NONE;
}

//---------------------------------------------------------------------------------------
mfxStatus ttv::broadcast::SysMemBufferAllocator::FreeBuffer(mfxMemId mid) {
  ttv::trace::Message("IntelSysAllocator", MessageLevel::Debug, "SysMemBufferAllocator::FreeBuffer 0x%x", mid);

  sBuffer *bs = static_cast<sBuffer *>(mid);
  if (!bs || (ID_BUFFER != bs->id)) {
    ttv::trace::Message(
      "IntelSysAllocator", MessageLevel::Error, "Inside SysMemBufferAllocator::FreeBuffer - Bad memory ID parameter");
    return MFX_ERR_INVALID_HANDLE;
  }

  AlignedFree(mid);

  return MFX_ERR_NONE;
}
