/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#if WIN32
#include <windows.h>
#endif

#include "mfxvideo++.h"
#include "twitchsdk/broadcast/id3ddevicemanager9videoframereceiver.h"
#include "twitchsdk/broadcast/iframewriter.h"
#include "twitchsdk/broadcast/intelsysallocator.h"
#include "twitchsdk/broadcast/intelvideoencoder.h"
#include "twitchsdk/broadcast/internal/rgbyuv.h"
#include "twitchsdk/broadcast/irawvideoframereceiver.h"
#include "twitchsdk/broadcast/packet.h"
#include "twitchsdk/broadcast/videoframe.h"
#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

#if D3D_SUPPORT_ENABLED
#include "twitchsdk/broadcast/inteld3dallocator.h"

#include <d3d9.h>
#endif

// When using the hardware encoder, we've encountered a MFX_ERR_DEVICE_FAILED error on certain machine/monitor
// combinations. As such we have defaulted to using software encoding
#define FORCE_SOFTWARE_ENCODER 1

#define MSDK_ASSERT_MSG(ERR) TTV_ASSERT(false && #ERR);
#define MSDK_CHECK_RESULT(P, X, ERR) \
  {                                  \
    if ((X) > (P)) {                 \
      MSDK_ASSERT_MSG(ERR);          \
      return ERR;                    \
    }                                \
  }
#define MSDK_CHECK_ERROR(P, X, ERR) \
  {                                 \
    if ((X) == (P)) {               \
      MSDK_ASSERT_MSG(ERR);         \
      return ERR;                   \
    }                               \
  }
#define MSDK_CHECK_NOT_EQUAL(P, X, ERR) \
  {                                     \
    if ((X) != (P)) {                   \
      MSDK_ASSERT_MSG(ERR);             \
      return ERR;                       \
    }                                   \
  }
#define MSDK_CHECK_EQUAL(P, X, ERR) \
  {                                 \
    if ((X) == (P)) {               \
      MSDK_ASSERT_MSG(ERR);         \
      return ERR;                   \
    }                               \
  }
#define MSDK_RETURN_ON_ERROR(P, ERR) \
  {                                  \
    if (MFX_ERR_NONE > (P)) {        \
      MSDK_ASSERT_MSG(ERR);          \
      return ERR;                    \
    }                                \
  }
#define MSDK_CHECK_POINTER(P, ERR) \
  {                                \
    if ((P) == NULL) {             \
      MSDK_ASSERT_MSG(ERR);        \
      return ERR;                  \
    }                              \
  }

namespace {
using namespace ttv::broadcast;

struct EncodeTask {
  EncodeTask::EncodeTask() : encSyncPoint(nullptr) {
    memset(&bitStream, 0, sizeof(mfxBitstream));
    ttv::trace::Message("IntelVideoEncoder", ttv::MessageLevel::Info, "EncodeTask created");
  }

  void Init(uint bufferSize) {
    Close();
    Reset();

    bitStream.Data = new mfxU8[bufferSize];
    bitStream.MaxLength = bufferSize;
  }

  void Close() {
    delete[] bitStream.Data;
    bitStream.Data = nullptr;
    encSyncPoint = nullptr;
  }

  void WriteBitstream(ttv::broadcast::Packet& outputPacket) {
    if (bitStream.DataLength > 0) {
      outputPacket.data.insert(outputPacket.data.begin(), bitStream.Data, bitStream.Data + bitStream.DataLength);
      outputPacket.timestamp = bitStream.TimeStamp;
      outputPacket.keyframe = (bitStream.FrameType & MFX_FRAMETYPE_IDR) == MFX_FRAMETYPE_IDR;
    }
  }

  void Reset() {
    // mark sync point as free
    encSyncPoint = nullptr;

    // prepare bit stream
    bitStream.DataOffset = 0;
    bitStream.DataLength = 0;

    memset(&control, 0, sizeof(mfxEncodeCtrl));
  }

  mfxSyncPoint encSyncPoint;
  mfxBitstream bitStream;
  mfxEncodeCtrl control;
};

const char* ImplString(mfxIMPL impl) {
  switch (impl & 0xFF) {
    case MFX_IMPL_AUTO:
      return "Auto";
    case MFX_IMPL_SOFTWARE:
      return "Software";
    case MFX_IMPL_HARDWARE:
      return "QSV Device 1";
    case MFX_IMPL_AUTO_ANY:
      return "Auto Any";
    case MFX_IMPL_HARDWARE_ANY:
      return "Any QSV Device";
    case MFX_IMPL_HARDWARE2:
      return "QSV Device 2";
    case MFX_IMPL_HARDWARE3:
      return "QSV Device 3";
    case MFX_IMPL_HARDWARE4:
      return "QSV Device 4";
    default:
      return "Unknown";
  }
}

const char* ViaString(mfxIMPL impl) {
  switch (impl & (~0xFF)) {
    case 0:
      return "None";
    case MFX_IMPL_VIA_ANY:
      return "Any";
    case MFX_IMPL_VIA_D3D9:
      return "d3d9";
    case MFX_IMPL_VIA_D3D11:
      return "d3d11";
    default:
      return "Unknown";
  }
}

const uint kMaxPathLength = MAX_PATH - 12;
wchar_t sDllLoadPath[kMaxPathLength + 1];
const uint32_t kInvalidSurfaceIndex = -1;
const uint32_t kAsyncDepth = 4;

void SetDllPath(const std::wstring& path) {
  TTV_ASSERT(path.size() < kMaxPathLength);
  auto len = path.copy(sDllLoadPath, kMaxPathLength);
  sDllLoadPath[len] = '\0';
}

void ClearDllPath() {
  sDllLoadPath[0] = '\0';
}

HMODULE LoadDll(const std::wstring& dllName) {
  std::wstring fullPath = dllName;
  if (sDllLoadPath[0] != '\0') {
    fullPath = sDllLoadPath + std::wstring(L"\\") + fullPath;
  }
  HMODULE handle = LoadLibraryW(fullPath.c_str());

  return handle;
}

/**
 * The video frame required to pack raw buffers.
 */
class RawVideoFrame : public VideoFrame {
 public:
  RawVideoFrame(const uint8_t* frameBuffer, PixelFormat pixelFormat, bool verticalFlip, uint64_t timeStamp,
    IRawVideoFrameReceiver::UnlockFunc unlockCallback)
      : VideoFrame(IRawVideoFrameReceiver::GetReceiverTypeId()), mFrameBuffer(frameBuffer), mPixelFormat(pixelFormat) {
    SetVerticalFlip(verticalFlip);
    SetTimeStamp(timeStamp);

    SetUnlockCallback([unlockCallback, frameBuffer]() { unlockCallback(frameBuffer); });
  }

  const uint8_t* GetFrameBuffer() const { return mFrameBuffer; }
  PixelFormat GetPixelFormat() const { return mPixelFormat; }

 private:
  const uint8_t* mFrameBuffer;
  PixelFormat mPixelFormat;
};

class RawReceiver : public IRawVideoFrameReceiver {
 public:
  virtual TTV_ErrorCode PackageFrame(const uint8_t* frameBuffer, PixelFormat pixelFormat, bool verticalFlip,
    uint64_t timeStamp, UnlockFunc unlockCallback, std::shared_ptr<VideoFrame>& result) override {
    result = std::make_shared<RawVideoFrame>(frameBuffer, pixelFormat, verticalFlip, timeStamp, unlockCallback);

    return TTV_EC_SUCCESS;
  }
};

#if D3D_SUPPORT_ENABLED
class D3D9Receiver : public ID3dDeviceManager9VideoFrameReceiver {
 public:
  virtual TTV_ErrorCode SetD3dDeviceManager(IDirect3DDeviceManager9* deviceManager) override {
    if (mSetFunc != nullptr) {
      return mSetFunc(deviceManager);
    } else {
      return TTV_EC_INVALID_STATE;
    }
  }

  using SetFunc = std::function<TTV_ErrorCode(IDirect3DDeviceManager9* deviceManager)>;
  SetFunc mSetFunc;
};
#endif
}  // namespace

class ttv::broadcast::IntelVideoEncoderInternalData {
 public:
  IntelVideoEncoderInternalData()
      : mTasks(nullptr),
        mTaskPoolSize(0),
        mTaskBufferStart(0),
        mEncSurfaces(nullptr),
        mFrameWriter(nullptr),
        mExternalAllocator(false),
        mStreamIndex(0),
        requestedBitRateKbps(0),
        mIntelDll(nullptr)
#if D3D_SUPPORT_ENABLED
        ,
        mD3dDeviceManager(nullptr)
#endif
        ,
        mInitialized(false),
        mUsingQuickSync(false) {
    memset(&mMfxEncParams, 0, sizeof(mMfxEncParams));
    memset(&mEncResponse, 0, sizeof(mEncResponse));
  }

  ~IntelVideoEncoderInternalData() {
    mMfxEncoder.reset();
    mMfxFrameAllocator.reset();
    mMfxBufferAllocator.reset();
    mMfxFrameAllocatorParams.reset();

    mMfxSession.Close();
    FreeLibrary(mIntelDll);
  }

  mfxStatus InitEncoderSession() {
    mfxStatus status = MFX_ERR_NONE;

    mfxU16 minor = 4;
    mfxU16 major = 1;

    mfxIMPL implementations[] = {
#if FORCE_SOFTWARE_ENCODER
      MFX_IMPL_SOFTWARE,
#else
      MFX_IMPL_HARDWARE_ANY,
      MFX_IMPL_HARDWARE_ANY | MFX_IMPL_VIA_ANY,
      MFX_IMPL_HARDWARE,
      MFX_IMPL_HARDWARE | MFX_IMPL_VIA_ANY,
      MFX_IMPL_HARDWARE_ANY | MFX_IMPL_VIA_D3D11,
      MFX_IMPL_HARDWARE | MFX_IMPL_VIA_D3D11,
#endif
    };

    // Every call to mfxInit causes dxgi.dll to load, and if the call fails the DLL
    // gets unloaded. This takes a noticeable amount of time so to speed things up
    // we load dxgi.dll into process space here
    HMODULE dxgiHandle = LoadLibraryA("dxgi.dll");
    mfxVersion ver = {minor, major};
    for (mfxIMPL* it = std::begin(implementations); it != std::end(implementations); ++it) {
      ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "Init MFX version %d.%d %s via %s", major, minor,
        ImplString(*it), ViaString(*it));
      status = mMfxSession.Init(*it, &ver);
      if (status == MFX_ERR_NONE)
        break;
    }

    FreeLibrary(dxgiHandle);

    if (status != MFX_ERR_NONE) {
      // Load the Intel DLL using the path provided by the client and try again in software mode
      const wchar_t* dllName = L"libmfxsw32.dll";
#ifdef _M_X64
      dllName = L"libmfxsw64.dll";
#endif
      mIntelDll = LoadDll(dllName);
      if (mIntelDll != nullptr) {
        status = mMfxSession.Init(MFX_IMPL_SOFTWARE, nullptr);
      }
      mUsingQuickSync = false;

      ttv::trace::Message("IntelVideoEncoder", MessageLevel::Info, "IntelVideoEncoder NOT using QuickSync");
    } else {
      mUsingQuickSync = true;

      ttv::trace::Message("IntelVideoEncoder", MessageLevel::Info, "IntelVideoEncoder using QuickSync");
    }

    if (status == MFX_ERR_NONE) {
      mfxIMPL impl;
      mfxVersion ver;
      mMfxSession.QueryIMPL(&impl);
      mMfxSession.QueryVersion(&ver);

      ttv::trace::Message("IntelVideoEncoder", MessageLevel::Info,
        "Successfully initialized mfx version %d.%d with %s via %s", ver.Major, ver.Minor, ImplString(impl),
        ViaString(impl));
    }
    return status;
  }

  mfxStatus GetFreeTask(EncodeTask** ppTask, ttv::broadcast::Packet& outputPacket) {
    mfxStatus sts = MFX_ERR_NONE;

    mfxU32 index = GetFreeTaskIndex();
    if (index >= mTaskPoolSize) {
      sts = MFX_ERR_NOT_FOUND;
    } else {
      *ppTask = &mTasks[index];
    }

    if (MFX_ERR_NOT_FOUND == sts) {
      sts = SynchronizeFirstTask(outputPacket);
      if (MFX_ERR_NOT_FOUND == sts) {
        ttv::trace::Message("IntelVideoEncoder", MessageLevel::Error,
          "Inside IntelVideoEncoder::GetFreeTask - Failed to synchronize task");
      }
      MSDK_CHECK_NOT_EQUAL(sts, MFX_ERR_NONE, sts);

      // try again
      index = GetFreeTaskIndex();
      if (index >= mTaskPoolSize) {
        sts = MFX_ERR_NOT_FOUND;
      } else {
        *ppTask = &mTasks[index];
      }

      if (MFX_ERR_NOT_FOUND == sts) {
        ttv::trace::Message("IntelVideoEncoder", MessageLevel::Error,
          "Inside IntelVideoEncoder::GetFreeTask - Could not get a free task");
      }
    }

    return sts;
  }

  mfxStatus SynchronizeFirstTask(ttv::broadcast::Packet& outputPacket) {
    TTV_ASSERT(mTaskPoolSize > 0);

    MSDK_CHECK_POINTER(mTasks, MFX_ERR_NOT_INITIALIZED);
    MSDK_CHECK_POINTER(mMfxSession, MFX_ERR_NOT_INITIALIZED);

    mfxStatus sts = MFX_ERR_NONE;

    const uint kWaitInterval = 60000;

    // non-null sync point indicates that task is in execution
    if (mTasks[mTaskBufferStart].encSyncPoint != nullptr) {
      sts = mMfxSession.SyncOperation(mTasks[mTaskBufferStart].encSyncPoint, kWaitInterval);
      TTV_ASSERT(sts == MFX_ERR_NONE);

      if (MFX_ERR_NONE == sts) {
        mTasks[mTaskBufferStart].WriteBitstream(outputPacket);
        outputPacket.streamIndex = mStreamIndex;

        // TODO: Try and use the unique_ptr directly everywhere in the code
        auto packet = std::make_unique<Packet>();
        *packet = outputPacket;

        mFrameWriter->WritePacket(std::move(packet));
      }

      if (sts != MFX_WRN_IN_EXECUTION && sts != MFX_WRN_DEVICE_BUSY) {
        mTasks[mTaskBufferStart].Reset();

        // move task buffer start to the next executing task
        // the first transform frame to the right with non zero sync point
        for (mfxU32 i = 0; i < mTaskPoolSize; i++) {
          mTaskBufferStart = (mTaskBufferStart + 1) % mTaskPoolSize;
          if (nullptr != mTasks[mTaskBufferStart].encSyncPoint) {
            break;
          }
        }
      }
    } else {
      ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug,
        "Inside IntelVideoEncoder::SynchronizeFirstTask - No tasks left in the task buffer");
      sts = MFX_ERR_NOT_FOUND;  // no tasks left in task buffer
    }

    return sts;
  }

  uint GetFreeTaskIndex() {
    TTV_ASSERT(mTaskPoolSize > 0);

    uint off = 0;

    if (mTasks != nullptr) {
      for (off = 0; off < mTaskPoolSize; off++) {
        if (nullptr == mTasks[(mTaskBufferStart + off) % mTaskPoolSize].encSyncPoint) {
          break;
        }
      }
    }

    if (off >= mTaskPoolSize) {
      return mTaskPoolSize;
    }

    return (mTaskBufferStart + off) % mTaskPoolSize;
  }

  mfxStatus AllocSurfaces() {
    TTV_ASSERT(mMfxEncoder != nullptr);
    TTV_ASSERT(mMfxFrameAllocator != nullptr);

    mfxFrameAllocRequest encRequest;
    memset(&encRequest, 0, sizeof(encRequest));

    // Determine how many encoder surfaces to allocate
    mfxStatus status = mMfxEncoder->QueryIOSurf(&mMfxEncParams, &encRequest);
    MSDK_CHECK_RESULT(status, MFX_ERR_NONE, status);
    unsigned short encSurfaceReqCount = encRequest.NumFrameSuggested + kAsyncDepth - 1;

    // Allocate encoder surfaces
    encRequest.NumFrameMin = encSurfaceReqCount;
    encRequest.NumFrameSuggested = encSurfaceReqCount;
    memcpy(&encRequest.Info, &mMfxEncParams.mfx.FrameInfo, sizeof(mfxFrameInfo));
    status = mMfxFrameAllocator->Alloc(mMfxFrameAllocator->pthis, &encRequest, &mEncResponse);
    TTV_ASSERT(status == MFX_ERR_NONE);

    // Create and fill our encoder surface pool
    mEncSurfaces = new mfxFrameSurface1[mEncResponse.NumFrameActual];
    MSDK_CHECK_POINTER(mEncSurfaces, MFX_ERR_MEMORY_ALLOC);
    for (uint i = 0; i < mEncResponse.NumFrameActual; ++i) {
      memset(&(mEncSurfaces[i]), 0, sizeof(mfxFrameSurface1));
      memcpy(&(mEncSurfaces[i].Info), &(mMfxEncParams.mfx.FrameInfo), sizeof(mfxFrameInfo));

      if (mExternalAllocator) {
        mEncSurfaces[i].Data.MemId = mEncResponse.mids[i];
        mEncSurfaces[i].Data.Pitch = encRequest.Info.Width;
      } else {
        status = mMfxFrameAllocator->Lock(mMfxFrameAllocator->pthis, mEncResponse.mids[i], &mEncSurfaces[i].Data);
        TTV_ASSERT(status == MFX_ERR_NONE);
      }
    }

    return MFX_ERR_NONE;
  }

  mfxStatus InitEncodeTaskPool(uint poolSize, uint bufferSize) {
    MSDK_CHECK_ERROR(poolSize, 0, MFX_ERR_UNDEFINED_BEHAVIOR);
    MSDK_CHECK_ERROR(bufferSize, 0, MFX_ERR_UNDEFINED_BEHAVIOR);

    mTaskPoolSize = poolSize;
    mTasks = new EncodeTask[mTaskPoolSize];
    MSDK_CHECK_POINTER(mTasks, MFX_ERR_MEMORY_ALLOC);

    for (uint i = 0; i < mTaskPoolSize; ++i) {
      mTasks[i].Init(bufferSize);
    }

    return MFX_ERR_NONE;
  }

  static int GetFreeSurface(const mfxFrameSurface1* pSurfacePool, uint poolSize) {
    int index = kInvalidSurfaceIndex;
    if (pSurfacePool) {
      for (uint i = 0; i < poolSize; ++i) {
        if (!pSurfacePool[i].Data.Locked) {
          index = static_cast<int>(i);
          break;
        }
      }
    }

    return index;
  }

  MFXVideoSession mMfxSession;
  std::unique_ptr<MFXFrameAllocator> mMfxFrameAllocator;
  std::unique_ptr<MFXBufferAllocator> mMfxBufferAllocator;
  std::unique_ptr<mfxAllocatorParams> mMfxFrameAllocatorParams;
  std::unique_ptr<MFXVideoENCODE> mMfxEncoder;
  mfxVideoParam mMfxEncParams;
  VideoParams mVideoParams;
  std::shared_ptr<RawReceiver> mRawReceiver;

#if D3D_SUPPORT_ENABLED
  std::shared_ptr<D3D9Receiver> mD3D9Receiver;
#endif

  // Encode task pool parameters
  EncodeTask* mTasks;
  uint mTaskPoolSize;
  uint mTaskBufferStart;

  mfxFrameAllocResponse mEncResponse;
  mfxFrameSurface1* mEncSurfaces;  // Encoder input frames

  std::shared_ptr<IFrameWriter> mFrameWriter;
  uint32_t mStreamIndex;
  uint32_t requestedBitRateKbps;

#if WIN32
  HMODULE mIntelDll;
#endif

#if D3D_SUPPORT_ENABLED
  IDirect3DDeviceManager9* mD3dDeviceManager;
#endif
  bool mExternalAllocator;
  bool mInitialized;
  bool mUsingQuickSync;
};

ttv::broadcast::IntelVideoEncoder::IntelVideoEncoder() : IVideoEncoder() {
  mInternalData = std::make_shared<IntelVideoEncoderInternalData>();

  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Info, "IntelVideoEncoder created");
}

ttv::broadcast::IntelVideoEncoder::~IntelVideoEncoder() {
  InternalStop();

  mInternalData.reset();

  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Info, "IntelVideoEncoder destroyed");
}

std::string ttv::broadcast::IntelVideoEncoder::GetName() const {
  return mInternalData->mUsingQuickSync ? "IntelQVC" : "IntelSW";
}

bool ttv::broadcast::IntelVideoEncoder::SupportsBitRateAdjustment() const {
  return true;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::Stop() {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::Stop()");

  InternalStop();
  return TTV_EC_SUCCESS;
}

void ttv::broadcast::IntelVideoEncoder::InternalStop() {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::InternalStop()");

  if (mInternalData->mInitialized) {
    FlushFrames();

    if (mInternalData->mMfxEncoder != nullptr) {
      mInternalData->mMfxEncoder->Close();
    }

    DestroyEncodeTaskPool();

    delete[] mInternalData->mEncSurfaces;

    mInternalData->mInitialized = false;
  }
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::SetFrameWriter(const std::shared_ptr<IFrameWriter>& frameWriter) {
  // TODO: Make sure not started
  mInternalData->mFrameWriter = frameWriter;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::ValidateVideoParams(const VideoParams& videoParams) const {
  if ((videoParams.outputHeight % 16) != 0 || (videoParams.outputWidth % 32) != 0) {
    ttv::trace::Message(
      "VideoStreamer", MessageLevel::Error, "Inside IntelVideoEncoder::ValidateVideoParams - Invalid resolution");
    return TTV_EC_BROADCAST_INVALID_RESOLUTION;
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::Initialize() {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::Initialize()");

  // Initialize the session
  //
  mfxStatus status = mInternalData->InitEncoderSession();
  if (status != MFX_ERR_NONE) {
    return TTV_EC_BROADCAST_INTEL_FAILED_SESSION_INIT;
  }

  // Create the allocators - we use external allocators so that all allocations go
  // through our overloaded new/delete
  //

  // Create the general buffer allocator
  mInternalData->mMfxBufferAllocator = std::make_unique<SysMemBufferAllocator>();
  TTV_ASSERT(mInternalData->mMfxBufferAllocator);
  status = mInternalData->mMfxSession.SetBufferAllocator(mInternalData->mMfxBufferAllocator.get());
  TTV_ASSERT(status >= MFX_ERR_NONE);

  // Create the video frame allocator
#if D3D_SUPPORT_ENABLED
  if (mD3dDeviceManager) {
    mMfxSession.SetHandle(MFX_HANDLE_D3D9_DEVICE_MANAGER, mD3dDeviceManager);

    mMfxFrameAllocator = std::make_unique<D3DFrameAllocator>();

    D3DAllocatorParams* pD3dAllocParams = new D3DAllocatorParams;
    TTV_ASSERT(pD3dAllocParams);
    pD3dAllocParams->pManager = mD3dDeviceManager;

    mMfxFrameAllocatorParams = pD3dAllocParams;

    mMfxSession.SetFrameAllocator(mMfxFrameAllocator);
    TTV_ASSERT(status >= MFX_ERR_NONE);
  } else
#endif
  {
    mInternalData->mMfxFrameAllocator = std::make_unique<SysMemFrameAllocator>();

    auto allocatorParams = std::make_unique<SysMemAllocatorParams>();

    // Use our general buffer allocator for the frame allocator
    allocatorParams->pBufferAllocator = mInternalData->mMfxBufferAllocator.get();

    mInternalData->mMfxFrameAllocatorParams = std::move(allocatorParams);
    mInternalData->mMfxSession.SetFrameAllocator(mInternalData->mMfxFrameAllocator.get());
    TTV_ASSERT(status >= MFX_ERR_NONE);
  }

  status = mInternalData->mMfxFrameAllocator->Init(mInternalData->mMfxFrameAllocatorParams.get());
  TTV_ASSERT(status >= MFX_ERR_NONE);
  mInternalData->mExternalAllocator = true;

  return (status == MFX_ERR_NONE) ? TTV_EC_SUCCESS : TTV_EC_BROADCAST_INTEL_FAILED_SESSION_INIT;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::Shutdown() {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::Shutdown()");

  return Stop();
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::Start(uint32_t streamIndex, const VideoParams& videoParams) {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::Start()");

  if (videoParams.outputWidth % 32) {
    ttv::trace::Message("IntelVideoEncoder", MessageLevel::Warning,
      "Inside IntelVideoEncoder::Start - Width is not a multiple of 32; this will incur additional memcpy costs");
  }

  if (mInternalData->mFrameWriter == nullptr) {
    TTV_ASSERT(false && "mFrameWriter != nullptr");
    ttv::trace::Message(
      "IntelVideoEncoder", MessageLevel::Error, "Inside IntelVideoEncoder::Start - Bad frame writer parameter");
    return TTV_EC_INVALID_ARG;
  }

  mInternalData->mStreamIndex = streamIndex;
  mInternalData->mVideoParams = videoParams;

  mfxStatus status = MFX_ERR_NONE;

  // Initialize the encoder
  SetupEncoderParams(videoParams);
  if (mInternalData->mMfxEncoder == nullptr) {
    mInternalData->mMfxEncoder = std::make_unique<MFXVideoENCODE>(mInternalData->mMfxSession);
    MSDK_CHECK_POINTER(mInternalData->mMfxEncoder, TTV_EC_MEMORY);
  }

  status = mInternalData->mMfxEncoder->Init(&mInternalData->mMfxEncParams);
  MSDK_RETURN_ON_ERROR(status, TTV_EC_BROADCAST_INTEL_FAILED_ENCODER_INIT);

  // Allocate the frame surfaces used by the encoder
  status = mInternalData->AllocSurfaces();
  MSDK_RETURN_ON_ERROR(status, TTV_EC_BROADCAST_INTEL_FAILED_SURFACE_ALLOCATION);

  // Initialize the task pool
  mfxVideoParam param;
  memset(&param, 0, sizeof(param));
  status = mInternalData->mMfxEncoder->GetVideoParam(&param);
  MSDK_RETURN_ON_ERROR(status, TTV_EC_BROADCAST_INTEL_FAILED_TASKPOLL_INIT);

  // For the encoded output buffer size use the larger of the output buffer size set during initialization or the raw
  // frame size
  uint rawFrameSize =
    mInternalData->mMfxEncParams.mfx.FrameInfo.Width * mInternalData->mMfxEncParams.mfx.FrameInfo.Height * 4;
  uint encodedDataBufferSize = std::max(static_cast<uint>(param.mfx.BufferSizeInKB * 1024), rawFrameSize);

  status = mInternalData->InitEncodeTaskPool(mInternalData->mEncResponse.NumFrameActual, encodedDataBufferSize);
  MSDK_RETURN_ON_ERROR(status, TTV_EC_BROADCAST_INTEL_FAILED_TASKPOLL_INIT);

  mInternalData->mInitialized = true;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::SetTargetBitRate(uint32_t kbps) {
  if (mInternalData == nullptr || mInternalData->mMfxEncoder == nullptr) {
    return TTV_EC_INVALID_STATE;
  }

  mInternalData->requestedBitRateKbps = kbps;

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::SubmitFrame(const std::shared_ptr<VideoFrame>& videoFrame) {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::SubmitFrame()");

  TTV_ASSERT(videoFrame != nullptr);
  TTV_ASSERT(mInternalData->mInitialized);

  TTV_ASSERT(videoFrame->GetReceiverTypeId() == IRawVideoFrameReceiver::GetReceiverTypeId());
  if (videoFrame->GetReceiverTypeId() != IRawVideoFrameReceiver::GetReceiverTypeId()) {
    return TTV_EC_BROADCAST_INVALID_SUBMISSION_METHOD;
  }

  if (mInternalData->mVideoParams.automaticBitRateAdjustmentEnabled) {
    mfxU16 targetKbps = static_cast<mfxU16>(mInternalData->requestedBitRateKbps);
    if (mInternalData->mMfxEncParams.mfx.TargetKbps != targetKbps) {
      ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug,
        "IntelVideoEncoder::SetTargetBitRate - Updating bit rate to %u kbps", mInternalData->requestedBitRateKbps);

      mInternalData->mMfxEncParams.mfx.TargetKbps = targetKbps;

      mfxStatus status = mInternalData->mMfxEncoder->Reset(&mInternalData->mMfxEncParams);
      MSDK_RETURN_ON_ERROR(status, TTV_EC_BROADCAST_INTEL_FAILED_ENCODER_INIT);
    }
  }

  auto rawFrame = std::static_pointer_cast<RawVideoFrame>(videoFrame);

  int kBytesPerPixel = 0;
  switch (rawFrame->GetPixelFormat()) {
    case PixelFormat::TTV_PF_BGRA:
    case PixelFormat::TTV_PF_ABGR:
    case PixelFormat::TTV_PF_RGBA:
    case PixelFormat::TTV_PF_ARGB:
      kBytesPerPixel = 4;
      break;
    default:
      TTV_ASSERT(false && "Unsupported pixel format");
      ttv::trace::Message(
        "IntelVideoEncoder", MessageLevel::Error, "Inside IntelVideoEncoder::SubmitFrame - Unsupported pixel format");
      return TTV_EC_BROADCAST_UNSUPPORTED_INPUT_FORMAT;
  }

  mfxStatus sts = MFX_ERR_NONE;

  EncodeTask* pCurrentTask = nullptr;  // a pointer to the current task
  int encSurfIdx = 0;                  // index of free surface for encoder input

  // Get a pointer to a free task (bit stream and sync point for encoder)
  Packet outputPacket;
  sts = mInternalData->GetFreeTask(&pCurrentTask, outputPacket);
  MSDK_RETURN_ON_ERROR(sts, TTV_EC_BROADCAST_INTEL_NO_FREE_TASK);

  // Find free surface for encoder input
  encSurfIdx = mInternalData->GetFreeSurface(mInternalData->mEncSurfaces, mInternalData->mEncResponse.NumFrameActual);
  MSDK_CHECK_EQUAL(encSurfIdx, kInvalidSurfaceIndex, TTV_EC_BROADCAST_INTEL_NO_FREE_SURFACE);

  mfxFrameSurface1* pSurf = &mInternalData->mEncSurfaces[encSurfIdx];

  if (mInternalData->mExternalAllocator) {
    sts = mInternalData->mMfxFrameAllocator->Lock(
      mInternalData->mMfxFrameAllocator->pthis, pSurf->Data.MemId, &(pSurf->Data));
    MSDK_RETURN_ON_ERROR(sts, TTV_EC_UNKNOWN_ERROR);
  }

  // Convert to RGB to NV12
  RGBtoYUV(rawFrame->GetFrameBuffer(),
    static_cast<uint32_t>(rawFrame->GetPixelFormat()),  // TODO: This enum usage is dangerous
    mInternalData->mMfxEncParams.mfx.FrameInfo.Width, mInternalData->mMfxEncParams.mfx.FrameInfo.Height, pSurf->Data.Y,
    pSurf->Data.UV, YUVFormat::TTV_YUV_NV12, videoFrame->GetVerticalFlip());

  if (mInternalData->mExternalAllocator) {
    sts = mInternalData->mMfxFrameAllocator->Unlock(
      mInternalData->mMfxFrameAllocator->pthis, pSurf->Data.MemId, &(pSurf->Data));
    MSDK_RETURN_ON_ERROR(sts, TTV_EC_UNKNOWN_ERROR);
  }

  pSurf->Data.TimeStamp = SystemTimeToMs(videoFrame->GetTimeStamp());

  if (videoFrame->IsKeyFrame()) {
    pCurrentTask->control.FrameType = MFX_FRAMETYPE_I | MFX_FRAMETYPE_REF | MFX_FRAMETYPE_IDR;
  }

  for (;;) {
    sts = mInternalData->mMfxEncoder->EncodeFrameAsync(&pCurrentTask->control, &mInternalData->mEncSurfaces[encSurfIdx],
      &pCurrentTask->bitStream, &pCurrentTask->encSyncPoint);

    if (MFX_ERR_NONE < sts) {
      // We got a warning
      //
      // If no sync point returned, repeat the call
      if (!pCurrentTask->encSyncPoint) {
        if (MFX_WRN_DEVICE_BUSY == sts) {
          ttv::Sleep(1);  // Wait if device is busy
        }
        continue;
      } else {
        // If we got a sync point, then ignore the warning
        sts = MFX_ERR_NONE;
        break;
      }
    } else {
      // No warning; either succeeded, or got an error which we handle outside the loop
      break;
    }
  }

  if (MFX_ERR_MORE_DATA == sts) {
    return TTV_EC_SUCCESS;  // Encoder needs more frame(s) before so just return
  } else {
    MSDK_RETURN_ON_ERROR(sts, TTV_EC_UNKNOWN_ERROR);  // Return error if we got any other errors
  }

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::IntelVideoEncoder::FlushFrames() {
  ttv::trace::Message("IntelVideoEncoder", MessageLevel::Debug, "IntelVideoEncoder::FlushFrames()");

  Packet dummyPacket;
  mfxStatus sts = mInternalData->SynchronizeFirstTask(dummyPacket);
  while (sts != MFX_ERR_NOT_FOUND) {
    sts = mInternalData->SynchronizeFirstTask(dummyPacket);
  }

  return;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::GetSpsPps(std::vector<uint8_t>& sps, std::vector<uint8_t>& pps) {
  TTV_ErrorCode ret = TTV_EC_SUCCESS;

  TTV_ASSERT(mInternalData->mInitialized);
  TTV_ASSERT(mInternalData->mMfxEncoder);

  if (mInternalData->mInitialized && mInternalData->mMfxEncoder) {
    const uint kMaxSpsPpsBufferSize = 128;
    sps.resize(kMaxSpsPpsBufferSize, 0);
    pps.resize(kMaxSpsPpsBufferSize, 0);

    std::shared_ptr<mfxExtCodingOptionSPSPPS> pExtSpsPps(new mfxExtCodingOptionSPSPPS());
    TTV_ASSERT(pExtSpsPps != nullptr);
    pExtSpsPps->Header.BufferId = MFX_EXTBUFF_CODING_OPTION_SPSPPS;
    pExtSpsPps->Header.BufferSz = sizeof(mfxExtCodingOptionSPSPPS);
    pExtSpsPps->SPSBuffer = sps.data();
    pExtSpsPps->PPSBuffer = pps.data();
    pExtSpsPps->PPSBufSize = kMaxSpsPpsBufferSize;
    pExtSpsPps->SPSBufSize = kMaxSpsPpsBufferSize;

    std::shared_ptr<mfxExtBuffer*> ppExtBuffs(new mfxExtBuffer*[1]);
    TTV_ASSERT(ppExtBuffs != nullptr);
    ppExtBuffs.get()[0] = reinterpret_cast<mfxExtBuffer*>(pExtSpsPps.get());

    mfxVideoParam param;
    memset(&param, 0, sizeof(mfxVideoParam));
    param.NumExtParam = 1;
    param.ExtParam = ppExtBuffs.get();
    mfxStatus status = mInternalData->mMfxEncoder->GetVideoParam(&param);
    TTV_ASSERT(pExtSpsPps->SPSBufSize <= kMaxSpsPpsBufferSize);
    TTV_ASSERT(pExtSpsPps->PPSBufSize <= kMaxSpsPpsBufferSize);
    sps.resize(pExtSpsPps->SPSBufSize);
    pps.resize(pExtSpsPps->PPSBufSize);

    if (status != MFX_ERR_NONE) {
      ret = TTV_EC_BROADCAST_NO_SPSPPS;
      ttv::trace::Message(
        "IntelVideoEncoder", MessageLevel::Error, "Inside IntelVideoEncoder::GetSpsPps - Unable to get video params");
    }
  } else {
    ret = TTV_EC_NOT_INITIALIZED;
    ttv::trace::Message(
      "IntelVideoEncoder", MessageLevel::Error, "Inside IntelVideoEncoder::GetSpsPps - Not initialized");
  }

  return ret;
}

TTV_ErrorCode ttv::broadcast::IntelVideoEncoder::ValidateFrame(const std::shared_ptr<VideoFrame>& videoframe) {
  TTV_ASSERT(videoframe->GetReceiverTypeId() == IRawVideoFrameReceiver::GetReceiverTypeId());
  auto rawFrame = std::static_pointer_cast<RawVideoFrame>(videoframe);

  // Since our RGB->YUV code uses SSE, the buffer must be 16-byte aligned
  if ((reinterpret_cast<uintptr_t>(rawFrame->GetFrameBuffer()) & 0xF) != 0) {
    return TTV_EC_ALIGN16_REQUIRED;
  }

  return TTV_EC_SUCCESS;
}

void ttv::broadcast::IntelVideoEncoder::SetupEncoderParams(const VideoParams& videoParams) {
  auto& mfxParams = mInternalData->mMfxEncParams;

  memset(&mfxParams, 0, sizeof(mfxParams));
  mfxParams.mfx.CodecId = MFX_CODEC_AVC;

  if (mInternalData->mUsingQuickSync) {
    // If using QuickSync then always aim for highest quality
    mfxParams.mfx.TargetUsage = MFX_TARGETUSAGE_BEST_QUALITY;
  } else {
    switch (videoParams.encodingCpuUsage) {
      case EncodingCpuUsage::Low:
        mfxParams.mfx.TargetUsage = MFX_TARGETUSAGE_BEST_SPEED;
        break;
      case EncodingCpuUsage::Medium:
        mfxParams.mfx.TargetUsage = MFX_TARGETUSAGE_BALANCED;
        break;
      case EncodingCpuUsage::High:
      case EncodingCpuUsage::Default:
        mfxParams.mfx.TargetUsage = MFX_TARGETUSAGE_BEST_QUALITY;
        break;
      default:
        mfxParams.mfx.TargetUsage = MFX_TARGETUSAGE_BEST_SPEED;
        TTV_ASSERT(false && "Invalid Encoding CPU usage specified");
        break;
    }
  }

  mInternalData->requestedBitRateKbps = videoParams.initialKbps;

  mfxParams.mfx.MaxKbps = static_cast<mfxU16>(videoParams.maximumKbps);  // Equivalent to x264's vbv_max_bitrate
  mfxParams.mfx.BufferSizeInKB =
    ((mfxParams.mfx.MaxKbps * 1000 * 2) / 8) / 1024;  // Equivalent to x264's vbv_buffer_size
  mfxParams.mfx.InitialDelayInKB = static_cast<mfxU16>(
    0.9f * static_cast<float>(mfxParams.mfx.BufferSizeInKB));  // Equivalent to x264's vbv_buffer_init

  mfxParams.mfx.RateControlMethod =
    videoParams.automaticBitRateAdjustmentEnabled ? MFX_RATECONTROL_AVBR : MFX_RATECONTROL_CBR;
  mfxParams.mfx.TargetKbps = static_cast<mfxU16>(videoParams.initialKbps);  // Equivalent to x264's bitrate

  // <--Only used in AVBR mode--> mfxParams.mfx.Accuracy = 10;    // bit rate accuracy in units of 0.1 %
  // <--Only used in AVBR mode--> mfxParams.mfx.Convergence = 1;  // 'converging' to target bit rate after this many
  // frames (in units of 100 frames)

  mfxParams.mfx.GopPicSize = 0;  // Don't specify a gop size, instead we will manually force key frames
  mfxParams.mfx.GopRefDist = 1;
  mfxParams.mfx.IdrInterval = 0;  // Every Nth I-frame will be an IDR (i.e. key frame).
                                  // 0 means every I-frame is a key frame.
  // We get an I-frame at the start of each GOP so we get an IDR frame every F frames where:
  // F = GopPicSize * (IdrInterval + 1)

  mfxParams.mfx.EncodedOrder = 0;
  mfxParams.IOPattern =
    MFX_IOPATTERN_IN_SYSTEM_MEMORY;  // TODO - if we get a DX surface, we should set this to video memory
  mfxParams.mfx.CodecProfile =
    MFX_PROFILE_AVC_MAIN;  // TODO: The sample doesn't set profile except in MVC. Do we need this?

  mfxParams.mfx.FrameInfo.FourCC = MFX_FOURCC_NV12;
  mfxParams.mfx.FrameInfo.ChromaFormat = MFX_CHROMAFORMAT_YUV420;
  mfxParams.mfx.FrameInfo.PicStruct = MFX_PICSTRUCT_PROGRESSIVE;  // ** IMPORTANT ** According to the samples, if we
                                                                  // change this to something other than progressive,
  // we may have to make sure height is multiple of 32 (as is, the requirement is multiple of 16)

  mfxParams.mfx.FrameInfo.Width = static_cast<mfxU16>(videoParams.outputWidth);
  mfxParams.mfx.FrameInfo.Height = static_cast<mfxU16>(videoParams.outputHeight);
  mfxParams.mfx.FrameInfo.CropX = 0;
  mfxParams.mfx.FrameInfo.CropY = 0;
  mfxParams.mfx.FrameInfo.CropW = static_cast<mfxU16>(videoParams.outputWidth);
  mfxParams.mfx.FrameInfo.CropH = static_cast<mfxU16>(videoParams.outputHeight);

  // The framerate value set here doesn't actually affect the rate at which frames are encoded.
  // Frames are encoded as they come in and their timestamp is used for the PTS. However, the
  // framerate specified here affects the size of the encoded frames. Since the encoder is trying to
  // achieve the specified bitrate (TargetKbps), it compresses frames assuming this framerate value.
  // As a result, if frames are submitted at a faster rate than set here, then the output size is
  // going to exceed what would be expected based on TargetKbps since the size of each compressed
  // frame will be larger than it should be for the actual framerate at which frames are being
  // submitted. Therefore we need to make sure that we do not submit frames to the encoder at a
  // faster rate than we set here.
  //
  mfxParams.mfx.FrameInfo.FrameRateExtN = static_cast<mfxU16>(videoParams.targetFramesPerSecond);
  mfxParams.mfx.FrameInfo.FrameRateExtD = 1;

#if D3D_SUPPORT_ENABLED
  if (mD3dDeviceManager) {
    mfxParams.IOPattern = MFX_IOPATTERN_IN_VIDEO_MEMORY;
  } else
#endif
  {
    mfxParams.IOPattern = MFX_IOPATTERN_IN_SYSTEM_MEMORY;
  }
}

void ttv::broadcast::IntelVideoEncoder::DestroyEncodeTaskPool() {
  if (mInternalData->mTasks != nullptr) {
    for (mfxU32 i = 0; i < mInternalData->mTaskPoolSize; i++) {
      mInternalData->mTasks[i].Close();
    }
  }

  delete[] mInternalData->mTasks;
  mInternalData->mTasks = nullptr;

  mInternalData->mTaskBufferStart = 0;
  mInternalData->mTaskPoolSize = 0;
}

bool ttv::broadcast::IntelVideoEncoder::SupportsReceiverProtocol(IVideoFrameReceiver::ReceiverTypeId typeId) const {
  return typeId == IRawVideoFrameReceiver::GetReceiverTypeId()
#if D3D_SUPPORT_ENABLED
         || typeId == ID3dDeviceManager9VideoFrameReceiver::GetReceiverTypeId()
#endif
    ;
}

std::shared_ptr<ttv::broadcast::IVideoFrameReceiver> ttv::broadcast::IntelVideoEncoder::GetReceiverImplementation(
  IVideoFrameReceiver::ReceiverTypeId typeId) {
  if (typeId == IRawVideoFrameReceiver::GetReceiverTypeId()) {
    if (mInternalData->mRawReceiver == nullptr) {
      mInternalData->mRawReceiver = std::make_shared<RawReceiver>();
    }

    return mInternalData->mRawReceiver;
  }

#if D3D_SUPPORT_ENABLED
  if (typeId == ID3dDeviceManager9VideoFrameReceiver::GetReceiverTypeId()) {
    if (mInternalData->mD3D9Receiver == nullptr) {
      mInternalData->mD3D9Receiver = std::make_shared<D3D9Receiver>();
      mInternalData->mRawReceiver->mSetFunc = [this](IDirect3DDeviceManager9* deviceManager) {
        TTV_ASSERT(deviceManager != nullptr);
        mInternalData->mD3dDeviceManager = deviceManager;

        return TTV_EC_SUCCESS;
      };
    }

    return mInternalData->mD3D9Receiver;
  }
#endif

  return nullptr;
}
