package tv.twitch.broadcast;

import tv.twitch.CoreAPI;
import tv.twitch.ErrorCode;
import tv.twitch.IJniCallable;
import tv.twitch.IJniThreadChecker;
import tv.twitch.IJniThreadValidator;
import tv.twitch.IModule;
import tv.twitch.JniThreadValidator;
import tv.twitch.ModuleState;
import tv.twitch.NativeProxy;
import tv.twitch.PassThroughJniThreadValidator;
import tv.twitch.ResultContainer;
import tv.twitch.broadcast.callbacks.FetchIngestListCallback;
import tv.twitch.broadcast.callbacks.RunCommercialCallback;
import tv.twitch.broadcast.callbacks.SetStreamInfoCallback;
import tv.twitch.broadcast.callbacks.StartBroadcastCallback;
import tv.twitch.broadcast.callbacks.StopBroadcastCallback;

/**
 * This module provides the ability to broadcast to Twitch by accepting video and audio data from a game/app, encoding
 * it and send it to the Twitch backend in realtime.
 *
 * The following properties must be set before calling Initialize().
 * - setCoreApi()
 * - setListener()
 *
 * Currently, only one user may be broadcasting at a time.  In order to specify which user will broadcast
 * setActiveUser() must be used.  The active user must be set and the user logged in before configuring any of the
 * encoders, capturers and other settings.  If the active user is changed or logged out then the configuration is lost.
 */
public class BroadcastAPI extends NativeProxy implements IModule {
    static { BroadcastErrorCode.forceClassInit(); }

    public static final int TTV_MIN_BITRATE = 230;
    public static final int TTV_MAX_BITRATE = 3500;
    public static final int TTV_MIN_FPS = 10;
    public static final int TTV_MAX_FPS = 60;
    public static final int TTV_MAX_WIDTH = 1920;
    public static final int TTV_MAX_HEIGHT = 1200;

    /**
     * Legacy Constructor
     * @Deprecated use {@link BroadcastAPI( IJniThreadChecker )}
     */
    @Deprecated
    public BroadcastAPI() {
        super(PassThroughJniThreadValidator.INSTANCE);
    }

    public BroadcastAPI(IJniThreadChecker jniThreadChecker) { super(new JniThreadValidator(jniThreadChecker)); }

    private native long CreateNativeInstance();
    // AND-9927 It is safe to dispose a broadcast API from any thread
    private native void DisposeNativeInstance(long nativeObjectPointer);

    private native ErrorCode SetCoreApi(long nativeObjectPointer, CoreAPI coreApi);
    private native ErrorCode SetListener(long nativeObjectPointer, IBroadcastAPIListener listener);

    private native ModuleState GetState(long nativeObjectPointer);
    private native String GetModuleName(long nativeObjectPointer);
    private native ErrorCode Initialize(long nativeObjectPointer, InitializeCallback callback);
    private native ErrorCode Shutdown(long nativeObjectPointer, ShutdownCallback callback);
    private native ErrorCode Update(long nativeObjectPointer);

    private native ErrorCode SetActiveUser(long nativeObjectPointer, int userId);
    private native ErrorCode SetBroadcasterSoftware(long nativeObjectPointer, String str);
    private native ErrorCode SetForceArchiveBroadcast(long nativeObjectPointer, boolean forceArchiveBroadcast);
    private native ErrorCode SetForceDontArchiveBroadcast(long nativeObjectPointer, boolean forceDontArchiveBroadcast);
    private native ErrorCode SetVideoEncoder(long nativeObjectPointer, IVideoEncoder encoder);
    private native ErrorCode SetAudioEncoder(long nativeObjectPointer, IAudioEncoder encoder);
    private native ErrorCode SetVideoCapturer(long nativeObjectPointer, IVideoCapture capturer);
    private native ErrorCode SetAudioCapturer(long nativeObjectPointer, int layer, IAudioCapture capturer);
    private native ErrorCode RemoveAudioCapturer(long nativeObjectPointer, int layer);
    private native ErrorCode SetAudioLayerVolume(long nativeObjectPointer, int layer, float volume);
    private native ErrorCode SetAudioLayerMuted(long nativeObjectPointer, int layer, boolean muted);
    private native ErrorCode SetAudioLayerEnabled(long nativeObjectPointer, int layer, boolean enabled);
    private native ErrorCode SetVideoParams(long nativeObjectPointer, VideoParams videoParams);
    private native ErrorCode GetVideoParams(long nativeObjectPointer, ResultContainer<VideoParams> videoParams);
    private native ErrorCode SetOutputPath(long nativeObjectPointer, String outputPath);
    private native ErrorCode SetConnectionType(long nativeObjectPointer, ConnectionType connectionType);
    private native ErrorCode SetSessionId(long nativeObjectPointer, String sessionId);
    private native ErrorCode AddBandwidthStatListener(long nativeObjectPointer, IBandwidthStatListener listener);
    private native ErrorCode RemoveBandwidthStatListener(long nativeObjectPointer, IBandwidthStatListener listener);
    private native ErrorCode StartBroadcast(
        long nativeObjectPointer, tv.twitch.broadcast.callbacks.StartBroadcastCallback callback);
    private native ErrorCode StopBroadcast(
        long nativeObjectPointer, String reason, tv.twitch.broadcast.callbacks.StopBroadcastCallback callback);
    private native ErrorCode GetCurrentBroadcastTime(long nativeObjectPointer, ResultContainer<Long> result);
    private native ErrorCode FetchIngestServerList(long nativeObjectPointer, FetchIngestListCallback callback);
    private native ErrorCode GetSelectedIngestServer(long nativeObjectPointer, ResultContainer<IngestServer> result);
    private native ErrorCode SetSelectedIngestServer(long nativeObjectPointer, IngestServer server);
    private native ErrorCode GetBroadcastState(long nativeObjectPointer, ResultContainer<BroadcastState> result);
    private native ErrorCode RunCommercial(
        long nativeObjectPointer, int userId, int channelId, int timebreakSeconds, RunCommercialCallback callback);
    private native ErrorCode SetStreamInfo(
        long nativeObjectPointer, int userId, int channelId, String game, String title, SetStreamInfoCallback callback);
    private native ErrorCode CreateIngestTester(long nativeObjectPointer, IJniThreadValidator jniThreadValidator,
        int userId, IIngestTesterListener listener, byte[] sampleData, ResultContainer<IIngestTester> result);
    private native ErrorCode SetFlvMuxerAsyncEnabled(long nativeObjectPointer, boolean enable);
    private native ErrorCode GetFlvMuxerAsyncEnabled(
        long nativeObjectPointer, ResultContainer<Boolean> isEnabledResult);

    @Override
    protected long createNativeInstance() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Long>() {
            @Override
            public Long call() {
                return CreateNativeInstance();
            }
        });
    }

    @Override
    protected void disposeNativeInstance(long nativeObjectPointer) {
        // AND-9927 It is safe to dispose a broadcast API from any thread
        DisposeNativeInstance(nativeObjectPointer);
    }

    public ErrorCode setCoreApi(final CoreAPI coreApi) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetCoreApi(getNativeObjectPointer(), coreApi);
            }
        });
    }

    public ErrorCode setListener(final IBroadcastAPIListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetListener(getNativeObjectPointer(), listener);
            }
        });
    }

    @Override
    public ModuleState getState() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ModuleState>() {
            @Override
            public ModuleState call() {
                return GetState(getNativeObjectPointer());
            }
        });
    }

    @Override
    public String getModuleName() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<String>() {
            @Override
            public String call() {
                return GetModuleName(getNativeObjectPointer());
            }
        });
    }

    @Override
    public ErrorCode initialize(final InitializeCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Initialize(getNativeObjectPointer(), callback);
            }
        });
    }

    @Override
    public ErrorCode shutdown(final ShutdownCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Shutdown(getNativeObjectPointer(), callback);
            }
        });
    }

    @Override
    public ErrorCode update() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Update(getNativeObjectPointer());
            }
        });
    }

    public ErrorCode setBroadcasterSoftware(final String str) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetBroadcasterSoftware(getNativeObjectPointer(), str);
            }
        });
    }

    public ErrorCode setForceArchiveBroadcast(final boolean forceArchiveBroadcast) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetForceArchiveBroadcast(getNativeObjectPointer(), forceArchiveBroadcast);
            }
        });
    }

    public ErrorCode setForceDontArchiveBroadcast(final boolean forceDontArchiveBroadcast) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetForceDontArchiveBroadcast(getNativeObjectPointer(), forceDontArchiveBroadcast);
            }
        });
    }

    /**
     * Sets the userid of the user who will be broadcasting.  This must be set before attempting to broadcast and cannot
     * be changed while broadcasting.  See the class description for more details.
     */
    public ErrorCode setActiveUser(final int userId) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetActiveUser(getNativeObjectPointer(), userId);
            }
        });
    }

    /**
     * Sets the desired video encoder to use.
     */
    public ErrorCode setVideoEncoder(final IVideoEncoder encoder) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetVideoEncoder(getNativeObjectPointer(), encoder);
            }
        });
    }

    /**
     * Sets the desired audio encoder to use.
     */
    public ErrorCode setAudioEncoder(final IAudioEncoder encoder) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetAudioEncoder(getNativeObjectPointer(), encoder);
            }
        });
    }

    /**
     * Sets the source of video frames.
     */
    public ErrorCode setVideoCapturer(final IVideoCapture capturer) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetVideoCapturer(getNativeObjectPointer(), capturer);
            }
        });
    }

    /**
     * Sets the source of audio samples for the given layer.  The layer ids are up to the client but generally start at
     * 0 and increment for each layer the application adds.  For instance, a capturer for system audio could be added
     * for layer 0 and a microphone capturer added for layer 1.
     */
    public ErrorCode setAudioCapturer(final int layer, final IAudioCapture capturer) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetAudioCapturer(getNativeObjectPointer(), layer, capturer);
            }
        });
    }

    /**
     * Removes a previously added audio capturer for the given layer.
     */
    public ErrorCode removeAudioCapturer(final int layer) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RemoveAudioCapturer(getNativeObjectPointer(), layer);
            }
        });
    }

    /**
     * Sets the volume of given audio layer and can be toggled freely during a stream.
     * The IAudioCapture for the layer must already be set via SetAudioCapturer().
     */
    public ErrorCode setAudioLayerVolume(final int layer, final float volume) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetAudioLayerVolume(getNativeObjectPointer(), layer, volume);
            }
        });
    }

    /**
     * This mutes the audio for the given audio layer and can be toggled freely during a stream.
     * The IAudioCapture for the layer must already be set via SetAudioCapturer().
     */
    public ErrorCode setAudioLayerMuted(final int layer, final boolean muted) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetAudioLayerMuted(getNativeObjectPointer(), layer, muted);
            }
        });
    }

    /**
     * This will enable or disable the associated capturer for the entire broadcast.  This cannot be changed while
     * broadcasting. The IAudioCapture for the layer must already be set via SetAudioCapturer().
     */
    public ErrorCode setAudioLayerEnabled(final int layer, final boolean enabled) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetAudioLayerEnabled(getNativeObjectPointer(), layer, enabled);
            }
        });
    }

    /**
     * Configures the broadcast settings so that a broadcast stays within the given maxKbps.  The resolution of the
     * broadcast will be computed automatically based on given parameters.
     */
    public ErrorCode setVideoParams(final VideoParams videoParams) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetVideoParams(getNativeObjectPointer(), videoParams);
            }
        });
    }

    public ErrorCode getVideoParams(final ResultContainer<VideoParams> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetVideoParams(getNativeObjectPointer(), result);
            }
        });
    }

    /**
     * Sets the output FLV file path for saving a broadcast locally for debugging and verification.
     */
    public ErrorCode setOutputPath(final String outputPath) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetOutputPath(getNativeObjectPointer(), outputPath);
            }
        });
    }

    /**
     * Sets the type of connection that will be used to broadcast. This value is used when sending tracking events.
     */
    public ErrorCode setConnectionType(final ConnectionType connectionType) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetConnectionType(getNativeObjectPointer(), connectionType);
            }
        });
    }

    /**
     * Sets a unique ID for the broadcasting session, which can span multiple starts/stops. This value is used when
     * sending tracking events.
     */
    public ErrorCode setSessionId(final String sessionId) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetSessionId(getNativeObjectPointer(), sessionId);
            }
        });
    }

    /**
     * Adds a listener that will be periodically called back with statistics about network usage. This should not be
     * invoked while broadcasting.
     */
    public ErrorCode addBandwidthStatListener(final IBandwidthStatListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return AddBandwidthStatListener(getNativeObjectPointer(), listener);
            }
        });
    }

    /**
     * Removes a previously added bandwidth stat listener. This should not be invoked while broadcasting.
     */
    public ErrorCode removeBandwidthStatListener(final IBandwidthStatListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RemoveBandwidthStatListener(getNativeObjectPointer(), listener);
            }
        });
    }

    /**
     * Initiates broadcasting for the active user.
     */
    public ErrorCode startBroadcast(final StartBroadcastCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return StartBroadcast(getNativeObjectPointer(), callback);
            }
        });
    }

    /**
     * Gracefully stops broadcasting.
     * @param[in] reason A string describing why the stream is being stopped, used for tracking. Pass
     * in "user_ended" if this was an explicit action from the user.
     */
    public ErrorCode stopBroadcast(final String reason, final StopBroadcastCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return StopBroadcast(getNativeObjectPointer(), reason, callback);
            }
        });
    }

    /**
     * Retrieves the current time in the live broadcast.
     */
    public ErrorCode getCurrentBroadcastTime(final ResultContainer<Long> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetCurrentBroadcastTime(getNativeObjectPointer(), result);
            }
        });
    }

    /**
     * Fetches the list of available ingest servers that can be used for broadcasting.
     */
    public ErrorCode fetchIngestServerList(final FetchIngestListCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchIngestServerList(getNativeObjectPointer(), callback);
            }
        });
    }

    /**
     * Retrieves the currently selected ingest server.  It's possible that none is selected and will contain empty
     * values.
     */
    public ErrorCode getSelectedIngestServer(final ResultContainer<IngestServer> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetSelectedIngestServer(getNativeObjectPointer(), result);
            }
        });
    }

    /**
     * Sets the selected ingest server to use when initiating broadcasts.
     */
    public ErrorCode setSelectedIngestServer(final IngestServer server) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetSelectedIngestServer(getNativeObjectPointer(), server);
            }
        });
    }

    /**
     * Retrieves the current state of broadcasting.
     */
    public ErrorCode getBroadcastState(final ResultContainer<BroadcastState> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetBroadcastState(getNativeObjectPointer(), result);
            }
        });
    }

    /**
     * Triggers a commercial of the specified duration in seconds.  This may be triggered independently from whether or
     * not a broadcast is in progress.  Note that only partnered users may run commercials.
     */
    public ErrorCode runCommercial(
        final int userId, final int channelId, final int timebreakSeconds, final RunCommercialCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RunCommercial(getNativeObjectPointer(), userId, channelId, timebreakSeconds, callback);
            }
        });
    }

    /**
     * Sets the current game and stream title for a channel.  These values may be set independently from whether or not
     * a broadcast is in progress.
     */
    public ErrorCode setStreamInfo(final int userId, final int channelId, final String game, final String title,
        final SetStreamInfoCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetStreamInfo(getNativeObjectPointer(), userId, channelId, game, title, callback);
            }
        });
    }

    /**
     * Returns an implementation of IIngestTester for the given owning user.  When done with the implementation it must
     * be released via IIngestTester::dispose().
     */
    public ErrorCode createIngestTester(final int userId, final IIngestTesterListener listener, final byte[] sampleData,
        final ResultContainer<IIngestTester> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateIngestTester(
                    getNativeObjectPointer(), mJniThreadValidator, userId, listener, sampleData, result);
            }
        });
    }

    /**
     * Configure the video pipeline to use async flv muxer. When enabled, the muxer queues up audio/video frames when
     * outputs (RTMP) are in connecting state and send them once the outputs are connected.
     */
    public ErrorCode setFlvMuxerAsyncEnabled(boolean enable) {
        return SetFlvMuxerAsyncEnabled(getNativeObjectPointer(), enable);
    }

    public ErrorCode getFlvMuxerAsyncEnabled(ResultContainer<Boolean> isEnabledResult) {
        return GetFlvMuxerAsyncEnabled(getNativeObjectPointer(), isEnabledResult);
    }
}
