package tv.twitch.broadcast;

import tv.twitch.ErrorCode;
import tv.twitch.IJniThreadValidator;
import tv.twitch.NativeProxy;
import tv.twitch.PassThroughJniThreadValidator;

/**
 * A proxy to the native PassThroughAudioCapture implementation.
 * PassThroughAudioCapture is safe to call from any thread, but is not thread-safe, so no two threads must use it at the
 * same time.
 */
public class PassThroughAudioCapture extends NativeProxy implements IAudioCapture {
    public PassThroughAudioCapture() {
        // AND-9927 PassThroughAudioCapture is safe to call from any thread, but is not thread-safe, so no two threads
        // must use it at the same time.
        super(PassThroughJniThreadValidator.INSTANCE);
    }

    // AND-9927 PassThroughAudioCapture is safe to call from any thread, but is not thread-safe, so no two threads must
    // use it at the same time.
    private native long CreateNativeInstance();
    private native void DisposeNativeInstance(long nativeObjectPointer);

    private native ErrorCode Initialize(long nativeObjectPointer);
    private native ErrorCode Shutdown(long nativeObjectPointer);
    private native String GetName(long nativeObjectPointer);
    private native int GetNumChannels(long nativeObjectPointer);
    private native boolean GetMuted(long nativeObjectPointer);
    private native void SetMuted(long nativeObjectPointer, boolean muted);
    private native int GetAudioLayer(long nativeObjectPointer);

    private native ErrorCode SetAudioFormat(long nativeObjectPointer, AudioFormat audioFormat);
    private native ErrorCode SetNumChannels(long nativeObjectPointer, int numChannels);
    private native ErrorCode EnqueueAudioPacket(long nativeObjectPointer, byte[] audioPacket, long timestamp);

    @Override
    protected long createNativeInstance() {
        return CreateNativeInstance();
    }

    @Override
    protected void disposeNativeInstance(long nativeObjectPointer) {
        // AND-9927 Disposing a PassThroughAudioCapture is safe from any thread
        DisposeNativeInstance(nativeObjectPointer);
    }

    @Override
    public ErrorCode initialize() {
        return Initialize(getNativeObjectPointer());
    }

    @Override
    public ErrorCode shutdown() {
        return Shutdown(getNativeObjectPointer());
    }

    @Override
    public String getName() {
        return GetName(getNativeObjectPointer());
    }

    @Override
    public int getNumChannels() {
        return 0;
    }

    @Override
    public boolean getMuted() {
        return false;
    }

    @Override
    public void setMuted(boolean muted) {
        SetMuted(getNativeObjectPointer(), muted);
    }

    @Override
    public int getAudioLayer() {
        return 0;
    }

    public ErrorCode setAudioFormat(AudioFormat audioFormat) {
        return SetAudioFormat(getNativeObjectPointer(), audioFormat);
    }

    public ErrorCode setNumChannels(int numChannels) { return SetNumChannels(getNativeObjectPointer(), numChannels); }

    public ErrorCode enqueueAudioPacket(byte[] audioPacket, long timestamp) {
        return EnqueueAudioPacket(getNativeObjectPointer(), audioPacket, timestamp);
    }
}
