/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/generated/jni_broadcastapi.h"

#include "twitchsdk/broadcast/broadcastapi.h"
#include "twitchsdk/broadcast/generated/java_all.h"
#include "twitchsdk/broadcast/java_bandwidthstatlistenerproxy.h"
#include "twitchsdk/broadcast/java_broadcastapilistenerproxy.h"
#include "twitchsdk/broadcast/java_broadcastutil.h"
#include "twitchsdk/broadcast/java_ingesttesterlistenerproxy.h"
#include "twitchsdk/broadcast/passthroughaudiocapture.h"
#include "twitchsdk/broadcast/passthroughaudioencoder.h"
#include "twitchsdk/broadcast/passthroughvideocapture.h"
#include "twitchsdk/broadcast/passthroughvideoencoder.h"
#include "twitchsdk/core/assertion.h"

#include <functional>

using namespace ttv;
using namespace ttv::broadcast;
using namespace ttv::binding::java;

#define GET_BROADCASTAPI_PTR(x) reinterpret_cast<BroadcastAPI*>(x)

JNIEXPORT jlong JNICALL Java_tv_twitch_broadcast_BroadcastAPI_CreateNativeInstance(JNIEnv* jEnv, jobject jThis) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  // Force load all types to ensure all classes are loaded by the time we need them.
  LoadAllBroadcastJavaClassInfo(jEnv);

  std::shared_ptr<BroadcastApiContext> context = std::make_shared<BroadcastApiContext>();
  context->broadcastApi = std::make_shared<BroadcastAPI>();
  context->nativeListener = std::make_shared<JavaBroadcastAPIListenerProxy>(jThis);

  gBroadcastApiNativeProxyRegistry.Register(context->broadcastApi, context, jThis);

  context->broadcastApi->SetListener(context->nativeListener);

  return reinterpret_cast<jlong>(context->broadcastApi.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_broadcast_BroadcastAPI_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject /*jThis*/, jlong jNativePointer) {
  gBroadcastApiNativeProxyRegistry.Unregister(jNativePointer);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetCoreApi(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCoreApi) {
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);
  auto coreApi = GetCoreApiInstance(jCoreApi);

  TTV_ErrorCode ec;

  if (coreApi != nullptr) {
    ec = api->SetCoreApi(coreApi);
  } else {
    ec = TTV_EC_INVALID_ARG;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    context->nativeListener->SetListener(jListener);
  } else {
    ec = TTV_EC_NOT_INITIALIZED;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetState(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  IModule::State state = api->GetState();

  JavaClassInfo& info = GetJavaClassInfo_ModuleState(jEnv);
  return GetJavaInstance_SimpleEnum(jEnv, info, state);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetModuleName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  std::string name = api->GetModuleName();

  return GetJavaInstance_String(jEnv, name.c_str());
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_Initialize(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_InitializeCallback(jEnv));

  TTV_ErrorCode ec = api->Initialize([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_Shutdown(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_ShutdownCallback(jEnv));

  TTV_ErrorCode ec = api->Shutdown([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_Update(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->Update();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetActiveUser(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetActiveUser(static_cast<UserId>(jUserId));
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetBroadcasterSoftware(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jStr) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jStr, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ScopedJavaUTFStringConverter str(jEnv, jStr);

    ec = api->SetBroadcasterSoftware(str.GetNativeString());
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetForceArchiveBroadcast(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jboolean jForce) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetForceArchiveBroadcast(jForce != JNI_FALSE);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetForceDontArchiveBroadcast(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jboolean jDontArchive) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetForceDontArchiveBroadcast(jDontArchive != JNI_FALSE);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetVideoEncoder(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jIVideoEncoder) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jIVideoEncoder, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // TODO: If we ever need to handle other encoders do it here
    auto encoder = gPassThroughVideoEncoderInstanceRegistry.LookupNativeInstance(jIVideoEncoder);
    if (encoder == nullptr) {
      return GetJavaInstance_ErrorCode(jEnv, TTV_EC_BROADCAST_INVALID_ENCODER);
    }

    ec = api->SetVideoEncoder(encoder);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetAudioEncoder(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jIAudioEncoder) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jIAudioEncoder, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // TODO: If we ever need to handle other encoders do it here
    auto encoder = gPassThroughAudioEncoderInstanceRegistry.LookupNativeInstance(jIAudioEncoder);
    if (encoder == nullptr) {
      return GetJavaInstance_ErrorCode(jEnv, TTV_EC_BROADCAST_INVALID_ENCODER);
    }

    ec = api->SetAudioEncoder(encoder);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetVideoCapturer(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jIVideoCapturer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jIVideoCapturer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // TODO: If we ever need to handle other capturers do it here
    auto capturer = gPassThroughVideoCaptureInstanceRegistry.LookupNativeInstance(jIVideoCapturer);
    if (capturer == nullptr) {
      return GetJavaInstance_ErrorCode(jEnv, TTV_EC_BROADCAST_INVALID_ENCODER);
    }

    ec = api->SetVideoCapturer(capturer);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetAudioCapturer(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jLayer, jobject jIAudioCapturer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jIAudioCapturer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // TODO: If we ever need to handle other capturers do it here
    auto capturer = gPassThroughAudioCaptureInstanceRegistry.LookupNativeInstance(jIAudioCapturer);
    if (capturer == nullptr) {
      return GetJavaInstance_ErrorCode(jEnv, TTV_EC_BROADCAST_INVALID_ENCODER);
    }

    ec = api->SetAudioCapturer(static_cast<AudioLayerId>(jLayer), capturer);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_RemoveAudioCapturer(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jLayer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->RemoveAudioCapturer(static_cast<AudioLayerId>(jLayer));
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetAudioLayerVolume(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jLayer, jfloat jVolume) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetAudioLayerVolume(static_cast<AudioLayerId>(jLayer), jVolume);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetAudioLayerMuted(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jLayer, jboolean jMuted) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetAudioLayerMuted(static_cast<AudioLayerId>(jLayer), jMuted != JNI_FALSE);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetAudioLayerEnabled(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jLayer, jboolean jEnabled) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetAudioLayerEnabled(static_cast<AudioLayerId>(jLayer), jEnabled != JNI_FALSE);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetVideoParams(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jVideoParams) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jVideoParams, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    VideoParams params;
    GetNativeFromJava_VideoParams(jEnv, params, jVideoParams);

    ec = api->SetVideoParams(params);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetVideoParams(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jResultContainer) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    VideoParams result;
    ec = api->GetVideoParams(result);

    if (TTV_SUCCEEDED(ec)) {
      AUTO_DELETE_LOCAL_REF(jEnv, jobject, jVideoParams, GetJavaInstance_VideoParams(gActiveJavaEnvironment, result));

      SetResultContainerResult(jEnv, jResultContainer, jVideoParams);
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetOutputPath(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jPath) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ScopedJavaWcharStringConverter path(jEnv, jPath);

    ec = api->SetOutputPath(path.GetNativeString());
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetConnectionType(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jConnectionType) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  ConnectionType connectionType = GetNativeFromJava_SimpleEnum<ConnectionType>(
    jEnv, GetJavaClassInfo_ConnectionType(jEnv), jConnectionType, ConnectionType::Unknown);
  TTV_ErrorCode ec = api->SetConnectionType(connectionType);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetSessionId(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jSessionId) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter sessionId(jEnv, jSessionId);
  TTV_ErrorCode ec = api->SetSessionId(sessionId.GetNativeString());

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_AddBandwidthStatListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto listenerProxy = std::make_shared<JavaBandwidthStatListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->AddBandwidthStatListener(listenerProxy);
    if (TTV_SUCCEEDED(ec)) {
      context->bandwidthListeners.push_back(listenerProxy);
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_RemoveBandwidthStatListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto iter = std::find_if(context->bandwidthListeners.begin(), context->bandwidthListeners.end(),
      [jEnv, jListener](const std::shared_ptr<JavaBandwidthStatListenerProxy>& listener) {
        return jEnv->IsSameObject(jListener, listener->GetListener());
      });

    if (iter != context->bandwidthListeners.end()) {
      ec = api->RemoveBandwidthStatListener(*iter);
      context->bandwidthListeners.erase(iter);
    } else {
      ec = TTV_EC_INVALID_ARG;
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_StartBroadcast(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto globalRef = std::make_shared<GlobalJavaObjectReference>();
    globalRef->Bind(jEnv, jCallback);

    ec = api->StartBroadcast([globalRef](TTV_ErrorCode callbackEc) {
      jobject obj = globalRef->GetInstance();
      if (obj != nullptr) {
        JavaClassInfo& info = GetJavaClassInfo_StartBroadcastCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        gActiveJavaEnvironment->CallVoidMethod(obj, info.methods["invoke"], jErrorCode);
      }
    });
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_StopBroadcast(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jReason, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto globalRef = std::make_shared<GlobalJavaObjectReference>();
    globalRef->Bind(jEnv, jCallback);

    ScopedJavaUTFStringConverter reason(jEnv, jReason);

    ec = api->StopBroadcast(reason.GetNativeString(), [globalRef](TTV_ErrorCode callbackEc) {
      jobject obj = globalRef->GetInstance();
      if (obj != nullptr) {
        JavaClassInfo& info = GetJavaClassInfo_StopBroadcastCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        gActiveJavaEnvironment->CallVoidMethod(obj, info.methods["invoke"], jErrorCode);
      }
    });
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetCurrentBroadcastTime(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jResultContainer) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    uint64_t time = 0;
    ec = api->GetCurrentBroadcastTime(time);

    if (TTV_SUCCEEDED(ec)) {
      AUTO_DELETE_LOCAL_REF(jEnv, jobject, jTime, GetJavaInstance_Long(gActiveJavaEnvironment, time));

      SetResultContainerResult(jEnv, jResultContainer, jTime);
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_FetchIngestServerList(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto globalRef = std::make_shared<GlobalJavaObjectReference>();
    globalRef->Bind(jEnv, jCallback);

    ec = api->FetchIngestServerList([globalRef](TTV_ErrorCode callbackEc, const std::vector<IngestServer>& result) {
      jobject obj = globalRef->GetInstance();
      if (obj != nullptr) {
        jobjectArray jArray = nullptr;

        if (TTV_SUCCEEDED(callbackEc)) {
          JavaClassInfo& ingestServerInfo = GetJavaClassInfo_IngestServer(gActiveJavaEnvironment);
          jArray = GetJavaInstance_Array(gActiveJavaEnvironment, ingestServerInfo, static_cast<uint32_t>(result.size()),
            [&result](uint32_t index) { return GetJavaInstance_IngestServer(gActiveJavaEnvironment, result[index]); });
        }

        AUTO_DELETE_LOCAL_REF_NO_DECLARE(gActiveJavaEnvironment, jobjectArray, jArray);

        JavaClassInfo& callbackInfo = GetJavaClassInfo_FetchIngestListCallback(gActiveJavaEnvironment);
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        gActiveJavaEnvironment->CallVoidMethod(obj, callbackInfo.methods["invoke"], jErrorCode, jArray);
      }
    });
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetSelectedIngestServer(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    IngestServer server;
    ec = api->GetSelectedIngestServer(server);

    jobject jIngestServer = nullptr;

    if (TTV_SUCCEEDED(ec)) {
      jIngestServer = GetJavaInstance_IngestServer(jEnv, server);
    }

    AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jobject, jIngestServer);

    SetResultContainerResult(jEnv, jResultContainer, jIngestServer);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetSelectedIngestServer(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jIngestServer) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jIngestServer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    IngestServer server;
    GetNativeFromJava_IngestServer(jEnv, server, jIngestServer);

    ec = api->SetSelectedIngestServer(server);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetBroadcastState(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jResultContainer) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    BroadcastState state;
    ec = api->GetBroadcastState(state);

    if (TTV_SUCCEEDED(ec)) {
      AUTO_DELETE_LOCAL_REF(jEnv, jobject, jState, GetJavaInstance_BroadcastState(gActiveJavaEnvironment, state));

      SetResultContainerResult(jEnv, jResultContainer, jState);
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_RunCommercial(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jint jTimebreakSeconds, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto globalRef = std::make_shared<GlobalJavaObjectReference>();
    globalRef->Bind(jEnv, jCallback);

    ec = api->RunCommercial(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
      static_cast<uint32_t>(jTimebreakSeconds), [globalRef](TTV_ErrorCode callbackEc) {
        jobject obj = globalRef->GetInstance();
        if (obj != nullptr) {
          JavaClassInfo& info = GetJavaClassInfo_RunCommercialCallback(gActiveJavaEnvironment);
          AUTO_DELETE_LOCAL_REF(
            gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
          gActiveJavaEnvironment->CallVoidMethod(obj, info.methods["invoke"], jErrorCode);
        }
      });
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetStreamInfo(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jstring jGame, jstring jTitle, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jGame, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jTitle, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ScopedJavaUTFStringConverter game(jEnv, jGame);
    ScopedJavaUTFStringConverter title(jEnv, jTitle);

    auto globalRef = std::make_shared<GlobalJavaObjectReference>();
    globalRef->Bind(jEnv, jCallback);

    ec = api->SetStreamInfo(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), game.GetNativeString(),
      title.GetNativeString(), [globalRef](TTV_ErrorCode callbackEc) {
        jobject obj = globalRef->GetInstance();
        if (obj != nullptr) {
          JavaClassInfo& info = GetJavaClassInfo_SetStreamInfoCallback(gActiveJavaEnvironment);
          AUTO_DELETE_LOCAL_REF(
            gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
          gActiveJavaEnvironment->CallVoidMethod(obj, info.methods["invoke"], jErrorCode);
        }
      });
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_CreateIngestTester(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jobject jJniThreadValidator, jint jUserId, jobject jListener, jbyteArray sampleData,
  jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IIngestTester> ingestTester;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  auto ingestTesterContext = std::make_shared<IngestTesterContext>();

  if (TTV_SUCCEEDED(ec)) {
    ingestTesterContext->nativeListener = std::make_shared<JavaIngestTesterListenerProxy>();
    ingestTesterContext->nativeListener->SetListener(jListener);

    jbyte* sampleDataBuffer = jEnv->GetByteArrayElements(sampleData, nullptr);
    jsize sampleDataLength = jEnv->GetArrayLength(sampleData);

    ec = api->CreateIngestTester(static_cast<UserId>(jUserId), ingestTesterContext->nativeListener,
      reinterpret_cast<const uint8_t*>(sampleDataBuffer), static_cast<uint32_t>(sampleDataLength), ingestTester);
    ingestTesterContext->instance = ingestTester;
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(ingestTester != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_IngestTesterProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jIngestTesterProxy,
      jEnv->NewObject(
        info.klass, info.methods["<init>"], reinterpret_cast<jlong>(ingestTester.get()), jJniThreadValidator));
    SetResultContainerResult(jEnv, jResultContainer, jIngestTesterProxy);

    gIngestTesterInstanceRegistry.Register(ingestTester, ingestTesterContext, jIngestTesterProxy);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_SetFlvMuxerAsyncEnabled(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jboolean jEnabled) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->SetFlvMuxerAsyncEnabled(jEnabled != JNI_FALSE);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_BroadcastAPI_GetFlvMuxerAsyncEnabled(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jResultContainer) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_BROADCASTAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gBroadcastApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    bool isEnabledResult = false;
    ec = api->GetFlvMuxerAsyncEnabled(isEnabledResult);

    if (TTV_SUCCEEDED(ec)) {
      AUTO_DELETE_LOCAL_REF(jEnv, jobject, jEnabled, GetJavaInstance_Boolean(gActiveJavaEnvironment, isEnabledResult));

      SetResultContainerResult(jEnv, jResultContainer, jEnabled);
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
