/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/broadcastapi.h"
#include "twitchsdk/broadcast/generated/java_all.h"
#include "twitchsdk/broadcast/generated/jni_ingesttesterproxyimpl.h"
#include "twitchsdk/broadcast/iingesttester.h"
#include "twitchsdk/broadcast/java_broadcastutil.h"

using namespace ttv;
using namespace ttv::broadcast;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<IIngestTester*>(x)

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetTestState(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  IIngestTester::TestState state;
  TTV_ErrorCode ec = instance->GetTestState(state);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult,
      GetJavaInstance_SimpleEnum<IIngestTester::TestState>(jEnv, GetJavaClassInfo_IngestTesterState(jEnv), state));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetIngestServer(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  IngestServer server;
  TTV_ErrorCode ec = instance->GetIngestServer(server);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_IngestServer(jEnv, server));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_Start(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jIngestServer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jIngestServer, TTV_EC_INVALID_ARG);

  IngestServer server;
  GetNativeFromJava_IngestServer(jEnv, server, jIngestServer);

  TTV_ErrorCode ec = instance->Start(server);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_Cancel(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_ErrorCode ec = instance->Cancel();

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetMeasuredKbps(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  uint32_t kbps = 0;
  TTV_ErrorCode ec = instance->GetMeasuredKbps(kbps);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_Integer(jEnv, kbps));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetTestError(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode testErrorCode = TTV_EC_SUCCESS;
  TTV_ErrorCode ec = instance->GetTestError(testErrorCode);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_ErrorCode(jEnv, testErrorCode));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_SetTestDurationMilliseconds(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jlong jMilliseconds) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_ErrorCode ec = instance->SetTestDurationMilliseconds(static_cast<uint64_t>(jMilliseconds));

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetTestDurationMilliseconds(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  uint64_t milliseconds = 0;
  TTV_ErrorCode ec = instance->GetTestDurationMilliseconds(milliseconds);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_Long(jEnv, milliseconds));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetProgress(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  float progress = 0;
  TTV_ErrorCode ec = instance->GetProgress(progress);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_Float(jEnv, progress));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_GetUserId(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  UserId userId = 0;
  TTV_ErrorCode ec = instance->GetUserId(userId);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_Integer(jEnv, userId));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT void JNICALL Java_tv_twitch_broadcast_IngestTesterProxy_DisposeNativeInstance(
  JNIEnv* jEnv, jobject jThis, jlong jNativeObjectPointer) {
  auto context = gIngestTesterInstanceRegistry.LookupNativeContext(jThis);
  if (context == nullptr) {
    return;
  }

  auto instance = gIngestTesterInstanceRegistry.LookupNativeInstance(jThis);
  if (instance == nullptr) {
    return;
  }

  // This should release the last reference to the native component and trigger shutdown
  gIngestTesterInstanceRegistry.Unregister(jThis);
}
