/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/generated/jni_passthroughaudiocapture.h"

#include "twitchsdk/broadcast/generated/java_all.h"
#include "twitchsdk/broadcast/java_broadcastutil.h"
#include "twitchsdk/broadcast/passthroughaudiocapture.h"

#include <functional>

using namespace ttv;
using namespace ttv::broadcast;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<PassThroughAudioCapture*>(x)

JNIEXPORT jlong JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_CreateNativeInstance(
  JNIEnv* jEnv, jobject jThis) {
  auto context = std::make_shared<ProxyContext<PassThroughAudioCapture>>();
  context->instance = std::make_shared<PassThroughAudioCapture>();

  gPassThroughAudioCaptureInstanceRegistry.Register(context->instance, context, jThis);

  return reinterpret_cast<jlong>(context->instance.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_DisposeNativeInstance(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  gPassThroughAudioCaptureInstanceRegistry.Unregister(jNativePointer);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_Initialize(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->Initialize();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_Shutdown(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->Shutdown();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_GetName(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto name = capturer->GetName();
    return GetJavaInstance_String(jEnv, name);
  } else {
    return nullptr;
  }
}

JNIEXPORT jint JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_GetNumChannels(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    return static_cast<jint>(capturer->GetNumChannels());
  } else {
    return 0;
  }
}

JNIEXPORT jboolean JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_GetMuted(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    return capturer->GetMuted() == JNI_TRUE;
  } else {
    return false;
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_SetMuted(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer, jboolean jMuted) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    capturer->SetMuted(jMuted == JNI_TRUE);
  }
}

JNIEXPORT jint JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_GetAudioLayer(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    return static_cast<jint>(capturer->GetAudioLayer());
  } else {
    return 0;
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_SetAudioFormat(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer, jobject jAudioFormat) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jAudioFormat, TTV_EC_INVALID_ARG);

  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    AudioFormat audioFormat =
      GetNativeFromJava_SimpleEnum(jEnv, GetJavaClassInfo_AudioFormat(jEnv), jAudioFormat, AudioFormat::None);

    ec = capturer->SetAudioFormat(audioFormat);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_SetNumChannels(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer, jint jNumChannels) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->SetNumChannels(static_cast<uint32_t>(jNumChannels));
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioCapture_EnqueueAudioPacket(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer, jbyteArray jAudioPacket, jlong jTimestamp) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jAudioPacket, TTV_EC_INVALID_ARG);

  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioCaptureInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // Bring the bytes into native
    std::vector<uint8_t> audioPacket;
    GetNativeFromJava_ByteArray(jEnv, jAudioPacket, audioPacket);

    // Enqueue the packet
    ec = capturer->EnqueueAudioPacket(std::move(audioPacket), static_cast<uint64_t>(jTimestamp));
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
