/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/generated/jni_passthroughaudioencoder.h"

#include "twitchsdk/broadcast/generated/java_all.h"
#include "twitchsdk/broadcast/java_broadcastutil.h"
#include "twitchsdk/broadcast/passthroughaudioencoder.h"

#include <functional>

using namespace ttv;
using namespace ttv::broadcast;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<PassThroughAudioEncoder*>(x)

JNIEXPORT jlong JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_CreateNativeInstance(
  JNIEnv* jEnv, jobject jThis) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto context = std::make_shared<ProxyContext<PassThroughAudioEncoder>>();
  context->instance = std::make_shared<PassThroughAudioEncoder>();

  gPassThroughAudioEncoderInstanceRegistry.Register(context->instance, context, jThis);

  return reinterpret_cast<jlong>(context->instance.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_DisposeNativeInstance(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  gPassThroughAudioEncoderInstanceRegistry.Unregister(jNativePointer);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_GetName(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughAudioEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto name = capturer->GetName();
    return GetJavaInstance_String(jEnv, name);
  } else {
    return nullptr;
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_Initialize(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->Initialize();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_Shutdown(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->Shutdown();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_SetAudioFormat(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer, jobject jAudioFormat) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jAudioFormat, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    AudioFormat audioFormat =
      GetNativeFromJava_SimpleEnum(jEnv, GetJavaClassInfo_AudioFormat(jEnv), jAudioFormat, AudioFormat::None);

    ec = capturer->SetAudioFormat(audioFormat);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughAudioEncoder_SetSamplesPerFrame(
  JNIEnv* jEnv, jobject jThis, jlong jNativePointer, jint jSamplesPerFrame) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughAudioEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->SetSamplesPerFrame(static_cast<uint32_t>(jSamplesPerFrame));
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
