/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/broadcast/internal/pch.h"

#include "twitchsdk/broadcast/generated/jni_passthroughvideoencoder.h"

#include "twitchsdk/broadcast/generated/java_all.h"
#include "twitchsdk/broadcast/java_broadcastutil.h"
#include "twitchsdk/broadcast/passthroughvideoencoder.h"
#include "twitchsdk/core/generated/java_errorcode.h"

#include <functional>

using namespace ttv;
using namespace ttv::broadcast;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<PassThroughVideoEncoder*>(x)

JNIEXPORT jlong JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_CreateNativeInstance(
  JNIEnv* /*jEnv*/, jobject jThis) {
  std::shared_ptr<PassThroughVideoEncoderContext> context = std::make_shared<PassThroughVideoEncoderContext>();
  context->instance = std::make_shared<PassThroughVideoEncoder>();

  gPassThroughVideoEncoderInstanceRegistry.Register(context->instance, context, jThis);

  return reinterpret_cast<jlong>(context->instance.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject /*jThis*/, jlong jNativePointer) {
  gPassThroughVideoEncoderInstanceRegistry.Unregister(jNativePointer);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_GetName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  auto context = gPassThroughVideoEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    auto name = capturer->GetName();
    return GetJavaInstance_String(jEnv, name);
  } else {
    return nullptr;
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_Initialize(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughVideoEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->Initialize();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_Shutdown(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  auto capturer = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughVideoEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = capturer->Shutdown();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_SetSps(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jbyteArray jSps) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jSps, TTV_EC_INVALID_ARG);

  auto encoder = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughVideoEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // Bring the bytes into native
    std::vector<uint8_t> sps;
    GetNativeFromJava_ByteArray(jEnv, jSps, sps);

    // Set the value
    ec = encoder->SetSps(sps);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_SetPps(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jbyteArray jPps) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jPps, TTV_EC_INVALID_ARG);

  auto encoder = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughVideoEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // Bring the bytes into native
    std::vector<uint8_t> pps;
    GetNativeFromJava_ByteArray(jEnv, jPps, pps);

    // Set the value
    ec = encoder->SetPps(pps);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_broadcast_PassThroughVideoEncoder_SetAdjustTargetBitRateFunc(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  auto encoder = GET_NATIVE_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gPassThroughVideoEncoderInstanceRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    if (jCallback != nullptr) {
      auto callbackReference = std::make_shared<GlobalJavaObjectReference>();
      callbackReference->Bind(jEnv, jCallback);

      ec = encoder->SetAdjustTargetBitRateFunc([callbackReference](uint32_t kbps) -> TTV_ErrorCode {
        AutoJEnv jCallbackEnv;

        auto callback = callbackReference->GetInstance();
        if (callback != nullptr) {
          auto callbackClassInfo = GetJavaClassInfo_PassThroughVideoEncoder_AdjustTargetBitRateFunc(jCallbackEnv);
          jobject jErrorCode =
            jCallbackEnv->CallObjectMethod(callback, callbackClassInfo.methods["invoke"], static_cast<jint>(kbps));

          AUTO_DELETE_LOCAL_REF_NO_DECLARE(jCallbackEnv, jobject, jErrorCode);

          return GetNativeFromJava_SimpleEnum<TTV_ErrorCode>(
            jCallbackEnv, GetJavaClassInfo_ErrorCode(jCallbackEnv), jErrorCode, TTV_EC_UNKNOWN_ERROR);
        } else {
          return TTV_EC_UNSUPPORTED;
        }
      });
    } else {
      ec = encoder->SetAdjustTargetBitRateFunc(nullptr);
    }
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
