package tv.twitch.test;

import java.io.File;
import java.util.*;
import tv.twitch.*;
import tv.twitch.UserInfo;
import tv.twitch.broadcast.*;
import tv.twitch.broadcast.callbacks.FetchIngestListCallback;
import tv.twitch.broadcast.callbacks.StartBroadcastCallback;
import tv.twitch.broadcast.callbacks.StopBroadcastCallback;

public class BroadcastTest extends TestBase {
    private static Frame[] loadTimestampFilesFromDirectory(String dir) {
        java.io.File directory = new File(dir);
        if (!directory.isDirectory()) {
            return null;
        }

        List<Frame> result = new ArrayList<Frame>();

        File[] files = directory.listFiles();
        for (File file : files) {
            long timestamp = -1;
            try {
                timestamp = Long.valueOf(file.getName());
                Frame frame = new Frame();
                frame.timestamp = timestamp;
                frame.packet = loadFileAsBytes(dir, file.getName());
                result.add(frame);
            } catch (Exception ex) {
                // Not a timestamp file
            }
        }

        // Sort by increasing timestamp
        result.sort(new Comparator<Frame>() {
            @Override
            public int compare(Frame a, Frame b) {
                if (a.timestamp < b.timestamp)
                    return -1;
                else if (a.timestamp > b.timestamp)
                    return 1;
                else
                    return 0;
            }
        });

        return result.toArray(new Frame[result.size()]);
    }

    private static class Frame {
        public byte[] packet;
        public long timestamp;
    }

    private static byte[] appendStartCode(byte[] bytes, boolean onlyIfNeeded) {
        if (onlyIfNeeded && bytes.length >= 4) {
            if (bytes[0] == 0 && bytes[1] == 0 && bytes[2] == 0 && bytes[3] == 1) {
                return bytes;
            }
        }

        byte[] startCode = new byte[] {0x00, 0x00, 0x00, 0x01};
        byte[] combined = new byte[bytes.length + startCode.length];
        for (int i = 0; i < combined.length; ++i) {
            combined[i] = i < startCode.length ? startCode[i] : bytes[i - startCode.length];
        }

        return combined;
    }

    /**
     * Offset all timestamps to be relative to 0 and convert microseconds to milliseconds
     */
    private static void adjustTimestamps(Frame[] files) {
        long startTimestamp = files[0].timestamp;
        for (Frame file : files) {
            file.timestamp = (file.timestamp - startTimestamp) / 1000;
        }
    }

    private static class PreEncodedVideoData {
        public class VideoFrame {
            public Frame frame;
            public boolean keyframe;
        }

        public int width;
        public int height;
        public byte[] sps;
        public byte[] pps;
        public List<VideoFrame> frames = new ArrayList<>();

        public PreEncodedVideoData(String directoryPath) {
            // TODO: This should be loaded as well
            width = 720;
            height = 720;

            // Load SPS
            sps = loadFileAsBytes(directoryPath, "sps");
            sps = appendStartCode(sps, true);

            // Load PPS
            pps = loadFileAsBytes(directoryPath, "pps");
            pps = appendStartCode(pps, true);

            Frame[] files = loadTimestampFilesFromDirectory(directoryPath);

            adjustTimestamps(files);

            // Load frames
            for (Frame file : files) {
                VideoFrame videoFrame = new VideoFrame();
                videoFrame.frame = file;
                videoFrame.keyframe = file.packet[4] == 0x65;

                frames.add(videoFrame);
            }
        }
    }

    private static class PreEncodedAudioData {
        public class AudioFrame { public Frame frame; }

        public AudioFormat audioFormat;
        public ArrayList<AudioFrame> frames = new ArrayList<>();

        public PreEncodedAudioData(String directoryPath) {
            // TODO: This should be loaded as well
            audioFormat = AudioFormat.AAC;

            Frame[] files = loadTimestampFilesFromDirectory(directoryPath);

            adjustTimestamps(files);

            // Load frames
            for (Frame file : files) {
                AudioFrame audioFrame = new AudioFrame();
                audioFrame.frame = file;

                frames.add(audioFrame);
            }
        }
    }

    private class BroadcastApiState {
        public ModuleState moduleState = ModuleState.Uninitialized;
        public BroadcastState broadcastState = BroadcastState.Initialized;
        public boolean ingestServersFetched = false;
        public IngestServer[] ingestServers;
    }

    public BroadcastTest(Library library) {
        super(library);
        BroadcastErrorCode.forceClassInit();
    }

    @Override
    public void run() throws Exception {
        System.out.println("Running BroadcastTest tests...");

        // Uncomment these below to test

        // Interfaces
        test_IBroadcastAPIListener();
        // test_IIngesterTesterListener();
        // test_IBandwidthStatListener();

        // Classes
        // test_BroadcastAPI();
        // test_PassThroughVideoCapture();
        // test_PassThroughVideoEncoder();
        // test_PassThroughAudioCapture();
        // test_PassThroughAudioEncoder();
        // test_VideoParams();
        // test_PassThroughBroadcast();

        // TODO: Fix ingest testing since it requires a VideoEncoder
        // test_IngestTester();

        System.out.println("Done running BroadcastTest tests...");
    }

    protected IBroadcastAPIListener getDefaultBroadcastApiListener() {
        return new IBroadcastAPIListener() {
            @Override
            public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {}

            @Override
            public void broadcastStateChanged(ErrorCode ec, BroadcastState state) {}

            @Override
            public void broadcastBandwidthWarning(ErrorCode ec, int backupMilliseconds) {}

            @Override
            public void broadcastFrameSubmissionIssue(ErrorCode ec) {}

            @Override
            public void streamInfoFetched(ErrorCode ec, StreamInfo streamInfo) {}

            @Override
            public void streamKeyError(CanTheyError canTheyError) {}
        };
    }

    protected ErrorCode setup(String oauthToken, final IBroadcastAPIListener broadcastApiListener,
        final ResultContainer<CoreAPI> coreApi, final ResultContainer<BroadcastAPI> broadcastApi,
        final ResultContainer<UserInfo> userInfo) throws InterruptedException {
        ErrorCode ec = CoreErrorCode.TTV_EC_SUCCESS;

        userInfo.result = new UserInfo();

        ICoreAPIListener coreApiListener = new ICoreAPIListener() {
            @Override
            public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {}

            @Override
            public void coreUserLoginComplete(String oauthToken, int userId, ErrorCode ec) {
                userInfo.result.userId = userId;
            }

            @Override
            public void coreUserLogoutComplete(int userId, ErrorCode ec) {}

            @Override
            public void coreUserAuthenticationIssue(int userId, String oauthToken, ErrorCode ec) {}

            @Override
            public void corePubSubStateChanged(int userId, CorePubSubState state, ErrorCode result) {}
        };

        coreApi.result = new CoreAPI();
        coreApi.result.setListener(coreApiListener);
        initializeModule(coreApi.result);
        addModule(coreApi.result);

        broadcastApi.result = new BroadcastAPI();
        broadcastApi.result.setCoreApi(coreApi.result);
        broadcastApi.result.setListener(broadcastApiListener);
        initializeModule(broadcastApi.result);
        addModule(broadcastApi.result);

        // Log in a user
        if (oauthToken != null) {
            ec = coreApi.result.logIn(oauthToken, null);
            if (ec.succeeded()) {
                while (userInfo.result.userId == 0) {
                    updateModules();
                    Thread.sleep(100);
                }
            }
        }

        return ec;
    }

    protected void teardown(CoreAPI coreApi, BroadcastAPI broadcastApi) {
        shutdownModule(broadcastApi);
        removeModule(broadcastApi);

        shutdownModule(coreApi);
        removeModule(coreApi);

        broadcastApi.dispose();
        coreApi.dispose();
    }

    private native void Test_IBroadcastAPIListener(IModule module, IBroadcastAPIListener listener);
    private native void Test_IIngestTesterListener(IModule module, IIngestTesterListener listener);
    private native void Test_PassThroughVideoEncoder(PassThroughVideoEncoder videoEncoder);
    private native void Test_IBandwidthStatListener(IModule module, IBandwidthStatListener listener);

    protected void test_IBroadcastAPIListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        IBroadcastAPIListener listener = new IBroadcastAPIListener() {
            @Override
            public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void broadcastStateChanged(ErrorCode ec, BroadcastState state) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void broadcastBandwidthWarning(ErrorCode ec, int backupMilliseconds) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void broadcastFrameSubmissionIssue(ErrorCode ec) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamInfoFetched(ErrorCode ec, StreamInfo streamInfo) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void streamKeyError(CanTheyError canTheyError) {
                calls.add(getCurrentMethodName());
            }
        };

        // Ask native to call each method on our interface
        Test_IBroadcastAPIListener(mDummyModule, listener);

        // Make sure they were all called
        checkAllMethodsCalled(IBroadcastAPIListener.class, calls);
    }

    protected void test_IIngesterTesterListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        IIngestTesterListener listener = new IIngestTesterListener() {
            @Override
            public void ingestTesterStateChanged() {
                calls.add(getCurrentMethodName());
            }
        };

        // Ask native to call each method on our interface
        Test_IIngestTesterListener(mDummyModule, listener);

        // Make sure they were all called
        checkAllMethodsCalled(IIngestTesterListener.class, calls);
    }

    protected void test_IBandwidthStatListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        IBandwidthStatListener listener = new IBandwidthStatListener() {
            @Override
            public void receivedBandwidthStat(BandwidthStat stat) {
                calls.add(getCurrentMethodName());
            }
        };

        // Ask native to call each method on our interface
        Test_IBandwidthStatListener(mDummyModule, listener);

        // Make sure they were all called
        checkAllMethodsCalled(IBandwidthStatListener.class, calls);
    }

    protected void test_BroadcastAPI() throws Exception {
        class CoreApiState {
            public boolean initialized = false;
        }

        final CoreApiState coreApiState = new CoreApiState();

        CoreAPI coreApi = null;
        BroadcastAPI broadcastApi = null;

        try {
            ICoreAPIListener coreApiListener = new ICoreAPIListener() {
                @Override
                public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
                    coreApiState.initialized = state == ModuleState.Initialized || state == ModuleState.ShuttingDown;
                }

                @Override
                public void coreUserLoginComplete(String oauthToken, int userId, ErrorCode ec) {}

                @Override
                public void coreUserLogoutComplete(int userId, ErrorCode ec) {}

                @Override
                public void coreUserAuthenticationIssue(int userId, String oauthToken, ErrorCode ec) {}

                @Override
                public void corePubSubStateChanged(int userId, CorePubSubState state, ErrorCode result) {}
            };

            coreApi = new CoreAPI();
            addModule(coreApi);

            coreApi.setListener(coreApiListener);

            ErrorCode ec = coreApi.initialize(null);

            while (!coreApiState.initialized) {
                Thread.sleep(100);
                updateModules();
            }

            final BroadcastApiState broadcastApiState = new BroadcastApiState();

            IBroadcastAPIListener broadcastApiListener = new IBroadcastAPIListener() {
                @Override
                public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
                    broadcastApiState.moduleState = state;
                }

                @Override
                public void broadcastStateChanged(ErrorCode ec, BroadcastState state) {
                    broadcastApiState.broadcastState = state;
                }

                @Override
                public void broadcastBandwidthWarning(ErrorCode ec, int backupMilliseconds) {}

                @Override
                public void broadcastFrameSubmissionIssue(ErrorCode ec) {}

                @Override
                public void streamInfoFetched(ErrorCode ec, StreamInfo streamInfo) {}

                @Override
                public void streamKeyError(CanTheyError canTheyError) {}
            };

            broadcastApi = new BroadcastAPI();
            addModule(broadcastApi);

            ec = broadcastApi.setCoreApi(coreApi);
            ec = broadcastApi.setListener(broadcastApiListener);

            ec = broadcastApi.initialize(null);
            while (broadcastApiState.moduleState != ModuleState.Initialized) {
                Thread.sleep(100);
                updateModules();
            }

            VideoParams videoParams = new VideoParams();
            videoParams.automaticBitRateAdjustmentEnabled = true;
            videoParams.encodingCpuUsage = EncodingCpuUsage.Low;
            videoParams.initialKbps = 789;
            videoParams.maximumKbps = 2000;
            videoParams.minimumKbps = 300;
            videoParams.outputHeight = 111;
            videoParams.outputWidth = 222;
            videoParams.targetFramesPerSecond = 30;

            ec = broadcastApi.setVideoParams(videoParams);

            ec = broadcastApi.setConnectionType(ConnectionType.Wifi);
            ec = broadcastApi.setSessionId("test_session_id");

            ec = broadcastApi.fetchIngestServerList(new FetchIngestListCallback() {
                @Override
                public void invoke(ErrorCode ec, IngestServer[] result) {
                    broadcastApiState.ingestServers = result;
                    broadcastApiState.ingestServersFetched = true;
                }
            });

            if (ec.failed()) {
                throw new Exception();
            }

            while (!broadcastApiState.ingestServersFetched) {
                Thread.sleep(100);
                updateModules();
            }

            // TODO: More tests

            ec = broadcastApi.shutdown(null);
            while (broadcastApiState.moduleState != ModuleState.Uninitialized) {
                Thread.sleep(100);
                updateModules();
            }
            ec = coreApi.shutdown(null);
            while (coreApiState.initialized) {
                Thread.sleep(100);
                updateModules();
            }
        } finally {
            if (broadcastApi != null) {
                broadcastApi.dispose();
            }
            if (coreApi != null) {
                coreApi.dispose();
            }
            clearModules();
        }
    }

    protected void test_PassThroughVideoCapture() throws Exception {
        PassThroughVideoCapture instance = new PassThroughVideoCapture();

        String name = instance.getName();
        ErrorCode ec = instance.initialize();

        byte[] packet = new byte[] {'a', 'b', 'c', 'd', 'e', 'f', 'g'};
        instance.enqueueVideoPacket(packet, true, 123456);

        ec = instance.shutdown();

        instance.dispose();
        instance = null;
    }

    protected void test_PassThroughVideoEncoder() throws Exception {
        PassThroughVideoEncoder instance = new PassThroughVideoEncoder();

        String name = instance.getName();
        ErrorCode ec = instance.initialize();

        byte[] sps = new byte[] {'a', 'b', 'c', 'd', 'e', 'f', 'g'};
        instance.setSps(sps);

        byte[] pps = new byte[] {'1', '2', '3', '4'};
        instance.setPps(pps);

        Test_PassThroughVideoEncoder(instance);

        final ResultContainer<Boolean> callbackResult = new ResultContainer<>();
        callbackResult.result = Boolean.FALSE;

        instance.setAdjustTargetBitRateFunc(new PassThroughVideoEncoder.AdjustTargetBitRateFunc() {
            @Override
            public ErrorCode invoke(int kbps) {
                callbackResult.result = true;
                return CoreErrorCode.TTV_EC_FORBIDDEN;
            }
        });

        Test_PassThroughVideoEncoder(instance);

        ec = instance.shutdown();

        instance.dispose();
        instance = null;
    }

    protected void test_PassThroughAudioCapture() throws Exception {
        PassThroughAudioCapture instance = new PassThroughAudioCapture();

        String name = instance.getName();
        ErrorCode ec = instance.initialize();

        int audioLayer = instance.getAudioLayer();
        boolean muted = instance.getMuted();
        instance.setMuted(true);

        instance.setAudioFormat(AudioFormat.AAC);
        instance.setNumChannels(1);

        byte[] packet = new byte[] {'a', 'b', 'c', 'd', 'e', 'f', 'g'};
        instance.enqueueAudioPacket(packet, 123456);

        ec = instance.shutdown();

        instance.dispose();
        instance = null;
    }

    protected void test_PassThroughAudioEncoder() throws Exception {
        PassThroughAudioEncoder instance = new PassThroughAudioEncoder();

        String name = instance.getName();
        ErrorCode ec = instance.initialize();

        instance.setAudioFormat(AudioFormat.AAC);
        instance.setSamplesPerFrame(11223344);

        ec = instance.shutdown();

        instance.dispose();
        instance = null;
    }

    protected void test_VideoParams() throws Exception {
        ResultContainer<VideoParams> result = new ResultContainer<>();

        ErrorCode ec = VideoParams.configureForBandwidth(2000, 30, Constants.kRecommendedBitsPerPixel, 1.0f, result);
        if (ec.failed()) {
            throw new Exception();
        }
        result.result = null;

        ec = VideoParams.configureForResolution(1024, 768, 30, Constants.kRecommendedBitsPerPixel, result);
        if (ec.failed()) {
            throw new Exception();
        }
    }

    protected void test_PassThroughBroadcast() throws Exception {
        IBroadcastAPIListener broadcastApiListener = getDefaultBroadcastApiListener();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        ResultContainer<BroadcastAPI> broadcastApiResult = new ResultContainer<>();
        ResultContainer<UserInfo> userInfoResult = new ResultContainer<>();
        ErrorCode ec = setup(m_OauthToken, broadcastApiListener, coreApiResult, broadcastApiResult, userInfoResult);

        CoreAPI coreApi = coreApiResult.result;
        BroadcastAPI broadcastApi = broadcastApiResult.result;
        UserInfo userInfo = userInfoResult.result;

        ec = broadcastApi.setActiveUser(userInfo.userId);

        // ec = broadcastApi.setOutputPath("D:\\Junk\\Android.flv");

        PreEncodedVideoData videoFrames = new PreEncodedVideoData("C:\\Drew\\TestData\\Android\\2sec\\video");
        PreEncodedAudioData audioFrames = new PreEncodedAudioData("C:\\Drew\\TestData\\Android\\2sec\\audio");

        // Setup the video capturer
        PassThroughVideoCapture videoCapturer = new PassThroughVideoCapture();
        ec = videoCapturer.initialize();
        ec = broadcastApi.setVideoCapturer(videoCapturer);

        // Setup the video encoder
        PassThroughVideoEncoder videoEncoder = new PassThroughVideoEncoder();
        videoEncoder.setSps(videoFrames.sps);
        videoEncoder.setPps(videoFrames.pps);
        ec = videoEncoder.initialize();
        ec = broadcastApi.setVideoEncoder(videoEncoder);

        boolean enableAudio = true;

        // Setup the audio capturer
        final int passthroughAudioLayerId = 1;
        PassThroughAudioCapture audioCapturer = null;
        PassThroughAudioEncoder audioEncoder = null;

        if (enableAudio) {
            audioCapturer = new PassThroughAudioCapture();
            ec = audioCapturer.initialize();
            audioCapturer.setAudioFormat(AudioFormat.AAC);
            audioCapturer.setNumChannels(1);
            ec = broadcastApi.setAudioCapturer(passthroughAudioLayerId, audioCapturer);

            // Setup the audio encoder
            audioEncoder = new PassThroughAudioEncoder();
            ec = audioEncoder.initialize();
            audioEncoder.setAudioFormat(AudioFormat.AAC);
            ec = broadcastApi.setAudioEncoder(audioEncoder);
        }

        ResultContainer<VideoParams> videoParamsContainer = new ResultContainer<>();
        ec = VideoParams.configureForResolution(
            videoFrames.width, videoFrames.height, 30, Constants.kRecommendedBitsPerPixel, videoParamsContainer);
        ec = broadcastApi.setVideoParams(videoParamsContainer.result);

        IngestServer server = new IngestServer();
        server.serverName = "SFO";
        server.serverUrl = "rtmp://live.twitch.tv/app/{stream_key}";
        ec = broadcastApi.setSelectedIngestServer(server);

        // Try and start the broadcast
        {
            final ResultContainer<ErrorCode> callbackResult = new ResultContainer<>();

            ec = broadcastApi.startBroadcast(new StartBroadcastCallback() {
                @Override
                public void invoke(ErrorCode ec) {
                    callbackResult.result = ec;
                }
            });

            if (ec.succeeded()) {
                while (callbackResult.result == null) {
                    updateModules();
                }

                ec = callbackResult.result;
            }
        }

        m_Library.setComponentMessageLevel("VideoFrameQueue", MessageLevel.TTV_ML_DEBUG);

        // Submit all the frames
        if (ec.succeeded()) {
            int videoFrameIndex = 0;
            int audioFrameIndex = 0;

            double videoFramesPerSecond = 30.0;
            double millisecondsPerVideoFrame = 1000.0 / videoFramesPerSecond;

            boolean loop = false;
            long startSystemTime = m_Library.getSystemClockTime();
            long videoBaseMilliseconds = 0;
            long audioBaseMilliseconds = 0;

            for (;;) {
                updateModules();

                ResultContainer<BroadcastState> broadcastStateResult = new ResultContainer<>();
                ec = broadcastApi.getBroadcastState(broadcastStateResult);
                if (broadcastStateResult.result != BroadcastState.Broadcasting) {
                    break;
                }

                // Submit a video frame
                PreEncodedVideoData.VideoFrame videoFrame =
                    videoFrames.frames.get(videoFrameIndex % videoFrames.frames.size());
                long videoTimestampMilliseconds = videoBaseMilliseconds + videoFrame.frame.timestamp;
                long videoTimestampSystemTime = startSystemTime + m_Library.msToSystemTime(videoTimestampMilliseconds);

                ec = videoCapturer.enqueueVideoPacket(
                    videoFrame.frame.packet, videoFrame.keyframe, videoTimestampSystemTime);

                // Submit all pending audio
                if (ec.succeeded() && enableAudio) {
                    for (;;) {
                        PreEncodedAudioData.AudioFrame audioFrame = audioFrames.frames.get(audioFrameIndex);

                        long audioTimestampMilliseconds = audioBaseMilliseconds + audioFrame.frame.timestamp;

                        if (videoTimestampMilliseconds >= audioTimestampMilliseconds) {
                            ec = audioCapturer.enqueueAudioPacket(audioFrame.frame.packet, audioTimestampMilliseconds);
                            audioFrameIndex++;
                        } else {
                            break;
                        }

                        if (ec.failed()) {
                            break;
                        }

                        // Loop the audio
                        if (audioFrameIndex == audioFrames.frames.size()) {
                            long audioTimestampDelta =
                                audioFrames.frames.get(1).frame.timestamp - audioFrames.frames.get(0).frame.timestamp;
                            audioBaseMilliseconds =
                                audioFrames.frames.get(audioFrames.frames.size() - 1).frame.timestamp
                                + audioTimestampDelta;
                            audioFrameIndex = 0;
                        }
                    }
                }

                if (ec.succeeded()) {
                    Thread.sleep((long) millisecondsPerVideoFrame);
                } else {
                    break;
                }

                videoFrameIndex++;

                if (videoFrameIndex == videoFrames.frames.size()) {
                    if (loop) {
                        long videoTimestampDelta =
                            videoFrames.frames.get(1).frame.timestamp - videoFrames.frames.get(0).frame.timestamp;
                        videoBaseMilliseconds +=
                            videoFrames.frames.get(videoFrames.frames.size() - 1).frame.timestamp + videoTimestampDelta;
                        videoFrameIndex = 0;
                    } else {
                        break;
                    }
                }
            }
        }

        // Stop the broadcast
        {
            final ResultContainer<ErrorCode> callbackResult = new ResultContainer<>();

            ec = broadcastApi.stopBroadcast("user_ended", new StopBroadcastCallback() {
                @Override
                public void invoke(ErrorCode ec) {
                    callbackResult.result = ec;
                }
            });

            if (ec.succeeded()) {
                while (callbackResult.result == null) {
                    updateModules();
                }

                ec = callbackResult.result;
            }
        }

        ec = videoEncoder.shutdown();
        ec = videoCapturer.shutdown();
        videoEncoder.dispose();
        videoCapturer.dispose();

        if (enableAudio) {
            ec = audioEncoder.shutdown();
            ec = audioCapturer.shutdown();
            audioEncoder.dispose();
            audioCapturer.dispose();
        }

        teardown(coreApi, broadcastApi);
    }

    protected void test_IngestTester() throws Exception {
        IBroadcastAPIListener broadcastApiListener = getDefaultBroadcastApiListener();

        ResultContainer<CoreAPI> coreApiResult = new ResultContainer<>();
        ResultContainer<BroadcastAPI> broadcastApiResult = new ResultContainer<>();
        ResultContainer<UserInfo> userInfoResult = new ResultContainer<>();
        ErrorCode ec = setup(m_OauthToken, broadcastApiListener, coreApiResult, broadcastApiResult, userInfoResult);

        CoreAPI coreApi = coreApiResult.result;
        BroadcastAPI broadcastApi = broadcastApiResult.result;
        UserInfo userInfo = userInfoResult.result;

        class Data {
            IIngestTester ingestTester;
            IngestTesterState state;
        }

        final Data data = new Data();

        // Create video encoder
        IVideoEncoder videoEncoder = new PassThroughVideoEncoder();

        IIngestTesterListener ingestTesterListener = new IIngestTesterListener() {
            @Override
            public void ingestTesterStateChanged() {
                ResultContainer<IngestTesterState> rc = new ResultContainer<>();
                data.ingestTester.getTestState(rc);
                data.state = rc.result;
            }
        };

        // Test disposal
        {
            ResultContainer<IIngestTester> rcIngestTester = new ResultContainer<>();
            ec = broadcastApi.createIngestTester(userInfo.userId, ingestTesterListener, new byte[0], rcIngestTester);

            IIngestTester ingestTester = rcIngestTester.result;

            ResultContainer<IngestServer> rcIngestServer = new ResultContainer<>();
            ec = ingestTester.getIngestServer(rcIngestServer);

            ResultContainer<Integer> rcKbps = new ResultContainer<>();
            ec = ingestTester.getMeasuredKbps(rcKbps);

            ResultContainer<Float> rcProgress = new ResultContainer<>();
            ec = ingestTester.getProgress(rcProgress);

            ec = ingestTester.setTestDurationMilliseconds(12345);

            ResultContainer<Long> rcDuration = new ResultContainer<>();
            ec = ingestTester.getTestDurationMilliseconds(rcDuration);

            ResultContainer<ErrorCode> rcTestError = new ResultContainer<>();
            ec = ingestTester.getTestError(rcTestError);

            ResultContainer<IngestTesterState> rcTestState = new ResultContainer<>();
            ec = ingestTester.getTestState(rcTestState);

            ResultContainer<Integer> rcUserId = new ResultContainer<>();
            ec = ingestTester.getUserId(rcUserId);

            ec = ingestTester.cancel();

            ingestTester.dispose();
            ingestTester = null;
        }

        // Run an ingest test
        {
            ResultContainer<IIngestTester> rcIngestTester = new ResultContainer<>();
            ec = broadcastApi.createIngestTester(userInfo.userId, ingestTesterListener, new byte[0], rcIngestTester);

            IIngestTester ingestTester = rcIngestTester.result;
            data.ingestTester = ingestTester;

            IngestServer server = new IngestServer();
            server.serverName = "SFO";
            server.serverUrl = "rtmp://live.twitch.tv/app/{stream_key}";

            ec = ingestTester.start(server);

            while (data.state != IngestTesterState.Finished && data.state != IngestTesterState.Failed) {
                updateModules();
                Thread.sleep(100);
            }

            data.ingestTester.dispose();
        }

        teardown(coreApi, broadcastApi);
    }
}
