/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "internal/pch.h"

#include "internal/assertion.h"
#include "internal/audioresample.h"
#include "internal/bindings/bindingframecapturer_d3d11.h"
#include "internal/bindings/bindingframecapturer_d3d9.h"
#include "internal/bindings/bindingframecapturer_ios_rendertexture_gles2.h"
#include "internal/bindings/bindingframecapturer_ios_swizzle_gles2.h"
#include "internal/bindings/bindingframecapturer_opengl2.h"
#include "internal/bindings/bindings_stream.h"
#include "internal/bindings/csharp/csharp_broadcast.h"
#include "internal/bindings/unity/unity.h"

#include <thread>

// http://docs.unity3d.com/Documentation/Manual/Plugins.html
// http://www.tedlindstrom.se/how-to-link-dylibs-into-unity3d/

namespace {
using namespace ttv;

GfxDeviceRenderer gCurrentRenderer = kGfxRendererCount;
std::shared_ptr<ttv::BindingFrameCapturer> gFrameCapturer;

#if TTV_PLATFORM_MAC || TTV_PLATFORM_UNITYIOS
AudioConverterRef gAudioConverter = nullptr;
UInt32 gOutputAudioPacketSize = 0;
int gDeviceType = kGfxRendererCount;
#endif

bool UnityAssertHandler(const char* /*component*/, const char* /*expr*/, const char* /*file*/, int /*line*/) {
  // throw them out, Unity will crash in debug if you let the default assert handler run
  return true;
}

void HackMemoryAllocator() {
  if (ttv::gAllocCallback == nullptr && ttv::gFreeCallback == nullptr) {
    ttv::gAllocCallback = DefaultAlignedMalloc;
    ttv::gFreeCallback = DefaultAlignedFree;
  }
}

// TODO: we really should create abstracted audio conversion functions
TTV_ErrorCode CreateAudioConverter(TTV_AudioFormat inputFormat, uint inputSampleRate, uint inputNumChannels,
  TTV_AudioFormat outputFormat, uint outputSampleRate, uint outputNumChannels);

void DestroyAudioConverter();
}  // namespace

// If exported by a plugin, this function will be called when graphics device is created, destroyed,
// before it's being reset (i.e. resolution changed), after it's being reset, etc.
extern "C" void EXPORT_API UnitySetGraphicsDevice(void* device, int deviceType, int evt) {
  // HACK: Unity loads the library and immediately calls this entry point so there is not time to setup the allocator
  // via TTV_Init.
  static bool firstTime = true;
  if (firstTime) {
    firstTime = false;

#if !TTV_PLATFORM_UNITYIOS
    HackMemoryAllocator();
#endif

    ttv::assertion::SetAssertHandler(UnityAssertHandler);
  }

  using namespace ttv;
  GfxDeviceEventType::Enum eventType = static_cast<GfxDeviceEventType::Enum>(evt);

  switch (eventType) {
    case GfxDeviceEventType::kGfxDeviceEventInitialize: {
      assert(gFrameCapturer == nullptr);

      switch (deviceType) {
        // Direct3D 9
        case kGfxRendererD3D9: {
#if TTV_SUPPORT_D3D9
          gFrameCapturer = std::make_shared<BindingFrameCapturer_D3D9>();
#endif
          break;
        }
        // Direct3D 11
        case kGfxRendererD3D11: {
#if TTV_SUPPORT_D3D11
          gFrameCapturer = std::make_shared<BindingFrameCapturer_D3D11>();
#endif
          break;
        }
        // OpenGL
        case kGfxRendererOpenGL: {
#if TTV_SUPPORT_OPENGL2
          gFrameCapturer = std::make_shared<BindingFrameCapturer_OpenGL2>();
#endif
          break;
        }
        // OpenGL ES 1.1
        case kGfxRendererOpenGLES: {
#if TTV_SUPPORT_GLES1
          // we don't support GLES1
#endif
          break;
        }
        // OpenGL ES 2.0 mobile variant
        case kGfxRendererOpenGLES20Mobile: {
#if TTV_SUPPORT_GLES2
#if TTV_PLATFORM_UNITYIOS
          gFrameCapturer = std::make_shared<BindingFrameCapturer_iOS_RenderTexture_GLES2>();
          /*
                     gFrameCapturer = std::make_shared<BindingFrameCapturer_iOS_Swizzle_GLES2>();
                     ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::SetInstance(gFrameCapturer);
                     */
#endif
#endif
          break;
        }
      }

      if (gFrameCapturer != nullptr) {
        gFrameCapturer->SetGraphicsDevice(device, eventType);
        gCurrentRenderer = static_cast<GfxDeviceRenderer>(deviceType);
      } else {
        gCurrentRenderer = kGfxRendererCount;
      }

      break;
    }
    case GfxDeviceEventType::kGfxDeviceEventShutdown: {
      // shutdown and destroy the capturer
      if (gFrameCapturer != nullptr) {
        gFrameCapturer->Shutdown();
        gFrameCapturer.reset();
      }

      break;
    }
    case GfxDeviceEventType::kGfxDeviceEventBeforeReset: {
      if (gFrameCapturer != nullptr) {
        gFrameCapturer->PreDeviceReset();
      }

      break;
    }
    case GfxDeviceEventType::kGfxDeviceEventAfterReset: {
      if (gFrameCapturer != nullptr) {
        gFrameCapturer->PostDeviceReset();
      }

      break;
    }
  }
}

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_Init() {
#if TTV_PLATFORM_UNITYIOS

  // detect which version of GLES is in use by Unity
  EAGLContext* context = [EAGLContext currentContext];
  if (context == nil) {
    return TTV_EC_NOT_INITIALIZED;
  }

  EAGLRenderingAPI api = [context API];

  if (api == kEAGLRenderingAPIOpenGLES3) {
    // TODO: if Unity ever supports GLES3 add it here
    return TTV_EC_GRAPHICS_API_ERROR;
  } else if (api == kEAGLRenderingAPIOpenGLES2) {
    gDeviceType = kGfxRendererOpenGLES20Mobile;
  } else if (api == kEAGLRenderingAPIOpenGLES1) {
    // we don't support GLES1
    return TTV_EC_GRAPHICS_API_ERROR;
  } else {
    return TTV_EC_GRAPHICS_API_ERROR;
  }

  // Unity iOS does not call the plugin functions as it should
  UnitySetGraphicsDevice(nullptr, gDeviceType, ttv::GfxDeviceEventType::kGfxDeviceEventInitialize);

#pragma clang diagnostic push
#pragma clang diagnostic warning "-Wdeprecated-declarations"

  // allow access to the microphone
  // NOTE: This seems to play nice with Unity but needs testing in a real game
  UInt32 category = kAudioSessionCategory_PlayAndRecord;
  OSStatus status = AudioSessionSetProperty(kAudioSessionProperty_AudioCategory, sizeof(category), &category);
  assert(status == 0);

  status = AudioSessionSetActive(true);
  assert(status == 0);

#pragma clang diagnostic pop

#endif

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_Shutdown() {
#if TTV_PLATFORM_UNITYIOS

  // Unity iOS does not call the plugin functions as it should
  UnitySetGraphicsDevice(nullptr, gDeviceType, ttv::GfxDeviceEventType::kGfxDeviceEventShutdown);
  gDeviceType = kGfxRendererCount;

#endif

  DestroyAudioConverter();

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_GetCapturePixelFormat(TTV_PixelFormat* format) {
  if (format == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  // TODO: this should probably be moved into the bindingframecapturer

  switch (gCurrentRenderer) {
    case kGfxRendererD3D9:
      *format = TTV_PF_BGRA;
      break;
    case kGfxRendererD3D11:
      *format = TTV_PF_RGBA;
      break;
    case kGfxRendererOpenGL:
      *format = TTV_PF_RGBA;
      break;
    case kGfxRendererOpenGLES:
      *format = TTV_PF_RGBA;
      break;
    case kGfxRendererOpenGLES20Mobile:
      *format = TTV_PF_BGRA;
      break;
    default:
      *format = TTV_PF_BGRA;
      break;
  }

  return TTV_EC_SUCCESS;
}

// If exported by a plugin, this function will be called for GL.IssuePluginEvent script calls.
// The function will be called on a rendering thread; note that when multithreaded rendering is used,
// the rendering thread WILL BE DIFFERENT from the thread that all scripts & other game logic happens!
// You have to ensure any synchronization with other plugin script calls is properly done by you.
// extern "C" void EXPORT_API UnityRenderEvent(int /*eventID*/)
//{
//  //TODO: handle multithreaded rendering
//  //std::thread::id tid = std::this_thread::get_id();
//  //trace::Message("blah", TTV_ML_ERROR, "UnityRenderEvent: %d", (int)tid.hash());
//}

namespace {
void HandleStartCallback(TTV_ErrorCode ec, void* /*userData*/) {
  ttv::gManagedBroadcastAPIListener.FireStartCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_Start(const TTV_VideoParams* videoParams,
  const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags, bool async) {
  TTV_RETURN_ON_DIFFERENT(ttv::gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(gFrameCapturer, TTV_EC_NOT_INITIALIZED);

  if (videoParams == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  // perform a vertical flip because the texture layout is different than opengl
  TTV_VideoParams copy = *videoParams;
  if (gFrameCapturer->RequiresVerticalFlip()) {
    copy.verticalFlip = !copy.verticalFlip;
  }

  TTV_ErrorCode ec;

  if (async) {
    ec = TTV_Start(&copy, audioParams, ingestServer, flags, &HandleStartCallback, nullptr);
  } else {
    ec = TTV_Start(&copy, audioParams, ingestServer, flags, nullptr, nullptr);
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = gFrameCapturer->Start(&copy, audioParams, ingestServer, flags);
  }

#if TTV_PLATFORM_UNITYIOS

  // handle ios audio capture
  if (TTV_SUCCEEDED(ec)) {
    // we expect at most one audio capture method to be configured
    assert(!audioParams->enablePlaybackCapture || !audioParams->enablePassthroughAudio);

    if (audioParams->enablePlaybackCapture) {
      assert(!audioParams->enablePassthroughAudio);
    } else if (audioParams->enablePassthroughAudio) {
      // NOTE: we expect Unity to call TTV_Unity_Broadcast_SubmitAudioSamples
    }
  }

#endif
  return ec;
}

namespace {
void HandleStopCallback(TTV_ErrorCode ec, void* /*userData*/) {
  assert(gFrameCapturer != nullptr);
  if (gFrameCapturer == nullptr) {
    return;
  }

  if (TTV_SUCCEEDED(ec)) {
    gFrameCapturer->Stop();
  }

  DestroyAudioConverter();

  ttv::gManagedBroadcastAPIListener.FireStopCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_Stop(bool async) {
  TTV_RETURN_ON_DIFFERENT(ttv::gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(gFrameCapturer, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec;

  if (async) {
    ec = TTV_Stop(&HandleStopCallback, nullptr);
  } else {
    ec = TTV_Stop(nullptr, nullptr);

    if (TTV_SUCCEEDED(ec)) {
      gFrameCapturer->Stop();
    }

    DestroyAudioConverter();
  }

  return ec;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_SubmitTexture(void* p, int width, int height) {
  assert(gFrameCapturer != nullptr);
  if (gFrameCapturer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  } else if (p == nullptr) {
    return TTV_EC_INVALID_ARG;
  }

  return gFrameCapturer->SubmitTexture(p, width, height);
}

namespace {
using namespace ttv;

const uint kDesiredSampleRate = 44100;
const uint kDesiredNumChannels = 2;
const TTV_AudioFormat kDesiredAudioFormat = AudioFormat_PCM;

#if TTV_PLATFORM_MAC || TTV_PLATFORM_UNITYIOS

TTV_ErrorCode CreateAudioConverter(TTV_AudioFormat inputFormat, uint inputSampleRate, uint inputNumChannels,
  TTV_AudioFormat outputFormat, uint outputSampleRate, uint outputNumChannels) {
  DestroyAudioConverter();

  // For PCM
  //   Sample: a single value for single point in time for a single channel
  //   Frame: the samples from all channels at a single point in time (one sample from each channel)
  //   Packet: the same as a frame (for PCM)

  UInt32 appleInputFormat = GetAppleAudioFormat(inputFormat);
  UInt32 appleInputFlags = GetAppleAudioFlags(inputFormat, true);
  UInt32 appleInputBitsPerChannel = GetAppleBitsPerChannel(inputFormat, true);

  UInt32 appleOutputFormat = GetAppleAudioFormat(outputFormat);
  UInt32 appleOutputFlags = GetAppleAudioFlags(outputFormat, false);
  UInt32 appleOutputBitsPerChannel = GetAppleBitsPerChannel(outputFormat, false);

  // Set up the recording format
  AudioStreamBasicDescription recordFormat;
  memset(&recordFormat, 0, sizeof(recordFormat));
  recordFormat.mFormatID = appleInputFormat;
  recordFormat.mChannelsPerFrame = inputNumChannels;
  recordFormat.mFormatFlags = appleInputFlags;
  recordFormat.mFramesPerPacket = 1;
  recordFormat.mBitsPerChannel = appleInputBitsPerChannel;
  recordFormat.mBytesPerFrame = recordFormat.mChannelsPerFrame * recordFormat.mBitsPerChannel / 8;
  recordFormat.mBytesPerPacket = recordFormat.mBytesPerFrame * recordFormat.mFramesPerPacket;
  recordFormat.mSampleRate = inputSampleRate;

  // Set up the output format
  AudioStreamBasicDescription resultFormat;
  memset(&resultFormat, 0, sizeof(resultFormat));
  resultFormat.mFormatID = appleOutputFormat;
  resultFormat.mChannelsPerFrame = outputNumChannels;
  resultFormat.mFormatFlags = appleOutputFlags;
  resultFormat.mFramesPerPacket = 1;
  resultFormat.mBitsPerChannel = appleOutputBitsPerChannel;
  resultFormat.mBytesPerFrame = resultFormat.mChannelsPerFrame * resultFormat.mBitsPerChannel / 8;
  resultFormat.mBytesPerPacket = resultFormat.mBytesPerFrame * resultFormat.mFramesPerPacket;
  resultFormat.mSampleRate = outputSampleRate;

  // Create the converter
  //
  OSStatus err = AudioConverterNew(&recordFormat, &resultFormat, &gAudioConverter);
  assert(err == noErr);

  if (err == noErr) {
    // Determine the max size of an output PCM packet
    UInt32 propSize = sizeof(gOutputAudioPacketSize);
    OSStatus err = AudioConverterGetProperty(
      gAudioConverter, kAudioConverterPropertyMaximumOutputPacketSize, &propSize, &gOutputAudioPacketSize);
    assert(err == noErr);
    assert(gOutputAudioPacketSize == sizeof(int16_t) * 2);

    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_APPLEAAC_FAILED_INIT;
  }
}

void DestroyAudioConverter() {
  if (gAudioConverter != nullptr) {
    OSStatus err = AudioConverterDispose(gAudioConverter);
    gAudioConverter = nullptr;
    gOutputAudioPacketSize = 0;
  }
}

OSStatus AudioConverterInputCallback(AudioConverterRef inAudioConverter, UInt32* ioNumberDataPackets,
  AudioBufferList* ioData, AudioStreamPacketDescription** outDataPacketDescription, void* inUserData) {
  float* inputBuffer = static_cast<float*>(inUserData);
  assert(inputBuffer);

  // On input, the minimum number of packets of input audio data the converter needs for its current conversion cycle.
  UInt32 numRequestedPackets = *ioNumberDataPackets;

  // printf("AudioConverterInputCallback: numRequestedPackets: %u, inUserData: 0x%x\n", (unsigned
  // int)numRequestedPackets, (unsigned int)inUserData);

  ioData->mNumberBuffers = 1;
  ioData->mBuffers[0].mData = inputBuffer;
  ioData->mBuffers[0].mNumberChannels = 2;  // NOTE: this assumes the input data has 2 channels
  ioData->mBuffers[0].mDataByteSize = numRequestedPackets * ioData->mBuffers[0].mNumberChannels * sizeof(float);

  // On output, the number of packets of audio data provided for conversion, or 0 if there is no more data to convert.
  *ioNumberDataPackets = numRequestedPackets;

  return noErr;
}

TTV_ErrorCode ConvertSamples(const float* samples, TTV_AudioFormat sourceFormat, uint numSamples, uint sampleRate,
  uint numChannels, std::vector<int16_t>& converted) {
  // printf("ConvertSamples numSamples: %u, numChannels: %u, samples: 0x%x\n", numSamples, numChannels, (unsigned
  // int)samples);

  // no conversion or resample required
  if (sourceFormat == kDesiredAudioFormat && sampleRate == kDesiredSampleRate) {
    return RescalePcmFloatToInt(samples, numSamples, numChannels, kDesiredNumChannels, converted);
  }
  // need to convert and/or resample
  else {
    if (gAudioConverter == nullptr) {
      CreateAudioConverter(
        sourceFormat, sampleRate, numChannels, AudioFormat_PCM, kDesiredSampleRate, kDesiredNumChannels);
    }

    if (gAudioConverter != nullptr) {
      float factor = static_cast<float>(kDesiredSampleRate) / static_cast<float>(sampleRate);
      uint numOutputSamples = static_cast<uint>(0.5f + factor * static_cast<float>(numSamples));
      uint bufferSize = numOutputSamples * gOutputAudioPacketSize / kDesiredNumChannels;

      converted.resize(bufferSize);

      OSStatus err = noErr;
      uint samplesWritten = 0;

      // Create an AudioBufferList to write the converted data to
      AudioBufferList convertedData;
      convertedData.mNumberBuffers = 1;
      convertedData.mBuffers[0].mNumberChannels = kDesiredNumChannels;
      convertedData.mBuffers[0].mDataByteSize = bufferSize;
      convertedData.mBuffers[0].mData = converted.data();

      // Now call the converter
      UInt32 ioOutputDataPacketSize = numOutputSamples / kDesiredNumChannels;
      AudioStreamPacketDescription* aspdList = new AudioStreamPacketDescription[ioOutputDataPacketSize];

      // printf("Begin encode\n");
      err = AudioConverterFillComplexBuffer(gAudioConverter, AudioConverterInputCallback,
        const_cast<void*>(static_cast<const void*>(samples)), &ioOutputDataPacketSize, &convertedData, aspdList);

      assert(err == noErr);
      samplesWritten = ioOutputDataPacketSize * kDesiredNumChannels;

      delete[] aspdList;

      // printf("End encode: %u, %u\n", (unsigned int)convertedData.mBuffers[0].mDataByteSize, (unsigned
      // int)samplesWritten);

      if (err == noErr) {
        converted.resize(samplesWritten);
      } else {
        // TODO: put some blank audio in so the video doesn't block?
        converted.resize(0);
        assert(false);
        return TTV_EC_APPLEAAC_FAILED_ENCODING;
      }
    }
  }

  return TTV_EC_SUCCESS;
}

#else

TTV_ErrorCode CreateAudioConverter(TTV_AudioFormat /*inputFormat*/, uint /*inputSampleRate*/, uint /*inputNumChannels*/,
  TTV_AudioFormat /*outputFormat*/, uint /*outputSampleRate*/, uint /*outputNumChannels*/) {
  DestroyAudioConverter();

  // TODO: implement a native converter

  return TTV_EC_UNKNOWN_ERROR;
}

void DestroyAudioConverter() {
  // TODO: implement a native converter
}

TTV_ErrorCode ConvertSamples(const float* samples, TTV_AudioFormat sourceFormat, uint numSamples, uint sampleRate,
  uint numChannels, std::vector<int16_t>& converted) {
  // no conversion or resample required
  if (sourceFormat == kDesiredAudioFormat && sampleRate == kDesiredSampleRate) {
    return RescalePcmFloatToInt(samples, numSamples, numChannels, kDesiredNumChannels, converted);
  }
  // need to convert and/or resample
  else {
    // TODO: do some optimized conversion for this platform
    converted.resize(0);
    assert(false);
  }

  return TTV_EC_SUCCESS;
}

#endif

}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_Unity_Broadcast_SubmitAudioSamples(
  const float* samples, uint numSamples, uint numChannels, uint sampleRate) {
  // TODO: reuse this buffer
  std::vector<int16_t> converted;

  TTV_ErrorCode ec = ConvertSamples(samples, AudioFormat_PCM, numSamples, sampleRate, numChannels, converted);

  if (TTV_SUCCEEDED(ec)) {
    ec = TTV_SubmitAudioSamples(converted.data(), static_cast<uint>(converted.size()));
  } else {
    // TODO: we're going to get a desync and/or frame queue backup
    assert(false);
  }

  return ec;
}
