/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "internal/pch.h"

#include "internal/bindings/csharp/csharp_broadcast.h"
#include "internal/bindings/xna/xna.h"

#if TTV_SUPPORT_D3D9

#include "twitchsdk.h"

namespace {
std::unique_ptr<ttv::BindingFrameCapturer_D3D9> gFrameCapturer;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_XNA_Broadcast_Init() {
  if (gFrameCapturer != nullptr) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  gFrameCapturer = std::make_unique<ttv::BindingFrameCapturer_D3D9>();

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_XNA_Broadcast_Shutdown() {
  if (gFrameCapturer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  gFrameCapturer->Shutdown();
  gFrameCapturer.reset();

  return TTV_EC_SUCCESS;
}

extern "C" EXPORT_API TTV_ErrorCode TTV_XNA_Broadcast_SetGraphicsDevice(void* device) {
  using namespace ttv;

  if (gFrameCapturer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  IDirect3DDevice9* d3dDevice = reinterpret_cast<IDirect3DDevice9*>(device);

  if (d3dDevice == nullptr) {
    gFrameCapturer->SetGraphicsDevice(d3dDevice, GfxDeviceEventType::kGfxDeviceEventShutdown);
  } else {
    gFrameCapturer->SetGraphicsDevice(d3dDevice, GfxDeviceEventType::kGfxDeviceEventInitialize);
  }

  return TTV_EC_SUCCESS;
}

namespace {
void HandleStartCallback(TTV_ErrorCode ec, void* /*userData*/) {
  ttv::gManagedBroadcastAPIListener.FireStartCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_XNA_Broadcast_Start(const TTV_VideoParams* videoParams,
  const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags, bool async) {
  TTV_RETURN_ON_DIFFERENT(ttv::gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(gFrameCapturer, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec;

  if (async) {
    ec = TTV_Start(videoParams, audioParams, ingestServer, flags, &HandleStartCallback, nullptr);
  } else {
    ec = TTV_Start(videoParams, audioParams, ingestServer, flags, nullptr, nullptr);
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = gFrameCapturer->Start(videoParams, audioParams, ingestServer, flags);
  }

  return ec;
}

namespace {
void HandleStopCallback(TTV_ErrorCode ec, void* /*userData*/) {
  assert(gFrameCapturer != nullptr);
  if (gFrameCapturer == nullptr) {
    return;
  }

  if (TTV_SUCCEEDED(ec)) {
    gFrameCapturer->Stop();
  }

  ttv::gManagedBroadcastAPIListener.FireStopCallback(ec);
}
}  // namespace

extern "C" EXPORT_API TTV_ErrorCode TTV_XNA_Broadcast_Stop(bool async) {
  TTV_RETURN_ON_DIFFERENT(ttv::gBroadcastInitialized, true, TTV_EC_NOT_INITIALIZED);
  TTV_RETURN_ON_NULL(gFrameCapturer, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec;

  if (async) {
    ec = TTV_Stop(&HandleStopCallback, nullptr);
  } else {
    ec = TTV_Stop(nullptr, nullptr);

    if (TTV_SUCCEEDED(ec)) {
      gFrameCapturer->Stop();
    }
  }

  return ec;
}

extern "C" EXPORT_API TTVSDK_API TTV_ErrorCode TTV_XNA_Broadcast_SubmitRenderTarget(void* p, int width, int height) {
  assert(gFrameCapturer != nullptr);
  if (gFrameCapturer == nullptr) {
    return TTV_EC_NOT_INITIALIZED;
  }

  return gFrameCapturer->SubmitTexture(p, width, height);
}

#endif
