#include "amf0.h"

#include "stdafx.h"

const uint8_t* DecodeElement(const uint8_t* data, std::shared_ptr<IAMF0> output);

template <typename _T>
_T ConvertTo(const uint8_t*& data) {
  _T result = 0;
  auto p = reinterpret_cast<uint8_t*>(&result);

  for (int i = 0; i < sizeof(_T); i++) {
    p[i] = data[sizeof(_T) - i - 1];
  }

  data += sizeof(_T);
  return result;
}

template <>
std::string ConvertTo(const uint8_t*& data) {
  auto length = ConvertTo<uint16_t>(data);

  std::string param(data, data + length);

  data += length;

  return param;
}

const uint8_t* DecodeString(const uint8_t* data, std::shared_ptr<IAMF0> output, AMFType kind) {
  assert(kind == string || kind == object || kind == ecmaArray);

  auto param = ConvertTo<std::string>(data);

  switch (kind) {
    case string:
      output->String(param);
      break;
    case object:
      output->ObjectProperty(param);
      break;
    case ecmaArray:
      output->EcmaArrayKey(param);
      break;
  }
  return data;
}

const uint8_t* DecodeDate(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto value = ConvertTo<double>(data);
  auto timeZome = ConvertTo<int16_t>(data);
  assert(timeZome == 0);

  output->Date(value);
  return data;
}

const uint8_t* DecodeNumber(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto value = ConvertTo<double>(data);

  output->Number(value);
  return data;
}

const uint8_t* DecodeEmcaArray(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto elements = ConvertTo<uint32_t>(data);
  output->EcmaArray(elements);

  for (uint32_t i = 0; i < elements; i++) {
    data = DecodeString(data, output, ecmaArray);
    data = DecodeElement(data, output);
  }

  return data;
}

const uint8_t* DecodeStrictArray(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  auto elements = ConvertTo<uint32_t>(data);
  output->StrictArray(elements);

  for (uint32_t i = 0; i < elements; i++) {
    data = DecodeElement(data, output);
  }

  return data;
}

const uint8_t* DecodeObject(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  output->Object();

  for (;;) {
    if (data[0] == 0 && data[1] == 0 && data[2] == objectEnd) {
      output->ObjectEnd();
      data += 3;
      break;
    }
    data = DecodeString(data, output, object);
    data = DecodeElement(data, output);
  }

  return data;
}

const uint8_t* DecodeBoolean(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  bool flag = (*data++ != 0);
  output->Boolean(flag);
  return data;
}

const uint8_t* DecodeElement(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  AMFType type = static_cast<AMFType>(*data++);

  assert(type != objectEnd);
  assert(type != movieclip);
  assert(type != recordset);
  assert(type != unsupported);

  switch (type) {
    case number:
      data = DecodeNumber(data, output);
      break;
    case boolean:
      data = DecodeBoolean(data, output);
      break;
    case string:
      data = DecodeString(data, output, string);
      break;
    case object:
      data = DecodeObject(data, output);
      break;
    case null:
      assert(false && "untested");
      output->Null();
      break;
    case undefined:
      assert(false && "untested");
      output->Undefined();
      break;
    case reference:
      assert(false && "unimplemented");
      output->Reference();
      break;
    case ecmaArray:
      data = DecodeEmcaArray(data, output);
      break;
    case strictArray:
      data = DecodeStrictArray(data, output);
      break;
    case date:
      data = DecodeDate(data, output);
      break;
    case longString:
      assert(false && "unimplemented");
      output->LongString();
      break;
    case xmlDocument:
      output->XmlDocument();
      break;
    case typedObject:
      assert(false && "unimplemented");
      output->TypedObject();
      break;
    default:
      assert(false);
      break;
  }

  return data;
}

const uint8_t* DecodeAMF(const uint8_t* data, std::shared_ptr<IAMF0> output) {
  assert(data);
  assert(output);

  return DecodeElement(data, output);
}
