#ifndef AMF0_H
#define AMF0_H

#include <ctime>
#include <string>

enum AMFType : uint8_t {
  // Tested Types */
  number = 0x00,
  boolean = 0x01,
  string = 0x02,
  object = 0x03,
  ecmaArray = 0x08,
  objectEnd = 0x09,
  strictArray = 0x0A,
  date = 0x0B,

  // Untested but implemented types
  null = 0x05,
  undefined = 0x06,

  // Unimplemented types
  reference = 0x07,
  longString = 0x0C,
  xmlDocument = 0x0F,
  typedObject = 0x10,

  // Unsupported types
  movieclip = 0x04,  // reserved, not supported
  recordset = 0x0E,  // reserved, not supported
  unsupported = 0x0D,
};

class IAMF0 {
 public:
  virtual void Number(double value) = 0;
  virtual void Boolean(bool flag) = 0;
  virtual void String(std::string param) = 0;
  virtual void Object() = 0;
  virtual void ObjectProperty(std::string propertyName) = 0;
  virtual void Movieclip() = 0;
  virtual void Null() = 0;
  virtual void Undefined() = 0;
  virtual void Reference() = 0;
  virtual void EcmaArray(uint32_t elements) = 0;
  virtual void EcmaArrayKey(std::string keyName) = 0;
  virtual void ObjectEnd() = 0;
  virtual void StrictArray(uint32_t elements) = 0;
  virtual void Date(double date) = 0;
  virtual void LongString() = 0;
  virtual void Unsupported() = 0;
  virtual void Recordset() = 0;
  virtual void XmlDocument() = 0;
  virtual void TypedObject() = 0;
};

class IAMF0Printer : public IAMF0 {
 public:
  IAMF0Printer(FILE* logFile) : mLogFile(logFile) {}
  void Number(double value) { fprintf(mLogFile, "number %f\n", value); }
  void Boolean(bool flag) { fprintf(mLogFile, "boolean is '%s'\n", flag ? "true" : "false"); }
  void String(std::string param) { fprintf(mLogFile, "String '%s'\n", param.c_str()); }
  void Object() { fprintf(mLogFile, "object START\n"); }
  void ObjectProperty(std::string propetyName) { fprintf(mLogFile, "ObjectProperty '%s'\n", propetyName.c_str()); }
  void Movieclip() { fprintf(mLogFile, "movieclip\n"); }
  void Null() { fprintf(mLogFile, "null\n"); }
  void Undefined() { fprintf(mLogFile, "undefined\n"); }
  void Reference() { fprintf(mLogFile, "reference\n"); }
  void EcmaArray(uint32_t elements) { fprintf(mLogFile, "ecmaArray with %u elements\n", elements); }
  void EcmaArrayKey(std::string keyName) { fprintf(mLogFile, "ecmaArrayKey '%s'\n", keyName.c_str()); }
  void ObjectEnd() { fprintf(mLogFile, "objectEnd\n"); }
  void StrictArray(uint32_t elements) { fprintf(mLogFile, "strictArray with %u elements\n", elements); }
  void Date(double date) {
    char prettyDateTime[128];
    tm localTime;

    time_t timeSince1970 = static_cast<time_t>(date / 1000.0);

#if WIN32
    localtime_s(&localTime, &timeSince1970);
#else
    std::localtime(&timeSince1970);
#endif

    strftime(prettyDateTime, 128, "%c", &localTime);

    fprintf(mLogFile, "%s\n", prettyDateTime);
  }
  void LongString() { fprintf(mLogFile, "longString\n"); }
  void Unsupported() { fprintf(mLogFile, "unsupported\n"); }
  void Recordset() { fprintf(mLogFile, "recordset\n"); }
  void XmlDocument() { fprintf(mLogFile, "xmlDocument\n"); }
  void TypedObject() { fprintf(mLogFile, "typedObject\n"); }

 private:
  FILE* mLogFile;
};

const uint8_t* DecodeAMF(const uint8_t* data, std::shared_ptr<IAMF0> output);

#endif
