// flvdump.cpp : Defines the entry point for the console application.
//

#include "IAMF0Encoder.h"
#include "amf0.h"
#include "flvformat.h"
#include "stdafx.h"

#include <algorithm>
#include <cstdlib>
#include <iostream>
#include <vector>

template <typename _T>
uint32_t QuickCheckSum(_T begin, _T end) {
  uint32_t result = 0;

  auto it = begin;
  while (it != end) {
    result ^= *it + 0x9e3779b9 + (result << 6) + (result >> 2);
    ++it;
  }

  return result;
}

inline errno_t fopen_s(FILE** f, const char* n, const char* a) {
  *f = fopen(n, a);

  return 0;
}

inline size_t fread_s(void* b, size_t s1, size_t s2, size_t s3, FILE* f) {
  return fread(b, s2, s3, f);
}

void flvDump(std::string baseFilename, bool txt, bool mp3, bool avc) {
  std::string flvFileName = baseFilename + ".flv";
  std::string txtFileName = baseFilename + ".txt";
  std::string mp3FileName = baseFilename + ".mp3raw";
  std::string avcFileName = baseFilename + ".avcraw";

  FILE* flvFile = 0;
  FILE* txtFile = 0;
  FILE* mp3File = 0;
  FILE* avcFile = 0;

  fopen_s(&flvFile, flvFileName.c_str(), "rb");
  assert(flvFile);

  if (!flvFile) {
    return;
  }

  if (txt) {
    fopen_s(&txtFile, txtFileName.c_str(), "wt");
    assert(txtFile);
  }

  if (mp3) {
    fopen_s(&mp3File, mp3FileName.c_str(), "wb");
    assert(mp3File);
  }

  if (avc) {
    fopen_s(&avcFile, avcFileName.c_str(), "wb");
    assert(avcFile);
  }

  flv::Header header;
  auto bytesRead = fread_s(&header, sizeof(flv::Header), 1, sizeof(flv::Header), flvFile);
  assert(bytesRead == sizeof(flv::Header));

  auto headerSize = BigToLittle(header.mHeaderSize);
  assert(header.mSigniture[0] == 'F' && header.mSigniture[1] == 'L' && header.mSigniture[2] == 'V');
  assert(headerSize == 9);

  fseek(flvFile, 4, SEEK_CUR);  // always 4 0s

  std::vector<uint8_t> buffer;

  for (;;) {
    flv::TagHeader tagHeader;
    bytesRead = fread_s(&tagHeader, sizeof(flv::TagHeader), 1, sizeof(flv::TagHeader), flvFile);
    if (bytesRead == 0) {
      break;
    }
    assert(bytesRead == sizeof(flv::TagHeader));

    auto payloadSize = BigToLittle(tagHeader.mPacketLength);
    buffer.resize(payloadSize);
    bytesRead = fread_s(&buffer[0], payloadSize, 1, payloadSize, flvFile);
    assert(bytesRead == payloadSize);

    switch (tagHeader.mPacketType) {
      case flv::TagTypes::Audio:
        if (txtFile)
          fprintf(txtFile, "Audio Packet: ");
        if (mp3File)
          fwrite(&buffer[1], bytesRead - 1, 1, mp3File);
        break;
      case flv::TagTypes::Video:
        if (txtFile)
          fprintf(txtFile, "Video Packet: ");
        if (avcFile)
          fwrite(&buffer[0], bytesRead, 1, avcFile);
        break;
      case flv::TagTypes::Meta:
        if (txtFile)
          fprintf(txtFile, "Meta Packet : ");
        break;
      default:
        if (txtFile)
          fprintf(txtFile, "Unknown Pkt : ");
        break;
    }

    uint32_t timestamp = BigToLittle(tagHeader.mPacketTimestamp) | tagHeader.mPacketExtendedTimestamp;
    if (txtFile)
      fprintf(txtFile, "Size = %-8d Timestamp = %d Stream = %d CheckSum = %08x\n", payloadSize, timestamp,
        BigToLittle(tagHeader.mStreamId), QuickCheckSum(buffer.begin(), buffer.end()));
    if (txtFile)
      fflush(txtFile);

    if (tagHeader.mPacketType == flv::TagTypes::Meta) {
      std::shared_ptr<IAMF0Encoder> amf(new IAMF0Encoder);
      std::shared_ptr<IAMF0> amfPrinter(new IAMF0Printer(txtFile));

      // First decode should name of the meta
      const uint8_t* p = DecodeAMF(&buffer[0], amf);
      DecodeAMF(&(amf->GetBuffer()[0]), amfPrinter);

      amf.reset(new IAMF0Encoder);
      // Second should be the data

      p = DecodeAMF(p, amfPrinter);
      //          const uint8_t* old = p;
      //          p = DecodeAMF(p, amf);
      //
      //          // verify that we did a good copy
      //          auto differs = std::mismatch(amf->GetBuffer().begin(), amf->GetBuffer().end(), old);
      //          assert (differs.first == amf->GetBuffer().end());
      //
      //          DecodeAMF(& (amf->GetBuffer()[0]), amfPrinter);
      //
      //          assert (p - &buffer[0] < buffer.size());
    }

    flv::TagFooter tagFooter;
    bytesRead = fread_s(&tagFooter, sizeof(flv::TagFooter), 1, sizeof(flv::TagFooter), flvFile);
    assert(bytesRead == sizeof(flv::TagFooter));

    auto pos = ftell(flvFile);
    assert(BigToLittle(tagFooter.mLength) == payloadSize + sizeof(flv::TagHeader));
  }

  if (mp3File)
    fclose(mp3File);
  if (avcFile)
    fclose(avcFile);
  if (flvFile)
    fclose(flvFile);
  if (txtFile)
    fclose(txtFile);
}

int main(int argc, char* argv[]) {
  bool showHelp = false;

  std::vector<std::string> filesToProcess;

  bool text = false;
  bool mp3 = false;
  bool avc = false;

  for (int arg = 1; arg < argc; ++arg) {
    std::string current(argv[arg]);
    if (current[0] != '-') {
      filesToProcess.push_back(current);
    } else if (current.compare("--txt") == 0) {
      text = true;
    } else if (current.compare("--mp3") == 0) {
      mp3 = true;
    } else if (current.compare("--avc") == 0) {
      avc = true;
    } else if (current.compare("--help") == 0) {
      showHelp = true;
    } else {
      showHelp = true;
      std::cout << "Unknown option" << current << std::endl;
    }
  }

  if (showHelp) {
    std::cout << "flvdumper [--help]" << std::endl
              << "flvdumper path/to/inputfile.flv [--txt] [--mp3] [--avc]" << std::endl
              << std::endl;
    std::cout << "\t--help\tShow this help" << std::endl;
    std::cout << "\t--txt\tDump textual info about the file into a .txt file" << std::endl;
    std::cout << "\t--mp3\tDump raw mp3 data into .mp3raw file" << std::endl;
    std::cout << "\t--avc\tDump raw avc data into .avcraw file" << std::endl;
  } else
    for (auto filename : filesToProcess) {
      std::cout << filename << std::endl;
      std::string baseFilename = filename;
      baseFilename.erase(baseFilename.find_last_of('.'), -1);
      flvDump(baseFilename, text, mp3, avc);
    }

  // flvDump("c:\\dev\\sdk\\sdktester\\sdktest.flv", "c:\\dev\\sdk\\sdktester\\sdktest.txt");
  // flvDump("c:\\dev\\sdk\\sdktester\\flvmuxer.flv", "c:\\dev\\sdk\\sdktester\\flvmuxer.txt");
  // flvDump("C:\\Users\\gareth\\Desktop\\sdktest.flv", "C:\\Users\\gareth\\Desktop\\sdktest.txt");
  // fopen_s (&f,"c:\\dev\\sdk\\sdktester\\sdktest.flv","rb");
  // system ("pause");
  return 0;
}
