#include <stdint.h>

#define HMAC_IPAD_VAL 0x36
#define HMAC_OPAD_VAL 0x5C

/// Embed an HMAC-SHA256 digest of a buffer into that buffer at the exact
/// location specified.
///
/// @param embed_pos The offset where the 32 byte hmac digest will be placed
///                  (and where the algorithm will skip when creating the
///                  digest). If it's set to -1, the digest will be embedded in
///                  the last 32 bytes of the buffer.
/// @param check     If 1, it doesn't actually insert the hmac digest but
///                  instead looks to see if the expected digest is already
///                  embedded.
///
/// @return          Actual embeded offset or -1 if check failed.
///
static ssize_t embed_hmac(uint8_t *buf, size_t buf_len, ssize_t embed_pos, uint8_t *key, size_t key_len, int check) {
  uint8_t hmac_buf[64 + 32] = {0};
  uint8_t hmac_check[32];
  int i;
  sha256_context sha;

  if (embed_pos == -1)
    embed_pos = buf_len - 32;

  // Get key's digest if it's too long
  if (key_len < 64)
    memcpy(hmac_buf, key, key_len);
  else {
    sha256_starts(&sha);
    sha256_update(&sha, key, key_len);
    sha256_finish(&sha, hmac_buf);
  }
  for (i = 0; i < 64; i++)
    hmac_buf[i] ^= HMAC_IPAD_VAL;

  sha256_starts(&sha);
  sha256_update(&sha, hmac_buf, 64);
  sha256_update(&sha, buf, embed_pos);
  sha256_update(&sha, buf + embed_pos + 32, buf_len - embed_pos - 32);
  sha256_finish(&sha, hmac_buf + 64);
  for (i = 0; i < 64; i++)
    hmac_buf[i] ^= HMAC_IPAD_VAL ^ HMAC_OPAD_VAL;

  sha256_starts(&sha);
  sha256_update(&sha, hmac_buf, 64 + 32);
  if (check) {
    sha256_finish(&sha, hmac_check);
    if (memcmp(hmac_check, buf + embed_pos, 32) == 0)
      return embed_pos;
    else
      return -1;
  } else {
    sha256_finish(&sha, buf + embed_pos);
    return embed_pos;
  }
}

/// Embed an HMAC-SHA256 digest of a buffer in that buffer, at the location
/// specified by RTMP conventions.
///
/// @param   embk    RTMP looks at 4 bytes starting here in the buffer, adds
///                  their values together, does some math on it and uses that
///                  as the offset for the digest.
/// @param   type    0 for normal RTMP, 1 for diffie-hellman offset calculation
/// @param   check   Check only to see if correct value is embedded in the
///                  expected position.
///
/// @return          The actual offset where the 32-byte digest was inserted,
///                  or -1 if check fails.
///
static int rtmp_embed_hmac(
  uint8_t *buf, size_t buf_len, uint8_t *key, size_t key_len, size_t embk, int type, int check) {
  size_t embed_offset = buf[embk] + buf[embk + 1] + buf[embk + 2] + buf[embk + 3];
  int modulus_by = type ? 632 : 728;
  int adjust_by = type ? -760 : 4;
  embed_offset = (embed_offset % modulus_by) + embk + adjust_by;
  return embed_hmac(buf, buf_len, embed_offset, key, key_len, check);
}
