#ifndef RTMP_H
#define RTMP_H

#include "internal/muxers/flvformat.h"

namespace rtmp {
/// Chunk header-type, indicated in first byte of chunk basic header
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum Chunktype : uint8_t { RTMP_CHUNKTYPE_LARGE = 0, RTMP_CHUNKTYPE_MEDIUM, RTMP_CHUNKTYPE_SMALL, RTMP_CHUNKTYPE_NONE };

/// Common Chunk-stream-ids. Not set in stone- they are just the usual convention...
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum Channel : uint8_t {
  RTMP_CHANNEL_NETWORK = 2,
  RTMP_CHANNEL_SYSTEM,
  RTMP_CHANNEL_SOURCE,
  RTMP_CHANNEL_NETSTREAM,
  RTMP_CHANNEL_VIDEO = 8,
  RTMP_CHANNEL_AUDIO
};

/// Packet types, indicated in the chunk header
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum PacketType : uint8_t {
  RTMP_PKT_CHUNK_SIZE = 0x01,
  RTMP_PKT_ABORT_MSG = 0x02,
  RTMP_PKT_BYTES_READ = 0x03,
  RTMP_PKT_CONTROL = 0x04,  ///< See RTMP_CTL_* below for possible values
  RTMP_PKT_WINACKSIZE = 0x05,
  RTMP_PKT_PEER_BW = 0x06,

  RTMP_PKT_EDGE_ORIGIN = 0x07,  ///< No payloads actually documented...

  RTMP_PKT_AUDIO = 0x08,
  RTMP_PKT_VIDEO = 0x09,

  RTMP_PKT_AMF3_DATA = 0x0F,  ///< amf3 metadata body
  RTMP_PKT_AMF3_SO = 0x10,    ///< amf3 shared object command (don't support)
  RTMP_PKT_AMF3 = 0x11,       ///< cmd-name, trans-id, amf3 body

  RTMP_PKT_AMF0_DATA = 0x12,  ///< amf0 metadata body
  RTMP_PKT_AMF0_SO = 0x13,    ///< amf0 shared object command (don't support)
  RTMP_PKT_AMF0 = 0x14,       ///< cmd-name, trans-id, amf0 body

  RTMP_PKT_AGGREGATE = 0x16  ///< needs to be chopped up
};

/// When packet type is RTMP_PKT_CONTROL- these are the possible payloads
///
/// enum names and values taken from imbibe, so kept the naming to
/// make debugging easier
enum CtlType : uint8_t {
  RTMP_CTL_STREAM_BEGIN = 0,   ///< Data: stream-id (4B)
  RTMP_CTL_STREAM_EOF,         ///< Data: stream-id (4B)
  RTMP_CTL_STREAM_DRY,         ///< Data: stream-id (4B)
  RTMP_CTL_SET_BUFFER_LENGTH,  ///< Data: stream-id, buffer-length in milliseconds (*** 8B ***)
  RTMP_CTL_STREAM_RECORDED,    ///< Server telling client it's not live, Data: stream-id (4B)
  RTMP_CTL_PING = 6,           ///< Data: timestamp (4B)
  RTMP_CTL_PONG                ///< Responds with the ping's timestamp, so RTT can be calculated (4B)
};

template <class Inserter>
inline void InsertBigEndian32(Inserter it, uint32_t val) {
  uint8_t* p = reinterpret_cast<uint8_t*>(&val);
  *it++ = p[3];
  *it++ = p[2];
  *it++ = p[1];
  *it++ = p[0];
}

template <class Inserter>
inline void InsertBigEndian(Inserter it, uint32_t val) {
  uint8_t* p = reinterpret_cast<uint8_t*>(&val);
  *it++ = p[2];
  *it++ = p[1];
  *it++ = p[0];
}

template <class Inserter>
inline void InsertBigEndian(Inserter it, uint16_t val) {
  uint8_t* p = reinterpret_cast<uint8_t*>(&val);
  *it++ = p[1];
  *it++ = p[0];
}

inline uint32_t BigToLittle(uint32_t source) {
  uint32_t target = 0;

  auto pSource = reinterpret_cast<uint8_t*>(&source);
  auto pTarget = reinterpret_cast<uint8_t*>(&target);
  pTarget[0] = pSource[3];
  pTarget[1] = pSource[2];
  pTarget[2] = pSource[1];
  pTarget[3] = pSource[0];

  return target;
}

inline uint32_t BigToLittle(uint24_t source) {
  uint32_t target = 0;

  auto pTarget = reinterpret_cast<uint8_t*>(&target);
  pTarget[0] = source[2];
  pTarget[1] = source[1];
  pTarget[2] = source[0];

  return target;
}

struct ChunkHeader {
  Chunktype type;
  uint8_t streamId;
  uint32_t timestamp;
  uint32_t packetLength;
  PacketType messageType;
  uint32_t messageStreamId;
};

int GetChunkSize(char firstByteOfChunk) {
  Chunktype type = static_cast<Chunktype>(firstByteOfChunk >> 6);

  // if this assert hits we got a really big chunk header
  assert((firstByteOfChunk & 0x3f) != 0x3f);

  switch (type) {
    case RTMP_CHUNKTYPE_NONE:
      return 1;
    case RTMP_CHUNKTYPE_SMALL:
      return 4;
    case RTMP_CHUNKTYPE_MEDIUM:
      return 8;
    case RTMP_CHUNKTYPE_LARGE:
      return 12;
  }

  return 0;
}

ChunkHeader PopulateChunkHeader(const char* buffer) {
  int pos = 0;

  ChunkHeader h;
  h.type = static_cast<Chunktype>(buffer[pos] >> 6);
  h.streamId = buffer[pos++] & 0x3f;

  if (h.type != RTMP_CHUNKTYPE_NONE) {
    uint24_t* as24 = (uint24_t*)&buffer[pos];
    pos += 3;
    h.timestamp = BigToLittle(*as24);
  }
  if (h.type < RTMP_CHUNKTYPE_SMALL) {
    uint24_t* as24 = (uint24_t*)&buffer[pos];
    pos += 3;
    h.packetLength = BigToLittle(*as24);
    h.messageType = static_cast<PacketType>(buffer[pos++]);
  }
  if (h.type == RTMP_CHUNKTYPE_LARGE) {
    uint8_t* p = reinterpret_cast<uint8_t*>(&h.messageStreamId);
    std::copy(buffer + pos, buffer + pos + 4, p);
    pos += 4;
  }

  return h;
}

template <class Inserter>
void CreateMessageHeader(Inserter it, Chunktype type, uint8_t streamId, uint32_t timestamp, uint32_t packetLength,
  PacketType messageType, uint32_t messageStreamId) {
  assert(streamId < 64);

  *it = (type << 6) + streamId;
  ++it;

  if (type != RTMP_CHUNKTYPE_NONE) {
    InsertBigEndian(it, timestamp);
  }
  if (type < RTMP_CHUNKTYPE_SMALL) {
    InsertBigEndian(it, packetLength);
    *it = messageType;
    ++it;
  }
  if (type == RTMP_CHUNKTYPE_LARGE) {
    uint8_t* p = reinterpret_cast<uint8_t*>(&messageStreamId);
    std::copy(p, p + 4, it);
  }
}

}  // namespace rtmp

#endif
