// Need to link with Ws2_32.lib
#include "internal/muxers/amf0encoder.h"
#include "internal/muxers/flvformat.h"
#include "rtmp.h"
#include "stdafx.h"

#include <sys/types.h>

#include <algorithm>

#pragma comment(lib, "ws2_32.lib")

#if WIN32  // Stuff that should be in sys/types.h but isn't in windows
#define LITTLE_ENDIAN 1234
#define BIG_ENDIAN 4321
#define BYTE_ORDER LITTLE_ENDIAN
#define ssize_t SSIZE_T
#endif

static uint8_t gClientKey[] = {'G', 'e', 'n', 'u', 'i', 'n', 'e', ' ', 'A', 'd', 'o', 'b', 'e', ' ', 'F', 'l', 'a', 's',
  'h', ' ', 'P', 'l', 'a', 'y', 'e', 'r', ' ', '0', '0', '1', 0xF0, 0xEE, 0xC2, 0x4A, 0x80, 0x68, 0xBE, 0xE8, 0x2E,
  0x00, 0xD0, 0xD1, 0x02, 0x9E, 0x7E, 0x57, 0x6E, 0xEC, 0x5D, 0x2D, 0x29, 0x80, 0x6F, 0xAB, 0x93, 0xB8, 0xE6, 0x36,
  0xCF, 0xEB, 0x31, 0xAE};

enum RTMPSendState {
  RTMP_STATE_INVALID,
  RTMP_STATE_HANDSHAKE,
  RTMP_STATE_CONNECT,
  RTMP_STATE_CREATESTREAM,
  RTMP_STATE_PUBLISH,
  RTMP_STATE_SEND_VIDEO,
  RTMP_STATE_SHUTDOWN
};

// parameters

// const char* gHostName="127.0.0.1";
// const char* gPort="2223";
// const char* gApplication="flvplayback";
// const char* gStreamName="test.flv";

const char* gHostName = "199.9.249.153";  // live.twitch.tv; dns was failing in getaddrinfo()
const char* gPort = "1935";
const char* gApplication = "app";
// const char* gStreamName="live_14919816_jILfpUmdOTUA2aMlccmZFnOjBqr5MX"; // GarethLewin
const char* gStreamName = "live_17411845_3dZPSytGHXkcZ8ITiJL45UiUBkAzcX";  // zadrtest

// const char* gFLVName="C:\\dev\\infratools\\ingest-ranker\\test.flv";
const char* gFLVName = "/Users/Zach/Downloads/test.flv";
// const char* gFLVName="/var/mobile/Applications/A7B739B2-53AF-40E9-83FD-D01E86C1335D/rtmpsender-ios.app/test.flv";

char gURL[100];
char* gReceiveBuffer;
const int gReceiveBufferSize = 0x10000;
int gReceiveBufferPos = 0;
double gStreamId = 0;
int gTotalSent = 0;
int gWindowAckSize = 0;
int gChunkSize = 128;
int gIncomingChunkSize = 128;

RTMPSendState gCurrentState = RTMP_STATE_HANDSHAKE;

void ttv_Log(const char* str, ...) __printflike(1, 0);
void ttv_Log(const char* str, ...) {
  time_t now = time(NULL);
  struct tm* local = localtime(&now);

  va_list list;
  va_start(list, str);

  printf("%02d-%02d-%02d %02d:%02d:%02d.%03ld - ", (local->tm_year + 1900), (local->tm_mon + 1), local->tm_mday,
    (local->tm_hour % 12), local->tm_min, local->tm_sec, (now % 100));
  vprintf(str, list);
  printf("\n");

  va_end(list);
}

#if SDK(WINDOWS)
#define ttv_Main _tmain
#define ttv_GetLastError WSAGetLastError()
#define ttv_WouldBlock WSAEWOULDBLOCK
#define ttv_NoBlock FIONBIO
#define ttv_setNonBlocking ioctlsocket
#define ttv_Socket SOCKET
#define ttv_SocketCleanup WSACleanup();
#define ttv_Close closesocket
#define ttv_ZeroMemory(addr, length) \
  do {                               \
    ZeroMemory(addr, length);        \
  } while (0);
// duration is given in Milliseconds
#define ttv_Sleep(duration) \
  do {                      \
    Sleep(duration);        \
    while (0)               \
      ;
#define _CRT_SECURE_NO_WARNINGS
#else
#define SOCKET_ERROR -1
#define INVALID_SOCKET -1
#define NO_ERROR 0

#define ttv_Main main
#define ttv_GetLastError errno
#define ttv_WouldBlock EWOULDBLOCK
#define ttv_NoBlock FIONBIO
#define ttv_setNonBlocking ioctl
#define ttv_Socket int
#define ttv_SocketCleanup
#define ttv_Close close
#define ttv_ZeroMemory(addr, length) \
  do {                               \
    memset(addr, 0, length);         \
  } while (0);
// duration is given in milliseconds, usleep takes microseconds.
#define ttv_Sleep(duration)  \
  do {                       \
    usleep(duration * 1000); \
  } while (0);
#endif

#pragma comment(lib, "ws2_32.lib")
void HandleInputData(ttv_Socket connectSocket);

int ForceSend(ttv_Socket sock, const char* buf, int len, int flags) {
  int pos = 0;
  while (pos < len) {
    auto sent = send(sock, &buf[pos], len - pos, flags);
    if (sent < 0) {
      auto lastError = ttv_GetLastError;
      if (lastError == ttv_WouldBlock) {
        printf("Blocked\n");
      } else {
        assert(lastError);
        printf("Socket Error %d\n", lastError);
        abort();
      }
    } else {
      pos += sent;
      if (pos < len) {
        //              printf ("Partial %d\n", sent);
      }
    }
  }

  gTotalSent += len;
  assert(pos == len);
  return pos;
}

std::vector<char> gHackyBuffer;

void BufferFlush(ttv_Socket sock, int flags) {
  if (gHackyBuffer.size() > 0) {
    ForceSend(sock, &gHackyBuffer[0], gHackyBuffer.size(), 0);
    gHackyBuffer.resize(0);
  }
}

int BufferedSend(ttv_Socket sock, const char* buf, int len, int flags) {
#if SDK(WINDOWS)
  if (gHackyBuffer.size() + len > gHackyBuffer.capacity()) {
    BufferFlush(sock, flags);
  }
  gHackyBuffer.insert(gHackyBuffer.end(), buf, &buf[len]);

  return len;
#else
  return ForceSend(sock, buf, len, flags);
#endif
}

// TODO: will probably ForceSend the header properly
template <class _T>
int SendBuffer(ttv_Socket connectSocket, const std::vector<_T>& buffer, int chunkSize, uint8_t channel) {
  size_t pos = 0;
  int ret = 0;
  int sent = 0;

  uint8_t chunkContinueHeader = channel + (rtmp::RTMP_CHUNKTYPE_NONE << 6);

  auto bufferSize = buffer.size() * sizeof(_T);

  while (bufferSize - pos > static_cast<size_t>(chunkSize)) {
    sent = BufferedSend(connectSocket, (char*)&buffer[pos / sizeof(_T)], chunkSize, 0);
    assert(sent == chunkSize);
    ret += sent;
    sent = BufferedSend(connectSocket, (char*)&chunkContinueHeader, 1, 0);
    assert(sent == 1);
    ret += sent;

    pos += chunkSize;
  }

  sent = BufferedSend(connectSocket, (char*)&buffer[pos / sizeof(_T)], bufferSize - pos, 0);
  assert(sent == bufferSize - pos);
  ret += sent;
  return ret;
}

int BlockingRecv(ttv_Socket sock, char* buf, int len, int flags) {
  int pos = 0;
  while (pos < len) {
    int received = recv(sock, &buf[pos], len - pos, flags);
    if (received < 0) {
      if (ttv_WouldBlock != ttv_GetLastError) {
        printf("blockingrecv: socket function failed with error: %ld\n", ttv_GetLastError);
        return received;
      }
      ttv_Sleep(1);
    } else if (received > 0) {
      pos += received;
    }
  }

  return pos;
}

// A good place to do move semantics
void BeginMessage(const std::string& name, ttv::IAMF0& amf) {
  static int transactionId = 0;

  amf.String(name);
  amf.Number(++transactionId);
}

int SendMetaData(ttv_Socket connectSocket, const std::vector<char>& buffer) {
  ttv::AMF0Encoder amfEncoder;
  std::vector<char> messageBuffer;

  amfEncoder.String("@setDataFrame");

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 4, 0,
    amfEncoder.GetBuffer().size() + buffer.size(), rtmp::RTMP_PKT_AMF0_DATA, 1);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);

  // HACK
  // Need this copy to handle chunking, will come up with a better solution
  messageBuffer.resize(0);
  std::copy(amfEncoder.GetBuffer().begin(), amfEncoder.GetBuffer().end(), std::back_inserter(messageBuffer));
  std::copy(buffer.begin(), buffer.end(), std::back_inserter(messageBuffer));
  //

  iResult += SendBuffer(connectSocket, messageBuffer, gChunkSize, 4);

  BufferFlush(connectSocket, 0);
  return iResult;
}

int SendVideoData(ttv_Socket connectSocket, const std::vector<char>& buffer, uint32_t timeStamp) {
  ttv::AMF0Encoder amfEncoder;
  std::vector<char> messageBuffer;

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 4, (timeStamp / 256),
    buffer.size(), rtmp::RTMP_PKT_VIDEO, 1);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return SendBuffer(connectSocket, buffer, gChunkSize, 4);
}

int SendAudioData(ttv_Socket connectSocket, const std::vector<char>& buffer, uint32_t timeStamp) {
  ttv::AMF0Encoder amfEncoder;
  std::vector<char> messageBuffer;

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 4, (timeStamp / 256),
    buffer.size(), rtmp::RTMP_PKT_AUDIO, 1);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return SendBuffer(connectSocket, buffer, gChunkSize, 4);
}

int SendFLV(ttv_Socket connectSocket, const std::string& flvName) {
  FILE* flvFile = 0;

  int ret = 0;

  flvFile = fopen(flvName.c_str(), "rb");
  assert(flvFile);

  if (!flvFile) {
    return 0;
  }

  flv::Header header;
  auto bytesRead = fread(&header, 1, sizeof(flv::Header), flvFile);
  assert(bytesRead == sizeof(flv::Header));

  auto headerSize = rtmp::BigToLittle(header.mHeaderSize);
  assert(header.mSigniture[0] == 'F' && header.mSigniture[1] == 'L' && header.mSigniture[2] == 'V');
  assert(headerSize == 9);

  fseek(flvFile, 4, SEEK_CUR);  // always 4 0s

  std::vector<char> buffer;

  for (;;) {
    flv::TagHeader tagHeader;
    bytesRead = fread(&tagHeader, 1, sizeof(flv::TagHeader), flvFile);
    if (bytesRead == 0) {
      break;
    }
    assert(bytesRead == sizeof(flv::TagHeader));

    auto payloadSize = rtmp::BigToLittle(tagHeader.mPacketLength);
    buffer.resize(payloadSize);
    bytesRead = fread(&buffer[0], 1, payloadSize, flvFile);
    assert(bytesRead == payloadSize);

    switch (tagHeader.mPacketType) {
      case flv::Audio:
        //          printf("Audio Packet: \n");
        ret += SendAudioData(connectSocket, buffer, rtmp::BigToLittle(tagHeader.mPacketTimestamp));
        break;
      case flv::Video:
        //          printf("Video Packet: \n");
        ret += SendVideoData(connectSocket, buffer, rtmp::BigToLittle(tagHeader.mPacketTimestamp));
        break;
      case flv::Meta:
        printf("Meta Packet : ");
        ret += SendMetaData(connectSocket, buffer);
        break;
      default:
        printf("Unknown Pkt : ");
        assert((tagHeader.mPacketType != flv::Audio && tagHeader.mPacketType != flv::Video &&
                tagHeader.mPacketType != flv::Meta));
        break;
    }

    // printf("Size = %-8d Timestamp = %d Stream = %d\n", payloadSize, rtmp::BigToLittle(tagHeader.mPacketTimestamp),
    // rtmp::BigToLittle(tagHeader.mStreamId));

    flv::TagFooter tagFooter;
    bytesRead = fread(&tagFooter, 1, sizeof(flv::TagFooter), flvFile);
    assert(bytesRead == sizeof(flv::TagFooter));

    auto pos = ftell(flvFile);
    assert(rtmp::BigToLittle(tagFooter.mLength) == payloadSize + sizeof(flv::TagHeader));

    HandleInputData(connectSocket);
  }

  BufferFlush(connectSocket, 0);

  fclose(flvFile);

  return ret;
}

int SendConnectMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("connect", amfEncoder);
  amfEncoder.Object();
  amfEncoder.ObjectProperty("app");
  amfEncoder.String(gApplication);
  amfEncoder.ObjectProperty("type");
  amfEncoder.String("nonprivate");
  amfEncoder.ObjectProperty("tcUrl");
  amfEncoder.String(gURL);
  amfEncoder.ObjectEnd();

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 3, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 0);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

int SendReleaseStreamMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("releaseStream", amfEncoder);
  amfEncoder.Null();
  amfEncoder.String(gStreamName);

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 3, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 0);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

int SendFCPublishMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("FCPublish", amfEncoder);
  amfEncoder.Null();
  amfEncoder.String(gStreamName);

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 3, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 0);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

int SendCreateStreamMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("createStream", amfEncoder);
  amfEncoder.Null();

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 3, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 0);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

int SendFCUnpublishMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("FCUnpublish", amfEncoder);
  amfEncoder.Null();
  amfEncoder.String(gStreamName);

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 3, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 0);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

int SendDeleteStreamMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("deleteStream", amfEncoder);
  amfEncoder.Null();
  amfEncoder.Number(gStreamId);

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 3, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 0);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

int SendSetChunkSize(ttv_Socket connectSocket, uint32_t chunkSize) {
  std::vector<char> messageBuffer;
  rtmp::CreateMessageHeader(
    std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 2, 0, 4, rtmp::RTMP_PKT_CHUNK_SIZE, 0);

  rtmp::InsertBigEndian32(std::back_inserter(messageBuffer), chunkSize);
  gChunkSize = chunkSize;

  return BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
}

//////////////////////////////////////////////////////////////////////////
// Publish
//////////////////////////////////////////////////////////////////////////
// The client sends the publish command to publish a named stream to the
// server. Using this name, any client can play this stream and receive
// the published audio, video, and data messages.
//
// The command structure from the client to the server is as follows:
//
// +--------------+----------+-----------------------------------------+
// | Field Name   |   Type   |              Description                |
// +--------------+----------+-----------------------------------------+
// | Command Name |  String  | Name of the command, set to publish.  |
// +--------------+----------+-----------------------------------------+
// | Transaction  |  Number  | Transaction ID set to 0.                |
// | ID           |          |                                         |
// +--------------+----------+-----------------------------------------+
// | Command      |  Null    | Command information object does not     |
// | Object       |          | exist. Set to null type.                |
// +--------------+----------+-----------------------------------------+
// | Publishing   |  String  | Name with which the stream is           |
// | Name         |          | published.                              |
// +--------------+----------+-----------------------------------------+
// | Publishing   |  String  | Type of publishing. Set to live,      |
// | Type         |          | record, or append.                  |
// |              |          | record : The stream is published and the|
// |              |          | data is recorded to a new file.The file |
// |              |          | is stored on the server in a            |
// |              |          | subdirectory within the directory that  |
// |              |          | contains the server application. If the |
// |              |          | file already exists, it is overwritten. |
// |              |          | append : The stream is published and the|
// |              |          | data is appended to a file. If no file  |
// |              |          | is found, it is created.                |
// |              |          | live : Live data is published without   |
// |              |          | recording it in a file.                 |
// +--------------+----------+-----------------------------------------+
// The server responds with the OnStatus command to mark the beginning
// of publish.

int SendPublishMessage(ttv_Socket connectSocket) {
  std::vector<char> messageBuffer;

  ttv::AMF0Encoder amfEncoder;

  BeginMessage("publish", amfEncoder);
  amfEncoder.Null();
  amfEncoder.String(gStreamName);
  amfEncoder.String("live");

  rtmp::CreateMessageHeader(std::back_inserter(messageBuffer), rtmp::RTMP_CHUNKTYPE_LARGE, 4, 0,
    amfEncoder.GetBuffer().size(), rtmp::RTMP_PKT_AMF0, 1);

  int iResult = BufferedSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);
  return iResult + SendBuffer(connectSocket, amfEncoder.GetBuffer(), gChunkSize, 3);
}

void HandleIncomingChunkSize(rtmp::ChunkHeader header, char* data) {
  uint32_t* as32 = reinterpret_cast<uint32_t*>(data);
  gIncomingChunkSize = rtmp::BigToLittle(as32[0]);
}

void HandleIncomingAbortMsg(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAbortMsg, Unsupported!\n");
  return;
}
void HandleIncomingBytesRead(rtmp::ChunkHeader header, char* data) {
  uint32_t* as32 = reinterpret_cast<uint32_t*>(data);
  int bytesReadByServer = rtmp::BigToLittle(as32[0]);

  static int sentCount = 0;
  printf("%d. Server has read %d/%d (%02f%%) bytes, off by %d\n", (sentCount++), bytesReadByServer, gTotalSent,
    (((float)bytesReadByServer / gTotalSent) * 100), (gTotalSent - bytesReadByServer));
}
void HandleIncomingControl(rtmp::ChunkHeader header, char* data) {
  rtmp::CtlType type = (rtmp::CtlType)data[0];
  switch (type) {
    case rtmp::RTMP_CTL_STREAM_BEGIN:
      printf("Go Stream Begin\n");
      break;
    default:
      printf("In HandleIncomingControl, Unsupported\n");
  }
}
void HandleIncomingWinacksize(rtmp::ChunkHeader header, char* data) {
  uint32_t* as32 = reinterpret_cast<uint32_t*>(data);
  gWindowAckSize = rtmp::BigToLittle(as32[0]);
}
void HandleIncomingPeerBW(rtmp::ChunkHeader header, char* data) {
  uint32_t* as32 = reinterpret_cast<uint32_t*>(data);
  int serverBw = rtmp::BigToLittle(as32[0]);
  char* type = "";
  switch (data[5]) {
    case 0:
      type = "Hard";
      break;
    case 1:
      type = "Soft";
      break;
    case 2:
      type = "Dynamic";
      break;
    default:
      type = "Error";
  }

  printf("Server Bandwidth %d (%s)\n", serverBw, type);
}

void HandleIncomingEdgeOrigin(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingEdgeOrigin, Unsupported!\n");
}
void HandleIncomingAudio(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAudio, Unsupported!\n");
}
void HandleIncomingVideo(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingVideo, Unsupported!\n");
}
void HandleIncomingAmf3Data(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAmf3Data, Unsupported!\n");
}
void HandleIncomingAmf3SharedObject(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAmf3SharedObject, Unsupported!\n");
}
void HandleIncomingAmf3(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAmf3, Unsupported!\n");
}
void HandleIncomingAmf0Data(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAmf0Data, Unsupported!!\n");
}
void HandleIncomingAmf0SharedObject(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAmf0SharedObject, Unsupported!\n");
}

bool CaseInsensitiveStringCompare(const std::string& str1, const std::string& str2) {
  std::string str1Cpy(str1);
  std::string str2Cpy(str2);
  std::transform(str1Cpy.begin(), str1Cpy.end(), str1Cpy.begin(), ::tolower);
  std::transform(str2Cpy.begin(), str2Cpy.end(), str2Cpy.begin(), ::tolower);
  return (str1Cpy.compare(str2Cpy) == 0);
}
class IAMF0StringDecoder : public ttv::AMF0Nop {
 public:
  IAMF0StringDecoder() {}

  virtual void String(std::string param) {
    if (mCommandName.length() == 0) {
      mCommandName = param;
    }
  }

  const std::string& GetCommandName() const { return mCommandName; }

 private:
  std::string mCommandName;
};

class IAMF0NumberDecoder : public ttv::AMF0Nop {
 public:
  IAMF0NumberDecoder() {}

  virtual void Number(double value) { mValue = value; }

  double GetValue() const { return mValue; }

 private:
  double mValue;
};

class IAMF0PropertyDecoder : public ttv::AMF0Nop {
 public:
  IAMF0PropertyDecoder(const std::string& searchTerm) : mSearchTerm(searchTerm), mFoundProperty(0) {}

  virtual void String(std::string param) {
    if (mFoundProperty) {
      mFoundValue = param;
      mFoundProperty = false;
    }
  }

  virtual void ObjectProperty(std::string propertyName) {
    mFoundProperty = CaseInsensitiveStringCompare(propertyName, mSearchTerm);
  }

  const std::string& GetFoundValue() const { return mFoundValue; }

 private:
  const std::string mSearchTerm;
  std::string mFoundValue;
  bool mFoundProperty;
};

void HandleIncomingAmf0(rtmp::ChunkHeader header, char* data) {
  std::shared_ptr<IAMF0StringDecoder> stringDecoder(new IAMF0StringDecoder());
  data = (char*)DecodeAMF((const uint8_t*)data, stringDecoder);

  // Skip over the number
  std::shared_ptr<ttv::AMF0Nop> nopDecoder(new ttv::AMF0Nop());
  data = (char*)DecodeAMF((const uint8_t*)data, nopDecoder);

  // command meta data
  data = (char*)DecodeAMF((const uint8_t*)data, nopDecoder);

  if (CaseInsensitiveStringCompare(stringDecoder->GetCommandName(), "_result") && gCurrentState == RTMP_STATE_CONNECT) {
    std::shared_ptr<IAMF0PropertyDecoder> propertyDecoder(new IAMF0PropertyDecoder("code"));
    data = (char*)DecodeAMF((const uint8_t*)data, propertyDecoder);

    if (CaseInsensitiveStringCompare(propertyDecoder->GetFoundValue(), "NetConnection.Connect.Success")) {
      gCurrentState = RTMP_STATE_CREATESTREAM;
    }
  } else if (CaseInsensitiveStringCompare(stringDecoder->GetCommandName(), "_result") &&
             gCurrentState == RTMP_STATE_CREATESTREAM) {
    std::shared_ptr<IAMF0NumberDecoder> numberDecoder(new IAMF0NumberDecoder());
    data = (char*)DecodeAMF((const uint8_t*)data, numberDecoder);

    gStreamId = numberDecoder->GetValue();

    gCurrentState = RTMP_STATE_PUBLISH;
  } else if (CaseInsensitiveStringCompare(stringDecoder->GetCommandName(), "onStatus") &&
             gCurrentState == RTMP_STATE_PUBLISH) {
    std::shared_ptr<IAMF0PropertyDecoder> propertyDecoder(new IAMF0PropertyDecoder("code"));
    data = (char*)DecodeAMF((const uint8_t*)data, propertyDecoder);

    if (CaseInsensitiveStringCompare(propertyDecoder->GetFoundValue(), "NetStream.Publish.Start")) {
      gCurrentState = RTMP_STATE_SEND_VIDEO;
    }
  } else {
    int i = 3;
    ++i;
  }
}

void HandleIncomingAggregate(rtmp::ChunkHeader header, char* data) {
  printf("In HandleIncomingAggregate, Unsupported!\n");
}

void HandleInputData(ttv_Socket connectSocket) {
  int iResult = recv(connectSocket, &gReceiveBuffer[gReceiveBufferPos], gReceiveBufferSize - gReceiveBufferPos, 0);
  if (iResult > 0) {
    gReceiveBufferPos += iResult;
  } else if (iResult == -1) {
    if (ttv_GetLastError != ttv_WouldBlock) {
      printf("handleinput: socket function failed with error: %ld\n", ttv_GetLastError);
    }
  }

  assert(gReceiveBufferPos < gReceiveBufferSize);

  int readPos = 0;
  if (gReceiveBufferPos) {
    int chunkSize = rtmp::GetChunkSize(gReceiveBuffer[readPos]);
    while ((readPos + chunkSize) < gReceiveBufferPos) {
      rtmp::ChunkHeader h = rtmp::PopulateChunkHeader(&gReceiveBuffer[readPos]);
      assert(chunkSize + h.packetLength < gReceiveBufferSize);
      readPos += rtmp::GetChunkSize(gReceiveBuffer[readPos]);

      int totalBytesWithHeaders = h.packetLength + h.packetLength / gIncomingChunkSize;

      if (h.packetLength > gIncomingChunkSize) {
        for (int i = 0; i < static_cast<int>(h.packetLength / gIncomingChunkSize); ++i) {
          int seperatorPos = gIncomingChunkSize * (i + 1);
          memmove(&gReceiveBuffer[readPos + seperatorPos], &gReceiveBuffer[readPos + seperatorPos + 1],
            totalBytesWithHeaders - seperatorPos - 1);
        }
      }

      if (readPos + static_cast<int>(h.packetLength) <= gReceiveBufferPos) {
        switch (h.messageType) {
          case rtmp::RTMP_PKT_CHUNK_SIZE:
            HandleIncomingChunkSize(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_ABORT_MSG:
            HandleIncomingAbortMsg(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_BYTES_READ:
            HandleIncomingBytesRead(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_CONTROL:
            HandleIncomingControl(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_WINACKSIZE:
            HandleIncomingWinacksize(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_PEER_BW:
            HandleIncomingPeerBW(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_EDGE_ORIGIN:
            HandleIncomingEdgeOrigin(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AUDIO:
            HandleIncomingAudio(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_VIDEO:
            HandleIncomingVideo(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AMF3_DATA:
            HandleIncomingAmf3Data(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AMF3_SO:
            HandleIncomingAmf3SharedObject(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AMF3:
            HandleIncomingAmf3(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AMF0_DATA:
            HandleIncomingAmf0Data(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AMF0_SO:
            HandleIncomingAmf0SharedObject(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AMF0:
            HandleIncomingAmf0(h, &gReceiveBuffer[readPos]);
            break;
          case rtmp::RTMP_PKT_AGGREGATE:
            HandleIncomingAggregate(h, &gReceiveBuffer[readPos]);
            break;
        }
      }

      readPos += totalBytesWithHeaders;
      chunkSize = rtmp::GetChunkSize(gReceiveBuffer[readPos]);
    }

    auto dataLeft = gReceiveBufferPos - readPos;
    assert(dataLeft >= 0);

    if (dataLeft > 0) {
      std::copy(&gReceiveBuffer[readPos], &gReceiveBuffer[gReceiveBufferPos], gReceiveBuffer);
      gReceiveBufferPos = dataLeft;
    } else {
      gReceiveBufferPos = 0;
    }
  }
}

void EnterState(ttv_Socket connectSocket) {
  int iResult = 0;
  switch (gCurrentState) {
    case RTMP_STATE_HANDSHAKE: {
      uint8_t C1[1537];
      int i = 0;
      C1[i++] = 0x3;  // RTMP
      C1[i++] = 0;    // 32bit 'uptime'
      C1[i++] = 0;
      C1[i++] = 0;
      C1[i++] = 0;

#define RTMP_CLIENT_VER1 9
#define RTMP_CLIENT_VER2 0
#define RTMP_CLIENT_VER3 124
#define RTMP_CLIENT_VER4 2
      C1[i++] = 0;  // RTMP_CLIENT_VER1;
      C1[i++] = 0;  // RTMP_CLIENT_VER2;
      C1[i++] = 0;  // RTMP_CLIENT_VER3;
      C1[i++] = 0;  // RTMP_CLIENT_VER4;

      // fill the rest with random data
      for (; i < 1537; ++i) {
        C1[i] = static_cast<char>(i);
      }

      C1[1] = 0;

      BufferFlush(connectSocket, 0);

      char S0[1];
      char S1[1536];

      iResult = BlockingRecv(connectSocket, S0, sizeof(S0), 0);
      iResult = BlockingRecv(connectSocket, S1, sizeof(S1), 0);

      iResult = BufferedSend(connectSocket, S1, sizeof(S1), 0);
      BufferFlush(connectSocket, 0);

      char S2[1536];
      iResult = BlockingRecv(connectSocket, S2, sizeof(S2), 0);
    } break;
    case RTMP_STATE_CONNECT:
      iResult = SendConnectMessage(connectSocket);
      BufferFlush(connectSocket, 0);
      break;
    case RTMP_STATE_CREATESTREAM:
      // iResult = SendSetChunkSize(connectSocket, 0x10000);
      iResult = SendReleaseStreamMessage(connectSocket);
      iResult = SendFCPublishMessage(connectSocket);
      iResult = SendCreateStreamMessage(connectSocket);
      BufferFlush(connectSocket, 0);
      break;
    case RTMP_STATE_PUBLISH: {
      iResult = SendPublishMessage(connectSocket);
      BufferFlush(connectSocket, 0);
    } break;
    case RTMP_STATE_SEND_VIDEO: {
      iResult = SendFLV(connectSocket, gFLVName);
      printf("Sent FLV\n");
    } break;
    case RTMP_STATE_SHUTDOWN: {
      iResult = SendFCUnpublishMessage(connectSocket);
      iResult = SendDeleteStreamMessage(connectSocket);
      BufferFlush(connectSocket, 0);
    } break;
  }
}

void UpdateState(ttv_Socket connectSocket) {
  int iResult = 0;
  switch (gCurrentState) {
    case RTMP_STATE_HANDSHAKE:
      gCurrentState = RTMP_STATE_CONNECT;
      break;

    case RTMP_STATE_CONNECT:       // Intentional
    case RTMP_STATE_CREATESTREAM:  // Intentional
    case RTMP_STATE_PUBLISH:
      HandleInputData(connectSocket);
      break;

    case RTMP_STATE_SEND_VIDEO:
      gCurrentState = RTMP_STATE_SHUTDOWN;
      break;
    case RTMP_STATE_SHUTDOWN:
      break;
  }
}

int ttv_Main(int argc, const char* argv[]) {
  errno = 0;
  sprintf(gURL, "rtmp://%s/%s", gHostName, gApplication);
  gHackyBuffer.reserve(0x10000);
  gReceiveBuffer = new char[gReceiveBufferSize];

  int iResult = 0;
#if SDK(WINDOWS)
  //----------------------
  // Initialize Winsock
  WSADATA wsaData;
  iResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
  if (iResult != NO_ERROR) {
    printf("WSAStartup function failed with error: %d\n", iResult);
    return 1;
  }
#endif

  addrinfo hints;
  ttv_ZeroMemory(&hints, sizeof(hints));
  hints.ai_family = AF_INET;
  hints.ai_socktype = SOCK_STREAM;
  hints.ai_protocol = IPPROTO_TCP;

  addrinfo* remoteHost;

  iResult = getaddrinfo(gHostName, gPort, &hints, &remoteHost);
  assert(remoteHost);

  //----------------------
  // Create a ttv_Socket for connecting to server
  ttv_Socket connectSocket;
  connectSocket = socket(remoteHost->ai_family, remoteHost->ai_socktype, remoteHost->ai_protocol);
  if (connectSocket == INVALID_SOCKET) {
    printf("ttvmain: socket function failed with error: %ld\n", ttv_GetLastError);
    ttv_SocketCleanup return 1;
  }

  //----------------------
  // Connect to server.
  iResult = connect(connectSocket, remoteHost->ai_addr, remoteHost->ai_addrlen);

  if (iResult == SOCKET_ERROR) {
    if (ttv_GetLastError == ttv_WouldBlock) {
      ttv_Sleep(1000);
    } else {
      printf("connect function failed with error: %ld\n", ttv_GetLastError);
      iResult = ttv_Close(connectSocket);
      if (iResult == SOCKET_ERROR)
        printf("closesocket function failed with error: %ld\n", ttv_GetLastError);
      ttv_SocketCleanup return 1;
    }
  }

  u_long iMode = 1;
  iResult = ttv_setNonBlocking(connectSocket, ttv_NoBlock, &iMode);

  printf("Connected to server.\n");

  RTMPSendState currentState = RTMP_STATE_INVALID;

  while (currentState != RTMP_STATE_SHUTDOWN) {
    if (currentState != gCurrentState) {
      assert(gCurrentState > currentState);
      currentState = gCurrentState;
      EnterState(connectSocket);
    } else {
      UpdateState(connectSocket);
    }
    ttv_Sleep(1);
  }
  // iResult = ForceSend(connectSocket, &messageBuffer[0], messageBuffer.size(), 0);

  iResult = ttv_Close(connectSocket);
  if (iResult == SOCKET_ERROR) {
    printf("closesocket function failed with error: %ld\n", ttv_GetLastError);
    ttv_SocketCleanup return 1;
  }

  ttv_SocketCleanup return 0;
}
