//
//  main.cpp
//  testresampling
//
//  Created by Gardner, Jackson on 2/2/18.
//
//  This is just some scratch code that exercises the audio resampling library and does some
//  performance comparisons against ffmpeg, spitting out wav files at the end from each conversion.
//

#include <twitchsdk/broadcast/internal/audioconvert/audioconvertcontext.h>
#include <twitchsdk/broadcast/internal/audioconvert/audioconverter.h>
#include <twitchsdk/broadcast/internal/audioconvert/audioconvertpipeline.h>
#include <twitchsdk/broadcast/internal/audioconvert/pcmbufferaudiosource.h>
#include <twitchsdk/broadcast/internal/wavefilewriter.h>
#include <twitchsdk/core/assertion.h>

#include <chrono>
#include <cmath>
#include <fstream>
#include <iostream>
#include <vector>

extern "C" {
#include <libavutil/opt.h>
#include <libswresample/swresample.h>
}

namespace {
struct AudioConvertOptions {
  static constexpr size_t FilterTapCount = 32;
};

static constexpr size_t kInputSampleRate = 48000;
static constexpr size_t kOutputSampleRate = 44100;
static constexpr size_t kChunkSize = 4096;
}  // namespace

void ttvResample(const int16_t* inputPointer, size_t inputCount, int16_t* outputPointer, size_t maxOutputCount) {
  auto start = std::chrono::system_clock::now();

  using InputBufferFormat = ttv::BufferFormat<int16_t, kInputSampleRate, 2>;
  using OutputBufferFormat = ttv::BufferFormat<int16_t, kOutputSampleRate, 2>;

  ttv::AudioConvertContext<AudioConvertOptions> context;
  auto pipeline = ttv::MakeAudioConvertPipeline<InputBufferFormat, OutputBufferFormat>(context);

  auto elapsed = std::chrono::system_clock::now() - start;
  std::cout << "ttv initialization took " << std::chrono::duration_cast<std::chrono::milliseconds>(elapsed).count()
            << " microseconds." << std::endl;

  start = std::chrono::system_clock::now();

  pipeline.BindInputBuffer(inputPointer, {0, inputCount});
  ttv::SampleRange outputRange = pipeline.TransferToOutputBuffer(outputPointer, {0, maxOutputCount});
  pipeline.UnbindInputBuffer();

  elapsed = std::chrono::system_clock::now() - start;
  std::cout << "ttv processing took " << std::chrono::duration_cast<std::chrono::microseconds>(elapsed).count()
            << " microseconds." << std::endl;

  ttv::broadcast::WaveFileWriter fileWriter;
  bool success = fileWriter.Open("/Users/gardjack/Desktop/ttv-piano2.wav", kOutputSampleRate, 2, false);
  assert(success);

  success = fileWriter.WriteSamples(outputPointer, outputRange.sampleCount * 2);
  assert(success);

  success = fileWriter.Close();
  assert(success);
}

void ttvChunkedResample(const int16_t* inputPointer, size_t inputCount, int16_t* outputPointer, size_t maxOutputCount) {
  auto start = std::chrono::system_clock::now();

  using InputBufferFormat = ttv::BufferFormat<int16_t, kInputSampleRate, 2>;
  using OutputBufferFormat = ttv::BufferFormat<int16_t, kOutputSampleRate, 2>;

  size_t inputOffset = 0;
  size_t bufferSize = std::min(inputCount - inputOffset, static_cast<size_t>(kChunkSize));

  ttv::AudioConvertContext<AudioConvertOptions> context;
  auto pipeline = ttv::MakeAudioConvertPipeline<InputBufferFormat, OutputBufferFormat>(context);

  auto elapsed = std::chrono::system_clock::now() - start;
  std::cout << "ttv chunked initialization took "
            << std::chrono::duration_cast<std::chrono::milliseconds>(elapsed).count() << " microseconds." << std::endl;

  start = std::chrono::system_clock::now();
  size_t outputSampleCount = 0;

  for (;;) {
    pipeline.BindInputBuffer(inputPointer + (inputOffset * 2), {inputOffset, bufferSize});
    ttv::SampleRange writtenRange = pipeline.TransferToOutputBuffer(outputPointer, {0, maxOutputCount});
    pipeline.UnbindInputBuffer();

    assert(writtenRange.startIndex == outputSampleCount);
    outputSampleCount += writtenRange.sampleCount;

    inputOffset += bufferSize;
    bufferSize = std::min(inputCount - inputOffset, static_cast<size_t>(kChunkSize));

    if (inputOffset >= inputCount) {
      break;
    }
  }

  elapsed = std::chrono::system_clock::now() - start;
  std::cout << "ttv chunked processing took " << std::chrono::duration_cast<std::chrono::microseconds>(elapsed).count()
            << " microseconds." << std::endl;

  ttv::broadcast::WaveFileWriter fileWriter;
  bool success = fileWriter.Open("/Users/gardjack/Desktop/ttv-chunked-piano2.wav", kOutputSampleRate, 2, false);
  assert(success);

  success = fileWriter.WriteSamples(outputPointer, outputSampleCount * 2);
  assert(success);

  success = fileWriter.Close();
  assert(success);
}

void ffResample(const int16_t* inputPointer, size_t inputCount, int16_t* outputPointer, size_t maxOutputCount) {
  auto start = std::chrono::system_clock::now();

  SwrContext* context = swr_alloc_set_opts(nullptr, AV_CH_LAYOUT_STEREO, AV_SAMPLE_FMT_S16, kOutputSampleRate,
    AV_CH_LAYOUT_STEREO, AV_SAMPLE_FMT_S16, kInputSampleRate, 0, 0);

  int result = swr_init(context);
  assert(result == 0);

  auto elapsed = std::chrono::system_clock::now() - start;
  std::cout << "ffmpeg initialization took " << std::chrono::duration_cast<std::chrono::milliseconds>(elapsed).count()
            << " microseconds." << std::endl;

  start = std::chrono::system_clock::now();
  int sampleCount = swr_convert(context, reinterpret_cast<uint8_t**>(&outputPointer), static_cast<int>(maxOutputCount),
    reinterpret_cast<const uint8_t**>(&inputPointer), static_cast<int>(inputCount));
  assert(sampleCount != 0);

  elapsed = std::chrono::system_clock::now() - start;
  std::cout << "ffmpeg processing took " << std::chrono::duration_cast<std::chrono::microseconds>(elapsed).count()
            << " microseconds." << std::endl;

  ttv::broadcast::WaveFileWriter fileWriter;
  bool success = fileWriter.Open("/Users/gardjack/Desktop/ff-piano2.wav", kOutputSampleRate, 2, false);
  assert(success);

  success = fileWriter.WriteSamples(outputPointer, sampleCount * 2);
  assert(success);

  success = fileWriter.Close();
  assert(success);
}

std::vector<char> loadDataAtPath(const char* path) {
  std::ifstream sampleDataFile(path, std::ios_base::binary);
  assert(sampleDataFile.is_open());

  std::vector<char> rawData;
  rawData.reserve(1210848);
  rawData.assign((std::istreambuf_iterator<char>{sampleDataFile}), (std::istreambuf_iterator<char>{}));

  return rawData;
}

int main(int argc, const char* argv[]) {
  std::vector<char> dataVector = loadDataAtPath("/Users/gardjack/Downloads/piano2.wav");

  int16_t* samplesPointer = reinterpret_cast<int16_t*>(dataVector.data() + 44);
  size_t sampleCount = (dataVector.size() - 44) / 4;

  size_t outputSampleCapacity = 1400000;
  std::vector<int16_t> outputVector;
  outputVector.resize(outputSampleCapacity);

  for (size_t index = 0; index < 20; index++) {
    std::fill(outputVector.begin(), outputVector.end(), 0);

    ttvChunkedResample(samplesPointer, sampleCount, outputVector.data(), outputSampleCapacity / 2);
    std::fill(outputVector.begin(), outputVector.end(), 0);

    ttvResample(samplesPointer, sampleCount, outputVector.data(), outputSampleCapacity / 2);

    std::fill(outputVector.begin(), outputVector.end(), 0);
    ffResample(samplesPointer, sampleCount, outputVector.data(), outputSampleCapacity / 2);

    std::cout << "--------------------" << std::endl;
  }

  return 0;
}
