/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chaterrortypes.h"
#include "twitchsdk/chat/chatlistener.h"
#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/internal/channel/bitsstatus.h"
#include "twitchsdk/chat/internal/channel/followersstatus.h"
#include "twitchsdk/chat/internal/channel/followingstatus.h"
#include "twitchsdk/chat/internal/channel/subscribersstatus.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/module.h"

#include <unordered_set>

#include <atomic>
#include <map>
#include <memory>
#include <string>
#include <vector>

namespace ttv {
class TaskRunner;
class UserRepository;
class SettingRepository;

namespace chat {
class ChatAPI;
class ChatAPITaskHost;
class ChatChannel;
class BitsConfigRepository;
class IChatObjectFactory;
class ChatChannelSet;
class ChatUserThreads;
class ChatUserBlockList;
class IChannelChatRoomManager;
class IChannelChatRoomManagerListener;
class IChatRoom;
class IChatRoomListener;
class IChatRoomNotifications;
class IChatRoomNotificationsListener;
class IChatChannel;
class IChatCommentManager;
class IChatRaid;
class IChatRaidListener;
class IBitsStatus;
class IBitsListener;
class IFollowersStatus;
class IFollowersListener;
class IFollowingStatus;
class IFollowingListener;
class ISubscribersStatus;
class ISubscribersListener;
class IChatChannelProperties;
class IChatChannelPropertyListener;
class ISubscriptionsNotifications;
class ISubscriptionsNotificationsListener;
class ISquadNotifications;
class ISquadNotificationsListener;
class IMultiviewNotifications;
class IMultiviewNotificationsListener;
struct ChatAPIInternalData;
}  // namespace chat
}  // namespace ttv

/**
 * Provides the main chat service functionality.
 *
 * The following properties must be set before calling Initialize().
 * - SetListener()
 * - SetTokenizationOptions()
 */
class ttv::chat::ChatAPI : public ttv::ModuleBase {
 public:
  /**
   * Used with FetchUserListForChannel()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user list was successfully fetched.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] userList The list of users in the channel.
   */
  using FetchUserListCallback = std::function<void(TTV_ErrorCode ec, UserList&& userList)>;

  /**
   * Used with BlockUser(), UnblockUser()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user list was successfully blocked/unblocked.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   */
  using BlockChangeCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with FetchUserEmoticonSets()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The emoticon sets were successfully fetched.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   *
   * @param[in] sets A list of the emoticon sets.
   */
  using FetchUserEmoticonSetsCallback = std::function<void(TTV_ErrorCode ec, const std::vector<EmoticonSet>& sets)>;

  /**
   * Used with FetchGlobalBadges() and FetchChannelBadges()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The badges were successfully fetched.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] badges The badges that were fetched.
   */
  using FetchBadgesCallback = std::function<void(TTV_ErrorCode ec, BadgeSet&& badges)>;

  /**
   * Used with FetchGlobalBitsConfiguration() and FetchChannelBitsConfiguration()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The bits configuration was successfully fetched.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] bitsConfig A shared pointer to the fetched bits configuration object. Cheermote tiers are sorted in
   * increasing order.
   */
  using FetchBitsConfigurationCallback =
    std::function<void(TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& bitsConfig)>;

  /**
   * Used with SetBroadcasterLanguageChatEnabled()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The broadcaster language settings was successfully changed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   */
  using SetBroadcasterLanguageChatEnabledCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with SendMessageToUser()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The message was successfully sent.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] messageID The ID of the sent message.
   */
  using SendMessageCallback = std::function<void(TTV_ErrorCode ec, MessageId messageID)>;

  /**
   * Used with FetchBlockedUsers()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The blocked users were fetched successfully.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] blockedUsers The list of blocked users.
   */
  using FetchBlockedUsersCallback = std::function<void(TTV_ErrorCode ec, const std::vector<UserInfo>& blockedUsers)>;

  /**
   * Used with FetchChannelVodCommentSettings()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The VOD comment settings were fetched successfully.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] settings The channel's VOD comment settings.
   */
  using FetchChannelVodCommentSettingsCallback =
    std::function<void(TTV_ErrorCode ec, ChannelVodCommentSettings&& settings)>;

  /**
   * Used with SetChannelVodFollowersOnlyDuration() and SetChannelVodPublishingMode()
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The VOD comment settings were set successfully.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the response.
   *   - TTV_EC_REQUEST_ABORTED: The request was aborted.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   */
  using SetChannelVodCommentSettingsCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with BanUser().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The ban was successfully passed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] error Error object received from GraphQL describing why banning the user failed.
   */
  using BanUserCallback = std::function<void(TTV_ErrorCode ec, BanUserError&& error)>;

  /**
   * Used with UnbanUser().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The unban was successfully passed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] error Error object received from GraphQL describing why the unbanning failed.
   */
  using UnbanUserCallback = std::function<void(TTV_ErrorCode ec, UnbanUserError&& error)>;

  /**
   * Used with ModUser().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user successfully became a mod.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] error Error object received from GraphQL describing why the modding failed.
   */
  using ModUserCallback = std::function<void(TTV_ErrorCode ec, ModUserError&& error)>;

  /**
   * Used with UnmodUser().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user successfully became unmodded.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] error Error object received from GraphQL describing why the unmodding failed.
   */
  using UnmodUserCallback = std::function<void(TTV_ErrorCode ec, UnmodUserError&& error)>;

  /**
   * Used with GrantVIP().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user successfully became a VIP.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] error Error object received from GraphQL describing why granting VIP failed.
   */
  using GrantVIPCallback = std::function<void(TTV_ErrorCode ec, GrantVIPErrorCode error)>;

  /**
   * Used with RevokeVIP().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user successfully lost VIP status.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_GRAPHQL_ERROR: Received an error code from the back-end.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] error Error object received from GraphQL describing why revoking VIP failed.
   */
  using RevokeVIPCallback = std::function<void(TTV_ErrorCode ec, RevokeVIPErrorCode error)>;

  /**
   * Used with FetchChannelVIPs().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The channel's moderators were successfully retrieved.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] vipNames List of VIP login names, up to 100, the max number of VIPs in a channel.
   */
  using FetchChannelVIPsCallback = std::function<void(TTV_ErrorCode ec, std::vector<std::string>&& vipNames)>;

  /**
   * Used with UpdateUserColor().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The user's chat color was successfully changed.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   */
  using UpdateUserColorCallback = std::function<void(TTV_ErrorCode ec)>;

  /**
   * Used with FetchChannelModerators().
   *
   * @param[in] ec
   *   - TTV_EC_SUCCESS: The channel's moderators were successfully retrieved.
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON: Unable to parse the API response.
   *   - TTV_EC_API_REQUEST_FAILED: Unable to reach the API endpoint.
   * @param[in] modNames List of moderator login names, up the GraphQL pagination limit (currently 100).
   * @param[in] nextCursor The cursor to fetch the next batch of moderator names. Is empty string if no more mods to
   * fetch.
   */
  using FetchChannelModeratorsCallback =
    std::function<void(TTV_ErrorCode ec, const std::vector<std::string>& modNames, const std::string& nextCursor)>;

  ChatAPI();

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  /**
   * Sets the pointer to the core API.
   *
   * @pre Needs to be called before Initialize().
   * @param[in] coreApi A non-null pointer to the core API.
   * @return
   *     - TTV_EC_SUCCESS: The core api was set.
   *     - TTV_EC_ALREADY_INITIALIZED: Chat API is already initialized.
   */
  TTV_ErrorCode SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi);

  /**
   * Set the Chat API listener.
   *
   * @pre Needs to be called before Initalize().
   * @param[in] listener A pointer to the chat API listener.
   * @return
   *     - TTV_EC_SUCCESS: The listener was set.
   *     - TTV_EC_ALREADY_INITIALIZED: Chat API is already initialized.
   */
  TTV_ErrorCode SetListener(const std::shared_ptr<IChatAPIListener>& listener);

  /**
   * Set the enabled features for chat.
   *
   * @pre Needs to be called before Initalize().
   * @param[in] features Represents the features that should be enabled.
   * @return
   *     - TTV_EC_SUCCESS: The feature flags were set.
   *     - TTV_EC_ALREADY_INITIALIZED: Chat API is already initialized.
   */
  TTV_ErrorCode SetEnabledFeatures(const FeatureFlags& features);

  /**
   * Set the tokenization options for messages in chat.
   *
   * @pre Needs to be called before Initalize().
   * @param[in] features Lists which tokenization options are set.
   * @return
   *     - TTV_EC_SUCCESS: The tokenization options were set.
   *     - TTV_EC_ALREADY_INITIALIZED: Chat API is already initialized.
   */
  TTV_ErrorCode SetTokenizationOptions(const TokenizationOptions& options);

  /**
   * Set the tokenization options for messages in chat.
   *
   * @pre Needs to be called before Initalize().
   * @param[in] features Lists which tokenization options are set.
   * @return
   *     - TTV_EC_SUCCESS: The tokenization options were set.
   *     - TTV_EC_ALREADY_INITIALIZED: Chat API is already initialized.
   */
  TTV_ErrorCode SetUserThreadsListener(UserId userId, const std::shared_ptr<IChatUserThreadsListener>& listener);

  /**
   * Connect to the channel's chat.
   *
   * @param[in] userId The currently logged in user that is connecting to chat. Can be the anonymous user with ID of 0.
   * @param[in] channelId The channel of the chat the user is connecting to.
   * @param[in] listener Receives unsolicited events regarding the channel's chat.
   * @return
   *     - TTV_EC_SUCCESS: Connected to chat successfully.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_LOGIN: The user is not logged in.
   *     - TTV_EC_NEED_TO_LOGIN: Chat channel not found for user.
   *     - TTV_EC_INVALID_ARG: The listener cannot be null.
   *     - TTV_EC_INVALID_CALLBACK: If the channel already exists, the same listener must be passed in.
   *     - TTV_EC_CHAT_LEAVING_CHANNEL: We are currently disconnecting from the channel.
   *     - TTV_EC_CHAT_NO_HOSTS: No chat hosts were configured.
   *     - TTV_EC_NO_FACTORIES_REGISTERED: No factories registered.
   *     - TTV_EC_UNIMPLEMENTED: Registered factories unable to create socket for allowed chat protocols.
   */
  TTV_ErrorCode Connect(UserId userId, ChannelId channelId, const std::shared_ptr<IChatChannelListener>& listener);

  /**
   * Disconnect from the channel's chat.
   *
   * @param[in] userId The currently logged in user that is disconneting from chat. Can be the anonymous user with ID of
   * 0.
   * @param[in] channelId The channel of the chat the user is disconnecting from.
   * @return
   *     - TTV_EC_SUCCESS: Disconnected from chat successfully.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_LOGIN: The user is not logged in.
   *     - TTV_EC_NEED_TO_LOGIN: Chat channel not found for user.
   *     - TTV_EC_CHAT_NOT_IN_CHANNEL: We are not connected to the given channel.
   *     - TTV_EC_CHAT_LEAVING_CHANNEL: We are currently disconnecting from the channel, or the chat module is shutting
   * down.
   */
  TTV_ErrorCode Disconnect(UserId userId, ChannelId channelId);

  /**
   * Send a message to the channel's chat.
   *
   * @param[in] userId The currently logged in user that is sending the message.
   * @param[in] channelId The channel of the chat the user is sending a message to.
   * @param[in] message The content of the message being sent.
   * @return
   *     - TTV_EC_SUCCESS: Sent message successfully.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_CHANNEL_ID: Channel id cannot be 0.
   *     - TTV_EC_INVALID_ARG: Message cannot be the empty string.
   *     - TTV_EC_CHAT_NOT_IN_CHANNEL: Not connected to the channel.
   *     - TTV_EC_CHAT_ANON_DENIED: Anonymous user is not allowed to chat.
   *     - TTV_EC_CHAT_TOO_MANY_REQUESTS: Too many messages are already queued up to be sent.
   *     - TTV_EC_CHAT_RESTRICTED: Message does not follow broadcaster language restriction.
   *     - TTV_EC_CHAT_MESSAGE_SPAM_DISCARDED: Messages are being sent to quickly.
   */
  TTV_ErrorCode SendChatMessage(UserId userId, ChannelId channelId, const std::string& message);

  /**
   * Fetches the user list for the channel.
   *
   * @param[in] userId The currently logged in user who is fetching the user list.
   * @param[in] channelId The channel we're fetching the user list for.
   * @param[in] callback Function that receives the returned user list.
   * @return
   *     - TTV_EC_SUCCESS: The user list was fetched successfully.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_CHANNEL_ID: Channel id cannot be 0.
   *     - TTV_EC_CHAT_NOT_IN_CHANNEL: Not connected to the channel.
   *     - TTV_EC_AUTHENTICATION: The chat channel does not have an associated user.
   *     - TTV_EC_CHAT_LEAVING_CHANNEL: We are currently disconnecting from the channel, or the chat module is being
   * shut down.
   */
  TTV_ErrorCode FetchUserListForChannel(UserId userId, ChannelId channelId, const FetchUserListCallback& callback);

  /**
   * Block another user.
   *
   * @param[in] userId The currently logged in user.
   * @param[in] blockUserId The user being blocked.
   * @param[in] reason The reason for blocking the user.
   * @param[in] whisper True if blocking due to a whisper.
   * @param[in] callback Function that receives whether the blocking was successful.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to block the user.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_USERID: There is not a properly logged in user.
   *     - TTV_EC_SHUT_DOWN: The chat module is shut down.
   */
  TTV_ErrorCode BlockUser(
    UserId userId, UserId blockUserId, const std::string& reason, bool whisper, const BlockChangeCallback& callback);

  /**
   * Unblock another user.
   *
   * @param[in] userId The currently logged in user.
   * @param[in] blockUserId The user being unblocked.
   * @param[in] callback Function that receives whether the unblocking was successful.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to unblock the user.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_USERID: There is not a properly logged in user.
   *     - TTV_EC_SHUT_DOWN: The chat module is shut down.
   */
  TTV_ErrorCode UnblockUser(UserId userId, UserId blockUserId, const BlockChangeCallback& callback);

  /**
   * Check whether a user is blocked or not.
   *
   * @param[in] userId The currently logged in user.
   * @param[in] blockedUserId The other user that we're checking if blocked.
   * @param[out] blocked True if `blockUserId` is blocked by `userId`.
   * @return
   *     - TTV_EC_SUCCESS: Successfully returned the result in `blocked`.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_USERID: There is not a properly logged in user.
   *     - TTV_EC_SHUT_DOWN: The chat module is shut down.
   */
  TTV_ErrorCode GetUserBlocked(UserId userId, UserId blockUserId, bool& blocked);

  /**
   * Fetches the list of blocked users (UserInfo) for the given user.
   * This will not use a cached value and will fetch whenever this is called.
   *
   * @param[in] userId The currently logged in user whose blocked users is being fetched.
   * @param[in] callback Function that receives a list of the blocked users.
   * @return
   *     - TTV_EC_SUCCESS: Successfully added the request to fetch the blocked users.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_USERID: There is not a properly logged in user.
   *     - TTV_EC_SHUT_DOWN: The chat module is shut down.
   */
  TTV_ErrorCode FetchBlockedUsers(UserId userId, const FetchBlockedUsersCallback& callback);

  /**
   * Fetches the emoticon sets available to the user.
   * Calling this function may potentially fire IChatAPIListener::ChatUserEmoticonSetsChanged()
   *
   * @param[in] userId The currently logged in user whose emoticons are being fetched.
   * @param[in] forceRefetch If false, will potentially call the callback immediately with cached values. Otherwise will
   * force a fetch for latest values.
   * @param[in] callback Function that receives a list of emoticon sets for the user.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to fetch the emoticon sets for the user.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_FEATURE_DISABLED: Emoticons are disabled.
   *     - TTV_EC_INVALID_LOGIN: The given user ID is not logged in.
   */
  TTV_ErrorCode FetchUserEmoticonSets(UserId userId, bool forceRefetch, const FetchUserEmoticonSetsCallback& callback);

  /**
   * Fetches the global badge sets. May use a cached value.
   *
   * @param[in] callback Function that receives the global badge set information.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to fetch the global badges.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_SHUTTING_DOWN: Unable to add the request as a task.
   */
  TTV_ErrorCode FetchGlobalBadges(const FetchBadgesCallback& callback);

  /**
   * Fetches the badge sets for the given channel. May use a cached value.
   *
   * @param[in] channelId The channel we're fetching badges for.
   * @param[in] callback Function that takes in the channel badge set information.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to fetch the channel badges.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_SHUTTING_DOWN: Unable to add the request as a task.
   */
  TTV_ErrorCode FetchChannelBadges(ChannelId channelId, const FetchBadgesCallback& callback);

  /**
   * Fetches the global bits configuration object. This object contains info about the various bits actions (cheers).
   * This value is cached over time, but the initial fetch can take some time.
   *
   * @param[in] callback Function that takes in the bits configuration information.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to fetch the global bits configuration.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   */
  TTV_ErrorCode FetchGlobalBitsConfiguration(const FetchBitsConfigurationCallback& callback);

  /**
   * Fetches the bits configuration object specifically for this channel.
   * The BitsConfiguration object we get back may contain additional actions/cheers specifically for the channel.
   * This value is cached for each channel.
   *
   * @param[in] userId The user requesting the cheermotes. Can be the anonymous user with ID 0.
   * @param[in] channelId The channel we want to fetch the bits configuration for.
   * @param[in] callback Function that takes in the bits configuration information.
   * @return
   *     - TTV_EC_SUCCESS: Successfully sent the request to fetch the channel bits configuration.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   */
  TTV_ErrorCode FetchChannelBitsConfiguration(
    UserId userId, ChannelId channelId, const FetchBitsConfigurationCallback& callback);

  /**
   * Set how interval duration for flushing messages.
   *
   * @param[in] milliseconds The interval duration to be set to.
   * @return
   *     - TTV_EC_SUCCESS: Successfully set the new message flush interval duration.
   *     - TTV_EC_SHUTTING_DOWN: Chat module is shutting down.
   */
  TTV_ErrorCode SetMessageFlushInterval(uint64_t milliseconds);

  /**
   * Get the current interval duration for flushing messages.
   *
   * @return 0 if there was an error, otherwise, the interval duration in milliseconds.
   */
  uint64_t GetMessageFlushInterval() const;

  /**
   * Generate the url for an emoticon with given ID and scale factor.
   *
   * @param[in] emoticonId The emoticon we're generating the url for.
   * @param[in] scale The scale factor for the emoticon.
   * @param[out] result The url for the emoticon.
   * @return
   *     - TTV_EC_SUCCESS: The url was returned in `result`.
   */
  TTV_ErrorCode GetEmoticonUrl(const std::string& emoticonId, float scale, std::string& result);

  /**
   * Returns an implementation of IChatChannel for connecting to a channel on behalf of a user.
   *
   * @param[in] userId The currently logged in user. Can be the anonymous user with ID of 0.
   * @param[in] channelId The channel the user wants the object for.
   * @param[in] listener Listener for unsolicited events on the chat channel (raids, hosting, subscriptions, info
   * changes, etc.)
   * @param[out] result The chat channel object for the channel.
   * @return
   *     - TTV_EC_SUCCESS: The chat channel object was returned in `result`.
   *     - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *     - TTV_EC_INVALID_ARG: Channel ID cannot be 0. Listener cannot be null.
   *     - TTV_EC_NEED_TO_LOGIN: There is not a properly logged in user.
   */
  TTV_ErrorCode CreateChatChannel(UserId userId, ChannelId channelId,
    const std::shared_ptr<IChatChannelListener>& listener, std::shared_ptr<IChatChannel>& result);

  /**
   * Returns an implementation of IChatChannelProperties for a given channelId. This object is used to receive
   * unsolicited chat channel property updates for a channel. Property updates that can be listened for:
   *   - Whether or not rituals is enabled/disabled (RitualsEnabled)
   *   - Whenever the host target changes (HostTargetChange)
   *
   * @param[in] userId The id of the user creating the ChatChannelProperties object. Can be the anonymous user with ID
   * of 0.
   * @param[in] channelId The id of the channel we are receiving property updates for.
   * @param[in] listener Listener for the user. Called when certain properties (listed above) are updated on a channel.
   * @param[out] result Stores the resulting ChatChannelProperties object.
   * @return
   *     - TTV_EC_SUCCESS: The result has been populated with a ChatChannelProperties object to use.
   *     - TTV_EC_NOT_INITIALIZED: If chat hasn't been initialized properly yet.
   *     - TTV_EC_INVALID_ARG: Channel ID cannot be 0. Listener cannot be null.
   *     - TTV_EC_NEED_TO_LOGIN: If there is not a properly logged in user.
   */
  TTV_ErrorCode CreateChatChannelProperties(UserId userId, ChannelId channelId,
    const std::shared_ptr<IChatChannelPropertyListener>& listener, std::shared_ptr<IChatChannelProperties>& result);

  /**
   * Returns an implementation of IChatCommentManager for the given owning user and VOD.
   *
   * @param[in] userId The id of the user creating the ChatCommentManager object.
   * @param[in] vodId The id of the VOD that is being watched.
   * @param[in] listener Listener for chat comments, called when the manager changes state or when comments are
   * received.
   * @return Result containing an error code and an IChatCommentManager object if successful.
   *   - TTV_EC_SUCCESS: The result has been populated with an IChatCommentManager to use.
   *   - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *   - TTV_EC_INVALID_ARG: VOD ID cannot be the empty string. Listener cannot be null.
   *   - TTV_EC_NEED_TO_LOGIN: There is not a properly logged in user.
   */
  Result<std::shared_ptr<IChatCommentManager>> CreateChatCommentManager(
    UserId userId, const std::string& vodId, const std::shared_ptr<IChatCommentListener>& listener);

  /**
   * Returns the VOD comment settings for the given channel.
   *
   * @param[in] userId The id of the user who's trying to fetch the comment settings. Must be the owner or moderator of
   * the channelId.
   * @param[in] channelId The id of the channel we're trying to fetch the settings for.
   * @param[in] callback Function that receives channel's VOD comment settings.
   * @return
   *   - TTV_EC_SUCCESS: Successfully sent the request to fetch the VOD comment settings.
   *   - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *   - TTV_EC_INVALID_LOGIN: There is not a properly logged in user.
   */
  TTV_ErrorCode FetchChannelVodCommentSettings(
    UserId userId, ChannelId channelId, FetchChannelVodCommentSettingsCallback&& callback);

  /**
   * Sets the minimum required duration a viewer must have followed the channel for in order to post comments on VODs
   * for the channel.
   *
   * @param[in] userId The id of the user who's trying to set the comment settings. Must be the owner or moderator of
   * the channelId.
   * @param[in] channelId The id of the channel we're trying to set the settings for.
   * @param[in] followersOnlyDurationSeconds The minimum number of seconds that someone must have followed the channel
   * for in order to post comments on VODs.
   * @param[in] callback Function that receives the channel's new VOD comment settings.
   * @return
   *   - TTV_EC_SUCCESS: Successfully sent the request to set the followers-only duration.
   *   - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *   - TTV_EC_INVALID_LOGIN: There is not a properly logged in user.
   */
  TTV_ErrorCode SetChannelVodFollowersOnlyDuration(UserId userId, ChannelId channelId,
    uint32_t followersOnlyDurationSeconds, SetChannelVodCommentSettingsCallback&& callback);

  /**
   * Sets the publishing mode for VODs for the given channel. Comments can be disabled or set to be held until reviewed.
   *
   * @param[in] userId The id of the user who's trying to set the publishing mode. Must be the owner or moderator of the
   * channelId.
   * @param[in] channelId The id of the channel we're trying to set the settings for.
   * @param[in] mode The publishing mode for comments on VODs that we want to set for the channel.
   * @param[in] callback Function that receives the channel's new VOD comment settings.
   * @return
   *   - TTV_EC_SUCCESS: The settings were set and the result passed to the callback function.
   *   - TTV_EC_NOT_INITIALIZED: Chat API is not initialized.
   *   - TTV_EC_INVALID_LOGIN: There is not a properly logged in user.
   */
  TTV_ErrorCode SetChannelVodPublishingMode(
    UserId userId, ChannelId channelId, CommentPublishingMode mode, SetChannelVodCommentSettingsCallback&& callback);

  /**
   * Returns an implementation of IChannelChatRoomManager for the given user.
   * This object handles any functions that aren't associated with a specific chat room, such as what rooms a user is
   * connected to.
   *
   * @param[in] userId The id of the user we're creating the ChannelChatRoomManager object for.
   * @param[in] channelId The id of the channel that owns the chat rooms that the user is entering.
   * @param[in] listener Listener for user chat rooms. Called when a user is kicked/banned/timed out of a room, is
   * mentioned, or has their view changed.
   * @param[out] result Stores the resulting ChannelChatRoomManager object.
   * @return
   *   - TTV_EC_SUCCESS: The result has been populated with a IChannelChatRoomManager object to use.
   *   - TTV_EC_NOT_INITIALIZED: If chat hasn't been initialized properly yet.
   *   - TTV_EC_NEED_TO_LOGIN: If there is not a properly logged in user.
   */
  TTV_ErrorCode CreateChannelChatRoomManager(UserId userId, ChannelId channelId,
    const std::shared_ptr<IChannelChatRoomManagerListener>& listener, std::shared_ptr<IChannelChatRoomManager>& result);

  /**
   * Returns an implementation of IChatRoom for the given room id. This object is used to manage a specific chat room.
   *
   * @param[in] userId The id of the user creating the chat room object.
   * @param[in] roomId The id of the room that the ChatRoom object is representing.
   * @param[in] channelId The channel that owns the created chat room.
   * @param[in] listener Listener for the chat room. Called when a message is sent/edited/deleted, or when the room
   * information has been updated.
   * @param[out] result Stores the resulting ChatRoom object.
   * @return
   *   - TTV_EC_SUCCESS: The result has been populated with a IChatRoom object to use.
   *   - TTV_EC_NOT_INITIALIZED: If chat hasn't been initialized properly yet.
   *   - TTV_EC_NEED_TO_LOGIN: If there is not a properly logged in user.
   */
  TTV_ErrorCode CreateChatRoom(UserId userId, const std::string& roomId, ChannelId channelId,
    const std::shared_ptr<IChatRoomListener>& listener, std::shared_ptr<IChatRoom>& result);

  /**
   * Returns an implementation of IChatRoomNotifications for the given user id. This object is used to receive
   * unsolicited chat room events for the user.
   *
   * @param[in] userId The id of the user creating the chat room notifications object. Cannot be the anonymous user.
   * @param[in] listener Listener for the user. Called when the user is banned/timed out, or when the user's room view
   * has been updated.
   * @param[out] result Stores the resulting ChatRoomNotifications object.
   * @return
   *   - TTV_EC_SUCCESS: The result has been populated with a IChatRoomNotifications object to use.
   *   - TTV_EC_NOT_INITIALIZED: If chat hasn't been initialized properly yet.
   *   - TTV_EC_NEED_TO_LOGIN: If there is not a properly logged in user.
   */
  TTV_ErrorCode CreateChatRoomNotifications(UserId userId,
    const std::shared_ptr<IChatRoomNotificationsListener>& listener, std::shared_ptr<IChatRoomNotifications>& result);

  /**
   * Returns an implementation of IChatRaid for the given owning user and channel, which is used later to start, join,
   * leave, or cancel a raid.
   *
   * @param[in] userId The id of the user creating the raid object.
   *  If the anonymous user is passed in, you can listen for raid updates but not start or join a raid.
   *  To start a raid after creation, the user must be the broadcaster or moderator of the channel.
   * @param[in] channelId The source channel id that the raid is starting from.
   * @param[in] listener Listener for the raid with callback functions when the raid is started, updated, completed, and
   * cancelled.
   * @param[out] result Stores the resulting ChatRaid object.
   * @return
   *   - TTV_EC_SUCCESS: The result has been populated with an IChatRaid object to use.
   *   - TTV_EC_NOT_INITIALIZED: If chat hasn't been initialized properly yet.
   *   - TTV_EC_NEED_TO_LOGIN: If there is not a properly logged in user.
   */
  TTV_ErrorCode CreateChatRaid(UserId userId, ChannelId channelId, const std::shared_ptr<IChatRaidListener>& listener,
    std::shared_ptr<IChatRaid>& result);

  /**
   * Returns an implementation of IBitsStatus for the user. This object is used to recieve bits updates for a user.
   * Bits updates that can be listened for:
   *   - When the user receives bits on their channel (BitsReceived). Only relevant for partners/affiliates who are able
   * to receive bits.
   *   - When the current user cheers bits on a different channel (BitsSent).
   *   - When the current user's bits balance changes (BitsBalanceUpdated).
   *
   * @param[in] userId The currently logged in user creating the bits status.
   * @param[in] listener The listener to receive bits updates for a user/channel.
   * @param[out] result BitsStatus object for the given user to receive bits updates on. This object maintains the
   * lifetime of the listener associated it.
   * @return
   *   - TTV_EC_SUCCESS: We've initialized and created BitsStatus successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: Cannot be the anonymous user. Channel ID cannot be 0. Listener cannot be null.
   */
  TTV_ErrorCode CreateBitsStatus(
    UserId userId, const std::shared_ptr<IBitsListener>& listener, std::shared_ptr<IBitsStatus>& result);

  /**
   * Returns an implementation of IFollowersStatus for the user.
   *
   * @param[in] userId The currently logged in user creating the followers status object. Can be the anonymous user with
   * ID of 0.
   * @param[in] channelId The channel we care about listening to for follows.
   * @param[in] listener The listener to receive info about follows in the channel.
   * @param[out] result FollowersStatus object for the given user/channel to receive follower updates on. This object
   * maintains the lifetime of the listener associated it.
   * @return
   *   - TTV_EC_SUCCESS: We've initialized and created FollowersStatus successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: Channel ID cannot be 0. Listener cannot be null.
   */
  TTV_ErrorCode CreateFollowersStatus(UserId userId, ChannelId channelId,
    const std::shared_ptr<IFollowersListener>& listener, std::shared_ptr<IFollowersStatus>& result);

  /**
   * Returns an implementation of IFollowingStatus for the user.
   *
   * @param[in] userId The currently logged in user creating the following status object.
   * @param[in] listener The listener to receive info about the user's follows.
   * @param[out] result FollowingStatus object to receive updates on the current user's follows. This object maintains
   * the lifetime of the listener associated it.
   * @return
   *   - TTV_EC_SUCCESS: We've initialized and created FollowingStatus successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: Listener cannot be null. UserId cannot be 0.
   */
  TTV_ErrorCode CreateFollowingStatus(
    UserId userId, const std::shared_ptr<IFollowingListener>& listener, std::shared_ptr<IFollowingStatus>& result);

  /**
   * Returns an implementation of ISubscribersStatus for the user.
   * This is only relevant for partners/affiliates who are able to have subscribers.
   *
   * @param[in] userId The currently logged in user creating the subscribers status object.
   * @param[in] listener The listener to receive info about subscribers in the channel.
   * @param[out] result SubscribersStatus object for the given user to receive subscriber updates on. This object
   * maintains the lifetime of the listener associated it.
   * @return
   *   - TTV_EC_SUCCESS: We've initialized and created SubscribersStatus successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: User ID nor channel ID cannot be 0. Listener cannot be null.
   */
  TTV_ErrorCode CreateSubscribersStatus(
    UserId userId, const std::shared_ptr<ISubscribersListener>& listener, std::shared_ptr<ISubscribersStatus>& result);

  /**
   * Returns an implementation of ISubscriptionsNotifications to listen for a user's new subscriptions.
   *
   * @param[in] userId The currently logged in user. Cannot be the anonymous user.
   * @param[in] listener Listener for when a user has a new subscription.
   * @return Result containing an error code and an ISubscriptionsNotifications if successful.
   *   - TTV_EC_SUCCESS: We've initialized and created ISubscriptionsNotifications successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: User ID cannot be 0. Listener cannot be null.
   */
  Result<std::shared_ptr<ISubscriptionsNotifications>> CreateSubscriptionsNotifications(
    UserId userId, const std::shared_ptr<ISubscriptionsNotificationsListener>& listener);

  /**
   * Returns an implementation of ISquadNotifications to listen for changes on a squad.
   *
   * @param[in] userId The currently logged in user.
   * @param[in] squadId The squad we are watching.
   * @param[in] listener Listener for when the squad changes.
   * @return Result containing an error code and an ISquadNotifications if successful.
   *   - TTV_EC_SUCCESS: We've initialized and created ISquadNotifications successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: Listener cannot be null.
   */
  Result<std::shared_ptr<ISquadNotifications>> CreateSquadNotifications(
    UserId userId, const std::string& squadId, const std::shared_ptr<ISquadNotificationsListener>& listener);

  /**
   * Returns an implementation of IMultiviewNotifications to listen for changes on a multiview stream.
   *
   * @param[in] userId The currently logged in user.
   * @param[in] channelId The multi-view stream we're watching.
   * @param[in] listener Listener for when the multi-view stream changes.
   * @return Result containing an error code and an IMultiviewNotifications if successful.
   *   - TTV_EC_SUCCESS: We've initialized and created IMultiviewNotifications successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: Listener cannot be null.
   */
  Result<std::shared_ptr<IMultiviewNotifications>> CreateMultiviewNotifications(
    UserId userId, ChannelId channelId, const std::shared_ptr<IMultiviewNotificationsListener>& listener);

  /**
   * Bans or timeouts the given user name from the given channel.
   *
   * @param[in] userId The id of the user who is issuing the ban. Must be owner or moderator of the channel.
   * @param[in] channelId The id of the channel the user is being banned from.
   * @param[in] bannedUserName The name of the user who is being banned.
   * @param[in] durationSeconds The duration of the ban/timeout in seconds. Value is 0 for a permanent ban.
   * @param[in] callback Receives an error code indicating the ban's success.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the ban request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode BanUser(UserId userId, ChannelId channelId, const std::string& bannedUserName, uint32_t durationSeconds,
    const BanUserCallback& callback);

  /**
   * Unbans or untimeouts the given user name from the given channel.
   *
   * @param[in] userId The id of the user who is reverting the ban. Must be owner or moderator of the channel.
   * @param[in] channelId The id of the channel the user is being unbanned from.
   * @param[in] bannedUserName The name of the user who is being unbanned.
   * @param[in] callback Receives an error code indicating the unban's success.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the unban request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode UnbanUser(
    UserId userId, ChannelId channelId, const std::string& bannedUserName, const UnbanUserCallback& callback);

  /**
   * Mods the given user name in the given channel.
   *
   * @param[in] userId The id of the user who is granting the mod privilege.
   * @param[in] channelId The id of the channel that the target user is becoming a mod of.
   * @param[in] modUserName The name of the user who is becoming a mod.
   * @param[in] callback Receives an error code indicating the success of the modding.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the mod request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode ModUser(
    UserId userId, ChannelId channelId, const std::string& modUserName, const ModUserCallback& callback);

  /**
   * Unmods the given user name in the given channel.
   *
   * @param[in] userId The id of the user who is issuing the unmod request.
   * @param[in] channelId The id of the channel that the target user is being unmodded from.
   * @param[in] unmodUserName The name of the user who is being unmodded.
   * @param[in] callback Receives an error code indicating the unmod's success.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the unmod request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode UnmodUser(
    UserId userId, ChannelId channelId, const std::string& unmodUserName, const UnmodUserCallback& callback);

  /**
   * Grants the target user VIP status in the given channel.
   *
   * @param[in] userId The id of the user who is granting the VIP status.
   * @param[in] channelId The id of the channel that the target user is becoming a VIP of.
   * @param[in] modUserName The name of the user who is becoming a VIP.
   * @param[in] callback Receives an error code indicating the success of the granting.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the VIP request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode GrantVIP(
    UserId userId, ChannelId channelId, const std::string& vipUserName, const GrantVIPCallback& callback);

  /**
   * Removes the target user's VIP status in the given channel.
   *
   * @param[in] userId The id of the user who is issuing the unvip request.
   * @param[in] channelId The id of the channel that the target user is losing their VIP status from.
   * @param[in] unmodUserName The name of the user who is losing their VIP status.
   * @param[in] callback Receives an error code indicating the unmod's success.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the unvip request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode RevokeVIP(
    UserId userId, ChannelId channelId, const std::string& unvipUserName, const RevokeVIPCallback& callback);

  /**
   * Changes the user's chat color.
   *
   * @param[in] userId The id of the user who is changing their color.
   * @param[in] color The new chat color of the user.
   * @param[in] callback Receives an error code indicating the change's success.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the ban request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode UpdateUserColor(UserId userId, const std::string& color, const UpdateUserColorCallback& callback);

  /**
   * Fetches the names of moderators in a channel.
   *
   * @param[in] channelId The id of the channel we're fetching moderators for.
   * @param[in] cursor The cursor to start fetching moderators names from. To start from the beginning, use empty
   * string.
   * @param[in] callback Receives an error code indicating the query's success and a list of the names of the
   * moderators.
   * @return
   *   - TTV_EC_SUCCESS: We've sent the fetch moderators request successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode FetchChannelModerators(
    ChannelId channelId, const std::string& cursor, const FetchChannelModeratorsCallback& callback);

  /**
   * Returns the names of all VIPs in the channel in the callback.
   *
   * @param[in] channelId The id of the channel we're fetching VIPs for.
   * @param[in] callback Receives and error code indiciating the query's success and a list of names of the VIPs.
   */
  TTV_ErrorCode FetchChannelVIPs(ChannelId channelId, const FetchChannelVIPsCallback& callback);

  /**
   * Adds to the `scopes` list the required scopes that Chat API requires.
   *
   * @param[in,out] scopes The old values in scopes, with the new scopes required by Chat API appended to the end.
   */
  static void GetRequiredAuthScopes(std::vector<std::string>& scopes);

  /**
   * Set the chat object factory.
   *
   * @param[in] factory Set the chat object factory. If `factory` is null, Chat API will use a default implementation.
   */
  void SetChatObjectFactory(const std::shared_ptr<IChatObjectFactory>& factory);

 private:
  class CoreApiClient : public ICoreApiClient {
   public:
    CoreApiClient(ChatAPI* api);

    // ICoreApiClient implementation
    virtual std::string GetClientName() override;
    virtual void GetRequiredOAuthScopes(std::vector<std::string>& scopes) override;
    virtual void CoreUserLoggedIn(std::shared_ptr<User> user) override;
    virtual void CoreUserLoggedOut(std::shared_ptr<User> user) override;

   private:
    ChatAPI* mApi;
  };

  TTV_ErrorCode GetChannelSet(UserId userId, std::shared_ptr<ChatChannelSet>& channelSet);

  std::shared_ptr<ChatUserBlockList> CreateUserBlockList(const std::shared_ptr<User>& user);

  // ICoreApiClient implementation
  void CoreUserLoggedIn(std::shared_ptr<User> user);
  void CoreUserLoggedOut(std::shared_ptr<User> user);

  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  TTV_ErrorCode GetBlockListForUser(UserId userId, std::shared_ptr<ChatUserBlockList>& blockList);
  TTV_ErrorCode GetUserThreadsForUser(UserId userId, std::shared_ptr<ChatUserThreads>& userThreads);

  static TTV_ErrorCode DisposeChatChannel(
    const std::shared_ptr<IChatChannel>& chatChannel, const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeChatChannelProperties(
    const std::shared_ptr<IChatChannelProperties>& chatChannelProperties,
    const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeChatCommentManager(const std::shared_ptr<IChatCommentManager>& chatCommentManager,
    const std::shared_ptr<ChatAPIInternalData>& internalData,
    const std::shared_ptr<ComponentContainer>& componentContainer);
  static TTV_ErrorCode DisposeChatRaid(
    const std::shared_ptr<IChatRaid>& chatRaid, const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeChannelChatRoomManager(
    const std::shared_ptr<IChannelChatRoomManager>& channelChatRoomManager,
    const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeChatRoom(
    const std::shared_ptr<IChatRoom>& chatRoom, const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeChatRoomNotifications(
    const std::shared_ptr<IChatRoomNotifications>& chatRoomNotifications,
    const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeBitsStatus(
    const std::shared_ptr<IBitsStatus>& bitsStatus, const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeFollowersStatus(
    const std::shared_ptr<IFollowersStatus>& followersStatus, const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeFollowingStatus(
    const std::shared_ptr<IFollowingStatus>& followingStatus, const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeSubscribersStatus(const std::shared_ptr<ISubscribersStatus>& subscribersStatus,
    const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeSubscriptionsNotifications(
    const std::shared_ptr<ISubscriptionsNotifications>& subscriptionsNotifications,
    const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeSquadNotifications(const std::shared_ptr<ISquadNotifications>& squadNotifications,
    const std::shared_ptr<ChatAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeMultiviewNotifications(
    const std::shared_ptr<IMultiviewNotifications>& multiviewNotifications,
    const std::shared_ptr<ChatAPIInternalData>& internalData);

  std::shared_ptr<CoreAPI> mCoreApi;
  std::shared_ptr<CoreApiClient> mCoreApiClient;
  std::shared_ptr<ChatAPIInternalData> mInternalData;
  std::shared_ptr<IChatObjectFactory> mChatObjectFactory;  //!> The internal chat object factory.
  std::shared_ptr<ChannelRepository> mChannelRepository;   //!> The cache of channel info.
  std::shared_ptr<UserRepository> mUserRepository;         //!> The set of all local users.
  std::shared_ptr<SettingRepository> mSettingRepository;
  std::shared_ptr<BitsConfigRepository> mBitsConfigRepository;
  std::shared_ptr<ChatAPITaskHost> mTaskHost;

  std::shared_ptr<IEventScheduler> mMainEventScheduler;
  std::shared_ptr<TaskRunner> mTaskRunner;  //!> The general task runner to use for api calls.
  std::shared_ptr<TaskRunner>
    mUserBlocksTaskRunner;  //!> The task runner to use for block lists.  Once we rework the way httprequest queueing
                            //! and retried works we can use the general one.

  FeatureFlags mEnabledFeatures;
  TokenizationOptions mTokenizationOptions;
};
