/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/errortypes.h"

#include <vector>

/**
 * The unique module identifier used in Chat.
 */
const uint32_t TTV_MODULE_ID_TWITCH_CHAT = 1;

#define TTV_CHAT_ERROR_IDS                                                                                                                                                                 \
  /* Errors */                                                                                                                                                                             \
  BEGIN_ERROR_IDS(TTV_EC_CHAT_BEGIN_ERRORS)                                                                                                                                                \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_NO_HOSTS)           /*!< No hosts to connect to. */                                                                                                         \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_ALREADY_IN_CHANNEL) /*!< Already in the channel. */                                                                                                         \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_LOST_CONNECTION)    /*!< Lost connection to the chat server. */                                                                                             \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_COULD_NOT_CONNECT)  /*!< Could not connect to the chat server. */                                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_NOT_IN_CHANNEL)     /*!< Not in the named channel. */                                                                                                       \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_INVALID_MESSAGE)    /*!< Not a valid message. */                                                                                                            \
  USE_TTV_ERROR_ID(                                                                                                                                                                        \
    TTV_EC_CHAT_TOO_MANY_REQUESTS) /*!< The request queue is growing too large and the request has been ignored. */                                                                        \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_LEAVING_CHANNEL) /*!< In the middle of leaving the channel. */                                                                                              \
  USE_TTV_ERROR_ID(                                                                                                                                                                        \
    TTV_EC_CHAT_ANON_DENIED) /*!< Attempted to perform an action that anonymous connections are not allowed to do. */                                                                      \
  USE_TTV_ERROR_ID(                                                                                                                                                                        \
    TTV_EC_CHAT_RESTRICTED) /*!< The sending of the message failed because chatting is restricted based on the current state of the channel info. */                                       \
  USE_TTV_ERROR_ID(                                                                                                                                                                        \
    TTV_EC_CHAT_MESSAGE_SPAM_DISCARDED) /*!< The client is submitting chat messages too quickly and the message was discarded.*/                                                           \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_EMOTICON_DATA_NOT_READY)   /*!< The emoticon data is not ready to use. */                                                                                   \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_EMOTICON_DATA_DOWNLOADING) /*!< The emoticon data is already downloading. */                                                                                \
  USE_TTV_ERROR_ID(                                                                                                                                                                        \
    TTV_EC_CHAT_EMOTICON_DATA_LOCKED) /*!< The emoticon data has been locked by the client by a call to TTV_Chat_GetEmoticonData which has not been freed by TTV_Chat_FreeEmoticonData. */ \
  USE_TTV_ERROR_ID(                                                                                                                                                                        \
    TTV_EC_CHAT_EMOTICON_DOWNLOAD_FAILED) /*!< There were too many download errors while trying to fetch the emoticon data. */                                                             \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_INVALID_CHAT_THREAD_ID)  /*!< The specified conversation thread id is invalid. */                                                                           \
  USE_TTV_ERROR_ID(TTV_EC_CHAT_INVALID_CHAT_MESSAGE_ID) /*!< The specified conversation message id is invalid. */                                                                          \
  END_ERROR_IDS(TTV_EC_CHAT_END_ERRORS)

#define USE_TTV_ERROR_ID(ec) ec,
#define USE_TTV_SUCCESS_ID(ec) ec = 0,
#define BEGIN_ERROR_IDS(ec) ec = CONSTRUCT_ERROR_ID_VALUE(TTV_MODULE_ID_TWITCH_CHAT, 1),
#define END_ERROR_IDS(ec) ec

/**
 * TTV_ChatErrorId - The error ids defined in chat.
 */
enum TTV_ChatErrorId { TTV_CHAT_ERROR_IDS };

#undef BEGIN_ERROR_IDS
#undef END_ERROR_IDS
#undef USE_TTV_SUCCESS_ID
#undef USE_TTV_ERROR_ID

namespace ttv {
namespace chat {
/**
 * The error code string lookup for chat.  This is not normally used directly by clients.  Use ttv::ErrorToString()
 * instead.
 */
const char* ChatErrorToString(TTV_ErrorCode ec);

void GetChatErrorCodeValues(std::vector<EnumValue>& result);

enum class GraphQLErrorCode {
  SUCCESS,                             //!< No error.
  UNKNOWN,                             //!< Received an unrecognized error code.
  AUTO_MOD_ENFORCEMENT_FAILED,         //!< Message failed automod check. sendRoomMessage.
  CHANNEL_NOT_FOUND,                   //!< Target channel does not exist. modUser, unmodUser.
  DURATION_INVALID,                    //!< Duration provided was too short or too long. banUser.
  EMOTE_ONLY_MODE_ENFORCEMENT_FAILED,  //!< Room is in emotes only mode and message contained non-emote text. sendRoomMessage.
  FORBIDDEN,                           //!< User is not allowed to perform the action.
  MAX_ROOMS_LIMIT_EXCEEDED,            //!< User cannot create any more rooms. createRoom.
  NAME_CONTAINS_INVALID_CHARACTERS,  //!< Name contains invalid characters (e.g. spaces). createRoom, updateRoom.
  NAME_INAPPROPRIATE,                //!< Name failed automod check. createRoom, updateRoom.
  NAME_LENGTH_INVALID,               //!< Name is too short or too long. createRoom, updateRoom.
  R9K_MODE_ENFORCEMENT_FAILED,       //!< Room is in R9K mode and message was not unique. sendRoomMessage.
  RATE_LIMIT_FAILED,                 //!< User is rate limited. sendRoomMessage.
  ROLES_INVALID,   //!< Attempted to create/update a room that would have an invalid combination of role permissions.
  ROOM_NOT_FOUND,  //!< Could not find a room with the given room ID. updateRoom. updateRoomModes.
  SLOW_MODE_DURATION_INVALID,    //!< Slow mode duration was too short, too long, or not a number. updateRoomModes.
  SLOW_MODE_ENFORCEMENT_FAILED,  //!< Room is in slow mode and user tried to send a message too quickly. sendRoomMessage.
  SPAM_ENFORCEMENT_FAILED,       //!< Message failed spam check. sendRoomMessage.
  TARGET_ALREADY_BANNED,         //!< User tried to ban already banned user. banUser.
  TARGET_ALREADY_MOD,            //!< Target is already a mod. modUser.
  TARGET_IS_ADMIN,               //!< User tried to ban admin. banUser.
  TARGET_IS_ANONYMOUS,           //!< User tried to ban anonymous user. banUser.
  TARGET_IS_BROADCASTER,         //!< User tried to ban broadcaster. banUser.
  TARGET_IS_CHAT_BANNED,         //!< User is banned from chat. modUser.
  TARGET_IS_GLOBAL_MOD,          //!< User tried to ban admin. banUser.
  TARGET_IS_MOD,                 //!< User tried to ban moderator without permission. banUser.
  TARGET_IS_SELF,                //!< User tried to ban themselves. banUser.
  TARGET_IS_STAFF,               //!< User tried to ban staff. banUser.
  TARGET_NOT_BANNED,             //!< Target is not banned in channel. unbanUser.
  TARGET_NOT_FOUND,              //!< Target does not exist. banUser, unbanUser, modUser, unmodUser.
  TARGET_NOT_MOD,                //!< Target is not a mod. unmodUser.
  TOPIC_INAPPROPRIATE,           //!< Topic failed automod check. createRoom, updateRoom.
  TOPIC_LENGTH_INVALID,          //!< Topic is too short or too long. createRoom, updateRoom.
  USER_CHAT_BANNED,              //!< User is banned from the channel. sendRoomMessage.
  USER_CHAT_TIMED_OUT,           //!< User is timed out from the channel sendRoomMessage.
  USER_SUSPENDED,                //!< User is suspended from Twitch. sendRoomMessage.
  ZALGO_ENFORCEMENT_FAILED,      //!< Message contained zalgo characters. sendRoomMessage.
};

struct SendRoomMessageError {
  SendRoomMessageError();

  GraphQLErrorCode code;  //!< Code describing the error.
  uint32_t
    slowModeDurationSeconds;  //!< On SLOW_MODE_ENFORCEMENT_FAILED, number of seconds a user must wait between sending messages.
  uint32_t
    remainingDurationSeconds;  //!< On SLOW_MODE_ENFORCMENT_FAILED, USER_CHAT_TIMED_OUT, the number of remaining seconds a user must wait before sending another message.
};

struct CreateRoomError {
  CreateRoomError();

  GraphQLErrorCode code;     //!< Code describing the error.
  uint32_t maxAllowedRooms;  //!< On MAX_ROOMS_LIMIT_EXCEEDED, max number of rooms user can create.
  uint32_t minLength;        //!< On NAME_LENGTH_INVALID, TOPIC_LENGTH_INVALID, minimum number of characters.
  uint32_t maxLength;        //!< On NAME_LENGTH_INVALID, TOPIC_LENGTH_INVALID, maximum number of characters.
};

struct UpdateRoomError {
  UpdateRoomError();

  GraphQLErrorCode code;  //!< Code describing the error.
  uint32_t minLength;     //!< On NAME_LENGTH_INVALID, TOPIC_LENGTH_INVALID, minimum number of characters.
  uint32_t maxLength;     //!< On NAME_LENGTH_INVALID, TOPIC_LENGTH_INVALID, maximum number of characters.
};

struct UpdateRoomModesError {
  UpdateRoomModesError();

  GraphQLErrorCode code;  //!< Code describing the error.
  uint32_t
    minimumSlowModeDurationSeconds;  //!< On SLOW_MODE_DURATION_INVALID, minimum number of seconds slow mode can be set to.
  uint32_t
    maximumSlowModeDurationSeconds;  //!< On SLOW_MODE_DURATION_INVALID, maximum number of seconds slow mode can be set to.
};

struct BanUserError {
  BanUserError();

  GraphQLErrorCode code;  //!< Code describing the error.
};

struct UnbanUserError {
  UnbanUserError();

  GraphQLErrorCode code;  //!< Code describing the error.
};

struct ModUserError {
  ModUserError();

  GraphQLErrorCode code;  //!< Code describing the error.
};

struct UnmodUserError {
  UnmodUserError();

  GraphQLErrorCode code;  //!< Code describing the error.
};

enum class GrantVIPErrorCode {
  SUCCESS,                     //!< No error.
  UNKNOWN,                     //!< Received an unrecognized error code.
  CHANNEL_NOT_FOUND,           //!< Channel does not exist.
  FORBIDDEN,                   //!< The granter does not have permisison to grant VIP status in the channel.
  GRANTEE_ALREADY_VIP,         //!< The target user is already a VIP
  GRANTEE_CHAT_BANNED,         //!< The target user is currently banned or timed out in the channel.
  GRANTEE_NOT_FOUND,           //!< The target user does not exist.
  MAX_VIPS_REACHED,            //!< The channel has the maximum number of VIP users.
  VIP_ACHIEVEMENT_INCOMPLETE,  //!< The channel has not unlocked the VIP feature
};

enum class RevokeVIPErrorCode {
  SUCCESS,            //!< No error.
  UNKNOWN,            //!< Received an unrecognized error code.
  CHANNEL_NOT_FOUND,  //!< Channel does not exist.
  FORBIDDEN,          //!< The granter does not have permisison to revoke VIP status in the channel.
  REVOKEE_NOT_FOUND,  //!< The target user does not exist.
  REVOKEE_NOT_VIP,    //!< The target user is not a VIP
};
}  // namespace chat
}  // namespace ttv
