/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/ichatcommentmanager.h"
#include "twitchsdk/core/module.h"

#include <map>
#include <memory>
#include <string>

namespace ttv {
namespace chat {
class IChatChannelListener;
class IChatAPIListener;
class IChatUserThreadsListener;
class IChatCommentListener;

struct ChatChannelListenerProxy;
}  // namespace chat
}  // namespace ttv

/**
 * Listener interface for callbacks from ChatAPI.
 */
class ttv::chat::IChatAPIListener : public ttv::IModuleListener {
 public:
  virtual void ChatUserEmoticonSetsChanged(UserId userId, const std::vector<EmoticonSet>& emoticonSets) = 0;
};

/**
 * Listener interface for callbacks from ChatChannel.
 */
class ttv::chat::IChatChannelListener {
 public:
  virtual ~IChatChannelListener() = default;

  virtual void ChatChannelStateChanged(
    UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) = 0;
  virtual void ChatChannelInfoChanged(UserId userId, ChannelId channelId, const ChatChannelInfo& channelInfo) = 0;
  virtual void ChatChannelRestrictionsChanged(
    UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) = 0;
  virtual void ChatChannelLocalUserChanged(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) = 0;
  virtual void ChatChannelMessagesReceived(
    UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) = 0;
  virtual void ChatChannelSubscriptionNoticeReceived(
    UserId userId, ChannelId channelId, const SubscriptionNotice& notice) = 0;
  virtual void ChatChannelFirstTimeChatterNoticeReceived(
    UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) = 0;
  virtual void ChatChannelRaidNoticeReceived(UserId userId, ChannelId channelId, const RaidNotice& notice) = 0;
  virtual void ChatChannelUnraidNoticeReceived(UserId userId, ChannelId channelId, const UnraidNotice& notice) = 0;
  virtual void ChatChannelGenericNoticeReceived(
    UserId userId, ChannelId channelId, const GenericMessageNotice& notice) = 0;
  virtual void ChatChannelNoticeReceived(UserId userId, ChannelId channelId, const std::string& noticeId,
    const std::map<std::string, std::string>& params) = 0;
  virtual void ChatChannelHostTargetChanged(
    UserId userId, ChannelId channelId, const std::string& targetChannelName, uint32_t numViewers) = 0;
  virtual void ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& senderLoginName, std::string&& messageContent) = 0;

  /**
   * Fired when all messages in the channel should be cleared.
   */
  virtual void ChatChannelMessagesCleared(UserId userId, ChannelId channelId) = 0;
  /**
   * Fired when all messages from the given user should be cleared in the given channel.
   */
  virtual void ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId) = 0;

  /*
   * Fired for the chatter when a message being sent is caught by AutoMod.
   */
  virtual void AutoModCaughtSentMessage(UserId userId, ChannelId channelId) = 0;
  virtual void AutoModDeniedSentMessage(UserId userId, ChannelId channelId) = 0;
  virtual void AutoModApprovedSentMessage(UserId userId, ChannelId channelId) = 0;

  /*
   * Fired for the chatter when a cheer message caught by AutoMod has been denied.
   */
  virtual void AutoModDeniedSentCheer(UserId userId, ChannelId channelId) = 0;
  virtual void AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) = 0;

  /*
   * Fired for the channel moderators when a message is caught by AutoMod.
   */
  virtual void AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) = 0;
  virtual void AutoModMessageApprovedByMod(
    UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) = 0;
  virtual void AutoModMessageDeniedByMod(
    UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) = 0;

  /*
   * Fired for the channel moderators when a cheer is caught by AutoMod.
   */
  virtual void AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) = 0;

  /**
   * Fired for channel moderators to give information on which moderator ran the command.
   */
  virtual void ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) = 0;
  virtual void ChatChannelModNoticeUserBanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) = 0;
  virtual void ChatChannelModNoticeUserUntimedOut(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) = 0;
  virtual void ChatChannelModNoticeUserUnbanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) = 0;
  virtual void ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) = 0;
  virtual void ChatChannelModNoticeClearChat(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeEmoteOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeEmoteOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeFollowersOnly(UserId userId, ChannelId channelId, UserId modId,
    std::string&& modName, uint32_t minimumFollowingDurationMinutes) = 0;
  virtual void ChatChannelModNoticeFollowersOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeR9K(UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeR9KOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeSlow(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t slowModeDurationSeconds) = 0;
  virtual void ChatChannelModNoticeSlowOff(UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeSubsOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
  virtual void ChatChannelModNoticeSubsOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) = 0;
};

/**
 * Listener interface for callbacks from the conversation system.
 */
class ttv::chat::IChatUserThreadsListener {
 public:
  virtual ~IChatUserThreadsListener() = default;

  virtual void ChatThreadRealtimeMessageReceived(
    UserId userId, const std::string& threadId, const WhisperMessage& messageList) = 0;
};

/**
 * Listener interface for callbacks from the ChatCommentsManager.
 *
 * Should be inherited by a concrete implementation that overrides all methods.
 */
class ttv::chat::IChatCommentListener {
 public:
  virtual ~IChatCommentListener() = default;

  /**
   * Fired to indicate the playback state has changed.
   *
   * @param[in] userId The logged in user that is watching the VOD.
   * @param[in] vodId The VOD that the user is watching.
   * @param[in] state The current playing state of the chat comment manager.
   */
  virtual void ChatCommentManagerStateChanged(
    UserId userId, const std::string& vodId, IChatCommentManager::PlayingState state) = 0;

  /**
   * Fired to notify that comments have been received at the current playhead time.
   *
   * @param[in] userId The logged in user that is watching the VOD.
   * @param[in] vodId The VOD that the user is watching.
   * @param[in] messageList A block of messages that should be displayed.
   */
  virtual void ChatCommentsReceived(
    UserId userId, const std::string& vodId, std::vector<ChatComment>&& messageList) = 0;

  /**
   * Fired to notify that an error was received from the server while fetching comments
   *
   * @param[in] errorMsg The error message for the error received while fetching comments
   * @param[in] ec The error code of the error received while fetching comments
   */
  virtual void ChatCommentsErrorReceived(const std::string& errorMsg, TTV_ErrorCode ec) = 0;
};

struct ttv::chat::ChatChannelListenerProxy : public IChatChannelListener {
  using ChatChannelStateChangedCallback =
    std::function<void(UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec)>;
  using ChatChannelInfoChangedCallback =
    std::function<void(UserId userId, ChannelId channelId, const ChatChannelInfo& channelInfo)>;
  using ChatChannelRestrictionsChangedCallback =
    std::function<void(UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions)>;
  using ChatChannelLocalUserChangedCallback =
    std::function<void(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo)>;
  using ChatChannelSubscriptionNoticeReceivedCallback =
    std::function<void(UserId userId, ChannelId channelId, const SubscriptionNotice& notice)>;
  using ChatChannelFirstTimeChatterNoticeReceivedCallback =
    std::function<void(UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice)>;
  using ChatChannelRaidNoticeReceivedCallback =
    std::function<void(UserId userId, ChannelId channelId, const RaidNotice& notice)>;
  using ChatChannelUnraidNoticeReceivedCallback =
    std::function<void(UserId userId, ChannelId channelId, const UnraidNotice& notice)>;
  using ChatChannelGenericNoticeReceivedCallback =
    std::function<void(UserId userId, ChannelId channelId, const GenericMessageNotice& notice)>;
  using ChatChannelMessagesReceivedCallback =
    std::function<void(UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList)>;
  using ChatChannelMessagesClearedCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using ChatChannelUserMessagesClearedCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId clearUserId)>;
  using ChatChannelMessageDeletedCallback = std::function<void(UserId userId, ChannelId channelId,
    std::string&& messageId, std::string&& senderLoginName, std::string&& messageContent)>;
  using ChatChannelModNoticeUserTimedOutCallback = std::function<void(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason)>;
  using ChatChannelModNoticeUserBannedCallback =
    std::function<void(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason)>;
  using ChatChannelModNoticeUserUntimedOutCallback =
    std::function<void(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo)>;
  using ChatChannelModNoticeUserUnbannedCallback =
    std::function<void(UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo)>;
  using ChatChannelModNoticeMessageDeletedCallback = std::function<void(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message)>;
  using ChatChannelModNoticeClearChatCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeEmoteOnlyCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeEmoteOnlyOffCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeFollowersOnlyCallback = std::function<void(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t minimumFollowingDurationMinutes)>;
  using ChatChannelModNoticeFollowersOnlyOffCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeR9KCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeR9KOffCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeSlowCallback = std::function<void(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t slowModeDurationSeconds)>;
  using ChatChannelModNoticeSlowOffCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeSubsOnlyCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelModNoticeSubsOnlyOffCallback =
    std::function<void(UserId userId, ChannelId channelId, UserId modId, std::string&& modName)>;
  using ChatChannelHostTargetChangedCallback =
    std::function<void(UserId userId, ChannelId channelId, const std::string& targetChannel, uint32_t numViewers)>;
  using ChatChannelNoticeReceivedCallback = std::function<void(
    UserId userId, ChannelId channelId, const std::string& noticeId, const std::map<std::string, std::string>& params)>;
  using ChatChannelSetBroadcasterLanguageChatEnabledCompleteCallback =
    std::function<void(UserId userId, ChannelId channelId, TTV_ErrorCode ec)>;
  using AutoModCaughtSentMessageCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using AutoModCaughtDeniedMessageCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using AutoModCaughtApprovedMessageCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using AutoModCaughtMessageForModsCallback = std::function<void(UserId userId, ChannelId channelId,
    std::string&& messageId, std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason)>;
  using AutoModMessageApprovedByModCallback = std::function<void(
    UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName)>;
  using AutoModMessageDeniedByModCallback = std::function<void(
    UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName)>;
  using AutoModCaughtDeniedCheerCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using AutoModCaughtTimedOutCheerCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using AutoModCaughtCheerForModsCallback = std::function<void(UserId userId, ChannelId channelId,
    std::string&& messageId, std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason)>;

  // IChatChannelListener implementation
  virtual void ChatChannelStateChanged(
    UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) override {
    if (chatChannelStateChangedCallback != nullptr) {
      chatChannelStateChangedCallback(userId, channelId, state, ec);
    }
  }

  virtual void ChatChannelInfoChanged(UserId userId, ChannelId channelId, const ChatChannelInfo& channelInfo) override {
    if (chatChannelInfoChangedCallback != nullptr) {
      chatChannelInfoChangedCallback(userId, channelId, channelInfo);
    }
  }

  virtual void ChatChannelRestrictionsChanged(
    UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) override {
    if (chatChannelRestrictionsChangedCallback != nullptr) {
      chatChannelRestrictionsChangedCallback(userId, channelId, restrictions);
    }
  }

  virtual void ChatChannelLocalUserChanged(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) override {
    if (chatChannelLocalUserChangedCallback != nullptr) {
      chatChannelLocalUserChangedCallback(userId, channelId, userInfo);
    }
  }

  virtual void ChatChannelSubscriptionNoticeReceived(
    UserId userId, ChannelId channelId, const SubscriptionNotice& notice) override {
    if (chatChannelSubscriptionNoticeReceivedCallback != nullptr) {
      chatChannelSubscriptionNoticeReceivedCallback(userId, channelId, notice);
    }
  }

  virtual void ChatChannelFirstTimeChatterNoticeReceived(
    UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) override {
    if (chatChannelFirstTimeChatterNoticeReceivedCallback != nullptr) {
      chatChannelFirstTimeChatterNoticeReceivedCallback(userId, channelId, notice);
    }
  }

  virtual void ChatChannelRaidNoticeReceived(UserId userId, ChannelId channelId, const RaidNotice& notice) override {
    if (chatChannelRaidNoticeReceivedCallback != nullptr) {
      chatChannelRaidNoticeReceivedCallback(userId, channelId, notice);
    }
  }

  virtual void ChatChannelUnraidNoticeReceived(
    UserId userId, ChannelId channelId, const UnraidNotice& notice) override {
    if (chatChannelUnraidNoticeReceivedCallback != nullptr) {
      chatChannelUnraidNoticeReceivedCallback(userId, channelId, notice);
    }
  }

  virtual void ChatChannelGenericNoticeReceived(
    UserId userId, ChannelId channelId, const GenericMessageNotice& notice) override {
    if (chatChannelGenericNoticeReceivedCallback != nullptr) {
      chatChannelGenericNoticeReceivedCallback(userId, channelId, notice);
    }
  }

  virtual void ChatChannelMessagesReceived(
    UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) override {
    if (chatChannelMessagesReceivedCallback != nullptr) {
      chatChannelMessagesReceivedCallback(userId, channelId, messageList);
    }
  }

  virtual void ChatChannelMessagesCleared(UserId userId, ChannelId channelId) override {
    if (chatChannelMessagesClearedCallback != nullptr) {
      chatChannelMessagesClearedCallback(userId, channelId);
    }
  }

  virtual void ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId) override {
    if (chatChannelUserMessagesClearedCallback != nullptr) {
      chatChannelUserMessagesClearedCallback(userId, channelId, clearUserId);
    }
  }

  virtual void ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& senderLoginName, std::string&& messageContent) override {
    if (chatChannelMessageDeletedCallback != nullptr) {
      chatChannelMessageDeletedCallback(
        userId, channelId, std::move(messageId), std::move(senderLoginName), std::move(messageContent));
    }
  }

  virtual void ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) override {
    if (chatChannelModNoticeUserTimedOutCallback != nullptr) {
      chatChannelModNoticeUserTimedOutCallback(
        userId, channelId, std::move(modActionInfo), timeoutDurationSeconds, std::move(reason));
    }
  }

  virtual void ChatChannelModNoticeUserBanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) override {
    if (chatChannelModNoticeUserBannedCallback != nullptr) {
      chatChannelModNoticeUserBannedCallback(userId, channelId, std::move(modActionInfo), std::move(reason));
    }
  }

  virtual void ChatChannelModNoticeUserUntimedOut(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override {
    if (chatChannelModNoticeUserUntimedOutCallback != nullptr) {
      chatChannelModNoticeUserUntimedOutCallback(userId, channelId, std::move(modActionInfo));
    }
  }

  virtual void ChatChannelModNoticeUserUnbanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override {
    if (chatChannelModNoticeUserUnbannedCallback != nullptr) {
      chatChannelModNoticeUserUnbannedCallback(userId, channelId, std::move(modActionInfo));
    }
  }

  virtual void ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) override {
    if (chatChannelModNoticeMessageDeletedCallback != nullptr) {
      chatChannelModNoticeMessageDeletedCallback(
        userId, channelId, std::move(modActionInfo), std::move(messageId), std::move(message));
    }
  }

  virtual void ChatChannelModNoticeClearChat(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeClearChatCallback != nullptr) {
      chatChannelModNoticeClearChatCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeEmoteOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeEmoteOnlyCallback != nullptr) {
      chatChannelModNoticeEmoteOnlyCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeEmoteOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeEmoteOnlyOffCallback != nullptr) {
      chatChannelModNoticeEmoteOnlyOffCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeFollowersOnly(UserId userId, ChannelId channelId, UserId modId,
    std::string&& modName, uint32_t minimumFollowingDurationMinutes) override {
    if (chatChannelModNoticeFollowersOnlyCallback != nullptr) {
      chatChannelModNoticeFollowersOnlyCallback(
        userId, channelId, modId, std::move(modName), minimumFollowingDurationMinutes);
    }
  }

  virtual void ChatChannelModNoticeFollowersOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeFollowersOnlyOffCallback != nullptr) {
      chatChannelModNoticeFollowersOnlyOffCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeR9K(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeR9KCallback != nullptr) {
      chatChannelModNoticeR9KCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeR9KOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeR9KOffCallback != nullptr) {
      chatChannelModNoticeR9KOffCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeSlow(UserId userId, ChannelId channelId, UserId modId, std::string&& modName,
    uint32_t slowModeDurationSeconds) override {
    if (chatChannelModNoticeSlowCallback != nullptr) {
      chatChannelModNoticeSlowCallback(userId, channelId, modId, std::move(modName), slowModeDurationSeconds);
    }
  }

  virtual void ChatChannelModNoticeSlowOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeSlowOffCallback != nullptr) {
      chatChannelModNoticeSlowOffCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeSubsOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeSubsOnlyCallback != nullptr) {
      chatChannelModNoticeSubsOnlyCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelModNoticeSubsOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {
    if (chatChannelModNoticeSubsOnlyOffCallback != nullptr) {
      chatChannelModNoticeSubsOnlyOffCallback(userId, channelId, modId, std::move(modName));
    }
  }

  virtual void ChatChannelHostTargetChanged(
    UserId userId, ChannelId channelId, const std::string& targetChannel, uint32_t numViewers) override {
    if (chatChannelHostTargetChangedCallback != nullptr) {
      chatChannelHostTargetChangedCallback(userId, channelId, targetChannel, numViewers);
    }
  }

  virtual void ChatChannelNoticeReceived(UserId userId, ChannelId channelId, const std::string& noticeId,
    const std::map<std::string, std::string>& params) override {
    if (chatChannelNoticeReceivedCallback != nullptr) {
      chatChannelNoticeReceivedCallback(userId, channelId, noticeId, params);
    }
  }

  virtual void AutoModCaughtSentMessage(UserId userId, ChannelId channelId) override {
    if (autoModCaughtSentMessageCallback != nullptr) {
      autoModCaughtSentMessageCallback(userId, channelId);
    }
  }

  virtual void AutoModApprovedSentMessage(UserId userId, ChannelId channelId) override {
    if (autoModApprovedSentMessageCallback != nullptr) {
      autoModApprovedSentMessageCallback(userId, channelId);
    }
  }

  virtual void AutoModDeniedSentMessage(UserId userId, ChannelId channelId) override {
    if (autoModDeniedSentMessageCallback != nullptr) {
      autoModDeniedSentMessageCallback(userId, channelId);
    }
  }

  virtual void AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override {
    if (autoModCaughtMessageForModsCallback != nullptr) {
      autoModCaughtMessageForModsCallback(userId, channelId, std::move(messageId), std::move(message), senderId,
        std::move(senderName), std::move(reason));
    }
  }

  virtual void AutoModMessageApprovedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
    UserId moderatorId, std::string&& moderatorName) override {
    if (autoModMessageApprovedByModCallback != nullptr) {
      autoModMessageApprovedByModCallback(
        userId, channelId, std::move(messageId), moderatorId, std::move(moderatorName));
    }
  }

  virtual void AutoModMessageDeniedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
    UserId moderatorId, std::string&& moderatorName) override {
    if (autoModMessageDeniedByModCallback != nullptr) {
      autoModMessageDeniedByModCallback(userId, channelId, std::move(messageId), moderatorId, std::move(moderatorName));
    }
  }

  virtual void AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) override {
    if (autoModTimedOutSentCheerCallback != nullptr) {
      autoModTimedOutSentCheerCallback(userId, channelId);
    }
  }

  virtual void AutoModDeniedSentCheer(UserId userId, ChannelId channelId) override {
    if (autoModDeniedSentCheerCallback != nullptr) {
      autoModDeniedSentCheerCallback(userId, channelId);
    }
  }

  virtual void AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override {
    if (autoModCaughtCheerForModsCallback != nullptr) {
      autoModCaughtCheerForModsCallback(userId, channelId, std::move(messageId), std::move(message), senderId,
        std::move(senderName), std::move(reason));
    }
  }

  ChatChannelStateChangedCallback chatChannelStateChangedCallback;
  ChatChannelInfoChangedCallback chatChannelInfoChangedCallback;
  ChatChannelRestrictionsChangedCallback chatChannelRestrictionsChangedCallback;
  ChatChannelLocalUserChangedCallback chatChannelLocalUserChangedCallback;
  ChatChannelSubscriptionNoticeReceivedCallback chatChannelSubscriptionNoticeReceivedCallback;
  ChatChannelFirstTimeChatterNoticeReceivedCallback chatChannelFirstTimeChatterNoticeReceivedCallback;
  ChatChannelRaidNoticeReceivedCallback chatChannelRaidNoticeReceivedCallback;
  ChatChannelUnraidNoticeReceivedCallback chatChannelUnraidNoticeReceivedCallback;
  ChatChannelGenericNoticeReceivedCallback chatChannelGenericNoticeReceivedCallback;
  ChatChannelMessagesReceivedCallback chatChannelMessagesReceivedCallback;
  ChatChannelMessagesClearedCallback chatChannelMessagesClearedCallback;
  ChatChannelUserMessagesClearedCallback chatChannelUserMessagesClearedCallback;
  ChatChannelMessageDeletedCallback chatChannelMessageDeletedCallback;
  ChatChannelModNoticeUserTimedOutCallback chatChannelModNoticeUserTimedOutCallback;
  ChatChannelModNoticeUserBannedCallback chatChannelModNoticeUserBannedCallback;
  ChatChannelModNoticeUserUntimedOutCallback chatChannelModNoticeUserUntimedOutCallback;
  ChatChannelModNoticeUserUnbannedCallback chatChannelModNoticeUserUnbannedCallback;
  ChatChannelModNoticeMessageDeletedCallback chatChannelModNoticeMessageDeletedCallback;
  ChatChannelModNoticeClearChatCallback chatChannelModNoticeClearChatCallback;
  ChatChannelModNoticeEmoteOnlyCallback chatChannelModNoticeEmoteOnlyCallback;
  ChatChannelModNoticeEmoteOnlyOffCallback chatChannelModNoticeEmoteOnlyOffCallback;
  ChatChannelModNoticeFollowersOnlyCallback chatChannelModNoticeFollowersOnlyCallback;
  ChatChannelModNoticeFollowersOnlyOffCallback chatChannelModNoticeFollowersOnlyOffCallback;
  ChatChannelModNoticeR9KCallback chatChannelModNoticeR9KCallback;
  ChatChannelModNoticeR9KOffCallback chatChannelModNoticeR9KOffCallback;
  ChatChannelModNoticeSlowCallback chatChannelModNoticeSlowCallback;
  ChatChannelModNoticeSlowOffCallback chatChannelModNoticeSlowOffCallback;
  ChatChannelModNoticeSubsOnlyCallback chatChannelModNoticeSubsOnlyCallback;
  ChatChannelModNoticeSubsOnlyOffCallback chatChannelModNoticeSubsOnlyOffCallback;
  ChatChannelHostTargetChangedCallback chatChannelHostTargetChangedCallback;
  ChatChannelNoticeReceivedCallback chatChannelNoticeReceivedCallback;
  ChatChannelSetBroadcasterLanguageChatEnabledCompleteCallback
    chatChannelSetBroadcasterLanguageChatEnabledCompleteCallback;
  AutoModCaughtSentMessageCallback autoModCaughtSentMessageCallback;
  AutoModCaughtDeniedMessageCallback autoModDeniedSentMessageCallback;
  AutoModCaughtApprovedMessageCallback autoModApprovedSentMessageCallback;
  AutoModCaughtMessageForModsCallback autoModCaughtMessageForModsCallback;
  AutoModMessageApprovedByModCallback autoModMessageApprovedByModCallback;
  AutoModMessageDeniedByModCallback autoModMessageDeniedByModCallback;
  AutoModCaughtDeniedCheerCallback autoModDeniedSentCheerCallback;
  AutoModCaughtTimedOutCheerCallback autoModTimedOutSentCheerCallback;
  AutoModCaughtCheerForModsCallback autoModCaughtCheerForModsCallback;
};
