/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/internal/pch.h"

namespace ttv {
namespace chat {
class ChatMessageHandler;
}
}  // namespace ttv

/**
 * Handles messages that are being sent in chat for special commands such as `/ignore`.
 * Custom callback functions must inherit ICallbacks and passed in through the constructor first.
 * Messages being sent will be passed in through HandleMessage().
 */
class ttv::chat::ChatMessageHandler {
 public:
  /**
   * Create your own callbacks class that inherits from this,
   * then pass in that class to the ChatMessageHandler constructor.
   */
  class ICallbacks {
   public:
    virtual ~ICallbacks() = default;

    /**
     * The message contains normal text or an unhandled command.
     * Callback typically should call ChatAPI::SendChatMessage().
     *
     * @param[in] message The message string being delivered through chat.
     * @return True if message was handled properly, else false.
     */
    virtual bool PassThrough(const std::string& message) = 0;

    /**
     * Called for valid `/ignore` and `/block` commands.
     * Typically should display the necessary UI for getting a reason, then call ChatAPI::BlockUser().
     *
     * @param[in] blockUserName The name of the user to be blocked.
     * @return True if the message was handled properly, else false.
     */
    virtual bool BlockUser(const std::string& blockUserName) = 0;

    /**
     * Called for valid `/unignore` commands.
     * Callback typically should call ChatAPI::UnblockUser().
     *
     * @param[in] unblockUserName The name of the user to be unblocked.
     * @return True if the message was handled properly, else false.
     */
    virtual bool UnblockUser(const std::string& unblockUserName) = 0;

    /**
     * Called for valid `/w` commands.
     * Callback typically should call ChatAPI::SendMessageToUser().
     *
     * @param[in] whisperUserName The name of the user that the whisper is being sent to.
     * @param[in] message The message being sent to the whispered user.
     */
    virtual bool WhisperUser(const std::string& whisperUserName, const std::string& message) = 0;

    /**
     * Called for valid `/raid` commands.
     * Callback typically should call ChatRaid.Start().
     *
     * @param[in] targetChannelName The name of the channel that the raid is targetting.
     */
    virtual bool CreateRaid(const std::string& targetUserName) = 0;

    /**
     * Called for valid `/unraid` commands.
     * Callback typically should call ChatRaid.Cancel().
     */
    virtual bool CancelRaid() = 0;

    /**
     * Called for valid `/vip` commands.
     * Callback typically should call ChatAPI::GrantVIP().
     *
     * @param[in] vipUserName The name of the user that is being granted VIP status.
     */
    virtual bool GrantVIP(const std::string& vipUserName) = 0;

    /**
     * Called for valid `/unvip` commands.
     * Callback typically should call ChatAPI::RevokeVIP().
     *
     * @param[in] unvipUserName The name of the user that is losing VIP status.
     */
    virtual bool RevokeVIP(const std::string& unvipUserName) = 0;

    /**
     * Called for valid `/vips` commands.
     * Callback typically should call ChatAPI::ListVIPs().
     */
    virtual bool ListVIPs() = 0;
  };

  ChatMessageHandler();

  /**
   * Takes in outgoing chat messages and call the corresponding callback function based on the contents of the message.
   *
   * @pre Callback functions must be set previously through SetCallbacks.
   * @param[in] message The message being sent to chat.
   * @return The result of the callback function.
   */
  bool HandleMessage(const std::string& message);

  /**
   * Sets the callbacks functions that are called when HandleMessage is called.
   *
   * @param[in] callbacks Pointer to the inherited ICallbacks class.
   */
  void SetCallbacks(const std::shared_ptr<ICallbacks>& callbacks) { mCallbacks = callbacks; }

 private:
  std::shared_ptr<ICallbacks> mCallbacks;
};
